from artap.problem import Problem
from artap.results import Results
from artap.datastore import FileDataStore
from artap.algorithm_bayesopt import BayesOptSerial
from artap.algorithm_nlopt import NLopt, LN_BOBYQA, LN_COBYLA
from artap.algorithm_genetic import NSGAII
from artap.algorithm_sweep import SweepAlgorithm
from artap.operators import LHSGeneration
from artap.surrogate_smt import SurrogateModelSMT
from artap.benchmark_functions import SurrogateBenchmarkLoudspeakerSingle

from agrossuite import agros as a2d
import math
import numpy as np


class LoudspeakerProblem(Problem):
    def evaluate_model(self, individual):
        print(individual.vector)

        # problem
        problem = a2d.problem(clear=True)
        problem.coordinate_type = "axisymmetric"
        problem.mesh_type = "triangle"
        problem.parameters["c1"] = individual.vector[0]
        problem.parameters["c2"] = individual.vector[1]
        problem.parameters["g1"] = individual.vector[2]
        problem.parameters["g2"] = individual.vector[3]

        # fields
        # magnetic
        magnetic = problem.field("magnetic")
        magnetic.analysis_type = "steadystate"
        magnetic.matrix_solver = "umfpack"
        magnetic.number_of_refinements = 1
        magnetic.polynomial_order = 2
        magnetic.adaptivity_type = "disabled"
        magnetic.solver = "newton"
        magnetic.solver_parameters['residual'] = 0
        magnetic.solver_parameters['relative_change_of_solutions'] = 0.01
        magnetic.solver_parameters['damping'] = "automatic"
        magnetic.solver_parameters['damping_factor'] = 0.8
        magnetic.solver_parameters['damping_factor_increase_steps'] = 1
        magnetic.solver_parameters['damping_factor_decrease_ratio'] = 1.2
        magnetic.solver_parameters['jacobian_reuse'] = True
        magnetic.solver_parameters['jacobian_reuse_ratio'] = 0.8
        magnetic.solver_parameters['jacobian_reuse_steps'] = 20

        # boundaries
        magnetic.add_boundary("A=0", "magnetic_potential", {"magnetic_potential_real": 0})

        # materials
        magnetic.add_material("Air", {"magnetic_conductivity": 0, "magnetic_current_density_external_real": 0, "magnetic_permeability": 1, "magnetic_remanence": 0, "magnetic_remanence_angle": 0, "magnetic_total_current_prescribed": 0, "magnetic_total_current_real": 0, "magnetic_velocity_angular": 0, "magnetic_velocity_x": 0, "magnetic_velocity_y": 0})
        magnetic.add_material("Magnetic Circuit", {"magnetic_conductivity": 0, "magnetic_current_density_external_real": 0, "magnetic_permeability": {"value": 10000, "x": [0, 1, 1.49, 1.94, 2.26, 2.73, 2.88, 3.15, 3.45, 3.78, 4.06, 4.42, 4.72, 4.94, 5.14, 5.25], "y": [1200, 1200, 1114, 906, 729, 565, 422, 315, 223, 147, 100, 57, 34, 22, 16, 13], "interpolation": "piecewise_linear", "extrapolation": "constant", "derivative_at_endpoints": "first"}, "magnetic_remanence": 0, "magnetic_remanence_angle": 0,
                                                   "magnetic_total_current_prescribed": 0, "magnetic_total_current_real": 0, "magnetic_velocity_angular": 0, "magnetic_velocity_x": 0, "magnetic_velocity_y": 0})
        magnetic.add_material("Magnet", {"magnetic_conductivity": 0, "magnetic_current_density_external_real": 0, "magnetic_permeability": 1, "magnetic_remanence": 0.8, "magnetic_remanence_angle": 90, "magnetic_total_current_prescribed": 0, "magnetic_total_current_real": 0, "magnetic_velocity_angular": 0, "magnetic_velocity_x": 0, "magnetic_velocity_y": 0})
        magnetic.add_material("Coil", {"magnetic_conductivity": 0, "magnetic_current_density_external_real": 1e-3, "magnetic_permeability": 1, "magnetic_remanence": 0, "magnetic_remanence_angle": 0, "magnetic_total_current_prescribed": 0, "magnetic_total_current_real": 0, "magnetic_velocity_angular": 0, "magnetic_velocity_x": 0, "magnetic_velocity_y": 0})

        # geometry
        geometry = problem.geometry()
        geometry.add_edge("0.006+0.0095+0.0023", "-0.087+0.035", 0.006, "-0.087+0.035")
        geometry.add_edge(0.006, "-0.087+0.035", 0.006, -0.087)
        geometry.add_edge(0.006, -0.087, "0.0155+c1", -0.087)
        geometry.add_edge("0.0155+c2", -0.08, "0.0155+c1", -0.087)
        geometry.add_edge("0.006+0.0095", -0.06, "0.006+0.0095+0.0023", -0.06)
        geometry.add_edge("0.006+0.0095", -0.06, 0.0155, -0.08)
        geometry.add_edge("0.006+0.0095+0.0023", "-0.087+0.035", "0.006+0.0095+0.0023", -0.06)
        geometry.add_edge(0.025, -0.08, 0.0155, -0.08)
        geometry.add_edge(0.025, -0.08, "0.0155+c2", -0.08)
        geometry.add_edge("0.0155+c2", -0.08, "0.025+0.026", -0.08)
        geometry.add_edge("0.025+0.026", "-0.08+0.02", "0.025+0.026", -0.08)
        geometry.add_edge(0.025, "-0.08+0.02", 0.025, -0.08)
        geometry.add_edge("0.019+g1", -0.06, "0.025+0.026", "-0.08+0.02")
        geometry.add_edge("0.019+g1", -0.06, 0.025, "-0.08+0.02")
        geometry.add_edge(0.019, "-0.06+0.008", 0.019, -0.06)
        geometry.add_edge(0.019, -0.06, 0.025, "-0.08+0.02")
        geometry.add_edge(0.019, "-0.06+0.008", "0.019+g2", "-0.06+0.008")
        geometry.add_edge("0.019+g2", "-0.06+0.008", "0.019+g1", -0.06)
        geometry.add_edge("0.0182+6e-4", "-0.0635+0.015", 0.0182, "-0.0635+0.015")
        geometry.add_edge(0.0182, "-0.0635+0.015", 0.0182, -0.0635)
        geometry.add_edge(0.0182, -0.0635, "0.0182+6e-4", -0.0635)
        geometry.add_edge("0.0182+6e-4", "-0.0635+0.015", "0.0182+6e-4", -0.0635)
        geometry.add_edge(0, "-0.08-0.16", 0.16, -0.08, angle=90, segments=6, boundaries={"magnetic": "A=0"})
        geometry.add_edge(0.16, -0.08, 0, "-0.08+0.16", angle=90, segments=6, boundaries={"magnetic": "A=0"})
        geometry.add_edge(0, "-0.08+0.16", 0, "-0.08-0.16", boundaries={"magnetic": "A=0"})

        geometry.add_label(0.0585696, 0.00490675, materials={"magnetic": "Air"})
        geometry.add_label(0.0103459, -0.0713362, materials={"magnetic": "Magnetic Circuit"})
        geometry.add_label(0.0355462, -0.0706859, materials={"magnetic": "Magnet"})
        geometry.add_label(0.0184561, -0.057864, materials={"magnetic": "Coil"})
        geometry.add_label(0.019559, -0.0564227, materials={"magnetic": "Magnetic Circuit"})

        # recipes
        magnetic.add_recipe_volume_integral("Volume", "magnetic_volume", [1, 4], -1, -1)
        magnetic.add_recipe_volume_integral("Force", "magnetic_lorentz_force_y", [3], -1, -1)
        magnetic.add_recipe_local_value("Br", "magnetic_flux_density_real", "x", 0.0185, -0.056, -1, -1)

        try:
            computation = problem.computation()
            computation.solve()

            solution = computation.solution("magnetic")
            force = 1e9 * solution.volume_integrals(labels=[3])["Fly"]
            volume = 7800.0 * solution.volume_integrals(labels=[1, 4])["V"]

            print('Evaluated force: {}, volume: {}'.format(force, volume))

            return [-abs(force), volume]
        except:
            return [-10, 100]


class LoudspeakerSingleProblem(LoudspeakerProblem):
    def set(self):
        self.name = "AgrosProblem"
        self.parameters = [{'name': 'c1', 'initial_value': 0.03, 'bounds': [0.005, 0.0345]},
                           {'name': 'c2', 'initial_value': 0.03, 'bounds': [0.015, 0.0345]},
                           {'name': 'g1', 'initial_value': 0.03, 'bounds': [0.01, 0.0315]},
                           {'name': 'g2', 'initial_value': 0.03, 'bounds': [0.005, 0.0345]}]
        self.costs = [{'name': 'F'}]

        self.data_store = FileDataStore(self, database_name="tmp/loudspeaker_single.db", mode="write")

    def evaluate(self, individual):
        res = self.evaluate_model(individual)
        return [res[0]]


class LoudspeakerMultiProblem(LoudspeakerProblem):
    def set(self):
        self.name = "AgrosProblem"
        self.parameters = [{'name': 'c1', 'initial_value': 0.03, 'bounds': [0.005, 0.0345]},
                           {'name': 'c2', 'initial_value': 0.03, 'bounds': [0.015, 0.0345]},
                           {'name': 'g1', 'initial_value': 0.03, 'bounds': [0.01, 0.0315]},
                           {'name': 'g2', 'initial_value': 0.03, 'bounds': [0.005, 0.0345]}]
        self.costs = [{'name': 'F1'},
                      {'name': 'F2'}]

        self.data_store = FileDataStore(self, database_name="tmp/loudspeaker_multi.db", mode="write")

    def evaluate(self, individual):
        res = self.evaluate_model(individual)
        return [res[0], res[1]]

def sweep():
    problem = LoudspeakerSingleProblem()

    # enable surrogate
    problem.surrogate = SurrogateModelSMT(problem)
    problem.surrogate.regressor = SurrogateModelSMT.get_kpls_regressor()
    problem.surrogate.train_step = 100
    # problem.surrogate.score_threshold = 0.0
    # problem.surrogate.distance_threshold = 0.004

    # sweep analysis (for training)
    gen = LHSGeneration(problem)
    gen.init(problem.surrogate.train_step)
    algorithm_sweep = SweepAlgorithm(problem, generator=gen)
    algorithm_sweep.run()

    print(problem.surrogate.x_data)
    print(problem.surrogate.y_data)


def bobyqa():
    problem = LoudspeakerSingleProblem()
    algorithm = NLopt(problem)
    algorithm.options['n_iterations'] = 30
    algorithm.options['algorithm'] = LN_BOBYQA
    algorithm.options['verbose_level'] = 0
    algorithm.run()

    results = Results(problem)
    optimum = results.find_minimum('F')
    print(optimum)


def single():
    problem = LoudspeakerSingleProblem()
    """
    # enable surrogate
    problem.surrogate = SurrogateModelSMT(problem)
    problem.surrogate.regressor = SurrogateModelSMT.get_kpls_regressor()
    problem.surrogate.train_step = 10
    problem.surrogate.score_threshold = 0.0
    problem.surrogate.distance_threshold = 0.004

    # sweep analysis (for training)
    gen = LHSGeneration(problem)
    gen.init(problem.surrogate.train_step)
    algorithm_sweep = SweepAlgorithm(problem, generator=gen)
    algorithm_sweep.run()
    """
    #print(problem.surrogate.x_data)
    #print(problem.surrogate.y_data)

    # set cached surrogate data
    """
    bench = SurrogateBenchmarkLoudspeaker(n_test=0)
    x_train, y_train, x_test, y_test = bench.values()
    problem.surrogate.x_data = x_train
    problem.surrogate.y_data = y_train
    problem.surrogate.train()
    """
    algorithm = BayesOptSerial(problem)
    algorithm.options['n_iterations'] = 100

    #algorithm = NLopt(problem)
    #algorithm.options['n_iterations'] = 100
    #algorithm.options['algorithm'] = LN_BOBYQA

    algorithm.options['verbose_level'] = 0
    algorithm.run()

    problem.logger.info("surrogate.predict_counter, {}".format(problem.surrogate.predict_counter))
    problem.logger.info("surrogate.eval_counter, {}".format(problem.surrogate.eval_counter))

    results = Results(problem)
    optimum = results.find_minimum('F')
    print(optimum)

def nsga2():
    problem = LoudspeakerMultiProblem()

    # enable surrogate
    problem.surrogate = SurrogateModelSMT(problem)
    problem.surrogate.regressor = SurrogateModelSMT.get_kpls_regressor()
    problem.surrogate.train_step = 60
    problem.surrogate.score_threshold = 0.0
    problem.surrogate.distance_threshold = 0.004

    # sweep analysis (for training)
    gen = LHSGeneration(problem)
    gen.init(problem.surrogate.train_step)
    algorithm_sweep = SweepAlgorithm(problem, generator=gen)
    algorithm_sweep.run()

    #print(problem.surrogate.x_data)
    #print(problem.surrogate.y_data)

    # optimization
    # NSGA_II
    algorithm = NSGAII(problem)
    algorithm.options['max_population_number'] = 100
    algorithm.options['max_population_size'] = 20
    algorithm.run()
    problem.logger.info("surrogate.predict_counter: {}".format(problem.surrogate.predict_counter))
    problem.logger.info("surrogate.eval_counter: {}".format(problem.surrogate.eval_counter))

    #results = GraphicalResults(problem)
    #results.plot_scatter('F1', 'F2', filename="/tmp/scatter.pdf")


def regressor_stats(regressor, x_test, y_test, verbose=1):
    # test
    print(regressor.__class__.__name__)
    y_pred = regressor.predict_values(np.array(x_test))
    if regressor.supports["variances"]:
        y_variances = regressor.predict_variances(np.array(x_test))

    # print(y_test)
    # print(y)

    verbose = 1
    for i in range(len(x_test)):
        x = x_test[i]
        y = y_test[i]
        val = y_pred[i]
        if regressor.supports["variances"]:
            var = y_variances[i]
        else:
            var = [1e10] * len(y_pred)

        if verbose == 1:
            print("sample: {}".format(i))
            print("x = {}".format(x))
            for j in range(len(y)):
                value_problem = y[j]
                value_surrogate = val[j]
                variance_surrogate = var[j]
                percent = 100.0 * math.fabs(value_problem - value_surrogate) / math.fabs(value_problem)

                # print("params = [{0:15.6E}, {1:11.6e}], \teval = {2:11.6e}, \tvariance = {2:11.6e}, \tpred = {3:11.6e}, \tdiff = {4:11.6e} \t({5:11.6e} %)".format(
                #    x[0], x[1], value_problem, value_surrogate,
                #    math.fabs(value_problem - value_surrogate), percent))
                print("eval = {0:20.17E}, \tpred = {1:20.17E}, \tvariance = {2:11.6e}, \tdiff = {3:20.17E} \t({4:13.10f} %)".format(
                    value_problem, value_surrogate, variance_surrogate, math.fabs(value_problem - value_surrogate), percent))

    me = 0
    if verbose == 1:
        pass
        """
        for j in range(len(y_test)):
            print('Mean Absolute Error:', metrics.mean_absolute_error(y_test[0][j], y_pred[0][j]))
            me = metrics.mean_squared_error(y_test[j], y_pred[0][j])
            print('Mean Squared Error:', me)
            print('Root Mean Squared Error:', np.sqrt(metrics.mean_squared_error(y_test[j], y_pred[0][j])))
        """

    return me

def check_regressor_test(regressor, bench):
    x_train, y_train, x_test, y_test = bench.values()
    print("Limits: {}".format(bench.limits()))

    # print(regressor)
    # fit
    regressor.set_training_values(np.array(x_train), np.array(y_train))
    regressor.train()

    # stats
    regressor_stats(regressor, x_test, y_test)

def check_regressor():
    search = True

    bench = SurrogateBenchmarkLoudspeakerSingle(n_test=10)
    # bench = SurrogateBenchmarkTeam(n_train=None)
    # bench = SurrogateBenchmarkLaser(n_train=None, n_test=10)

    # regressor_test(SurrogateModelScikit.get_mlp_regressor(search=search))

    check_regressor_test(SurrogateModelSMT.get_krg_regressor(), bench)
    check_regressor_test(SurrogateModelSMT.get_idw_regressor(), bench)
    # check_regressor_test(RMTB(xlimits=np.array(bench.limits()), order=4, num_ctrl_pts=20, energy_weight=1e-15, regularization_weight=0.0), bench)
    check_regressor_test(SurrogateModelSMT.get_ls_regressor(), bench)
    check_regressor_test(SurrogateModelSMT.get_qp_regressor(), bench)
    check_regressor_test(SurrogateModelSMT.get_krg_regressor(), bench)
    check_regressor_test(SurrogateModelSMT.get_kpls_regressor(), bench)
    check_regressor_test(SurrogateModelSMT.get_kplsk_regressor(), bench)
    # check_regressor_test(GEKPLS(theta0=[1e-2], xlimits=np.array(bench.limits()), extra_points=1, print_prediction=False), bench)

def results_single():
    # N0LV = 1.0 / (1.04615e-06 * 1e-3) * 100 / 6e-4
    N0LV = 1.0 / 1e-3 * 100 * 1.0 / 6e-4
    # LoudspeakerSingleProblem-457652 - run (159 - BayesOpt: elapsed time: 771.9253129959106 s
    # LoudspeakerSingleProblem-834774 - single (294) - surrogate.predict_counter, 0
    # LoudspeakerSingleProblem-834774 - single (295) - surrogate.eval_counter, 112
    single_bayes_x = [[0.02152857, 0.02016088, 0.01788947, 0.0290181 ], [0.02410959, 0.03053753, 0.01609794, 0.01021797], [0.01328093, 0.02810831, 0.02855717, 0.0058843 ], [0.01759749, 0.0252636 , 0.01201532, 0.01276679], [0.00725458, 0.02381152, 0.02249652, 0.01862521], [0.03149543, 0.01619455, 0.01872581, 0.02027064], [0.00850722, 0.03206871, 0.0294352 , 0.03232965], [0.01555306, 0.01815343, 0.01247314, 0.01491405], [0.03444295, 0.03382992, 0.02635542, 0.0280864 ], [0.02816788, 0.02176717, 0.02366692, 0.0235153 ], [0.0345   , 0.0345   , 0.0315   , 0.0053451], [0.03449858, 0.02598966, 0.03149995, 0.03045552], [0.005 , 0.0345, 0.0315, 0.005 ], [0.01760183, 0.03449981, 0.02977505, 0.00500278], [0.01810275, 0.0345    , 0.03081311, 0.01431385], [0.03449937, 0.03449962, 0.02887031, 0.00500293], [0.01679421, 0.03448476, 0.03076295, 0.00506162], [0.00628404, 0.03449736, 0.03011993, 0.00504292], [0.005 , 0.015 , 0.0315, 0.005 ], [0.0345, 0.0345, 0.0315, 0.0345], [0.02125652, 0.03217552, 0.03032178, 0.00838202], [0.03449941, 0.02980987, 0.02816685, 0.00500291], [0.01270516, 0.03390718, 0.0303335 , 0.00807164], [0.01341483, 0.03449326, 0.03048666, 0.00504297], [0.00894727, 0.03449972, 0.02658366, 0.00500287], [0.01730023, 0.03238544, 0.02848129, 0.01347553], [0.01476348, 0.03449457, 0.03044141, 0.00506676], [0.00500248, 0.03449971, 0.02530401, 0.03449968], [0.01458559, 0.03449488, 0.03051798, 0.00516447], [0.01512589, 0.03449101, 0.0315    , 0.00519948], [0.01370915, 0.03447682, 0.03050458, 0.00645293], [0.01484414, 0.03422278, 0.03043724, 0.005     ], [0.01413371, 0.03390896, 0.03050082, 0.00502891], [0.01502689, 0.03409128, 0.03042182, 0.005     ], [0.01428134, 0.03409394, 0.03033341, 0.005     ], [0.01516139, 0.03406693, 0.03054361, 0.005     ], [0.01383703, 0.03411831, 0.03055835, 0.00500017], [0.01436147, 0.03413197, 0.03052056, 0.00500118], [0.01413996, 0.03413342, 0.03051442, 0.00500115], [0.01494974, 0.03412852, 0.03051168, 0.005     ], [0.01435244, 0.03413552, 0.03062066, 0.00500526], [0.01449143, 0.03414996, 0.03054953, 0.00500338], [0.01415395, 0.03415898, 0.03053895, 0.00500372], [0.01439496, 0.03416694, 0.03051777, 0.00500187], [0.01440998, 0.03416936, 0.03053085, 0.005     ], [0.01462936, 0.03412933, 0.03057358, 0.00500257], [0.01538262, 0.03409352, 0.03058588, 0.00500285], [0.01385316, 0.03411689, 0.0305817 , 0.00500249], [0.0140113 , 0.0341635 , 0.03044796, 0.00500467], [0.01381008, 0.03414241, 0.03056991, 0.00500142], [0.0143124 , 0.03417503, 0.03045794, 0.00500351], [0.01433859, 0.03409285, 0.03067927, 0.00500069], [0.00935541, 0.01554506, 0.02700529, 0.0271668 ], [0.03449916, 0.02806433, 0.026268  , 0.03449937], [0.01423957, 0.03417093, 0.03063203, 0.0050004 ], [0.01151544, 0.03198689, 0.03149996, 0.00500273], [0.03449914, 0.03449971, 0.02133962, 0.03449908], [0.01410169, 0.0341703 , 0.03056072, 0.00500386], [0.01424674, 0.03418372, 0.03043668, 0.00500816], [0.01425198, 0.03417429, 0.03049844, 0.00500271], [0.01422936, 0.03417787, 0.03047865, 0.00500549], [0.03449885, 0.03449928, 0.02421203, 0.00500474], [0.01408842, 0.03414632, 0.03058295, 0.00500078], [0.01425065, 0.03415406, 0.03062055, 0.00500525], [0.01404908, 0.03417714, 0.03048492, 0.00500001], [0.0142056 , 0.03417832, 0.03050181, 0.0050016 ], [0.01426266, 0.03417542, 0.03052263, 0.00500069], [0.01442489, 0.03417896, 0.03048028, 0.00500684], [0.01458378, 0.03418286, 0.030531  , 0.00500027], [0.01362551, 0.03418016, 0.03046593, 0.00500112], [0.01459324, 0.03420489, 0.03047023, 0.00500107], [0.01468535, 0.03421292, 0.03046651, 0.0050041 ], [0.005     , 0.0345    , 0.02224102, 0.005     ], [0.01473899, 0.03413767, 0.03080232, 0.00500132], [0.01448887, 0.0342063 , 0.03056518, 0.00500042], [0.01264379, 0.0338637 , 0.02898497, 0.00502916], [0.01479069, 0.03418564, 0.03056552, 0.00501167], [0.01409223, 0.03420999, 0.03057627, 0.0050019 ], [0.01464107, 0.03419035, 0.03052546, 0.0050104 ], [0.01438653, 0.03421   , 0.03058224, 0.00500416], [0.01467188, 0.03420321, 0.0305143 , 0.00500511], [0.01463526, 0.03421331, 0.03051914, 0.00500028], [0.01451851, 0.03421081, 0.030513  , 0.005     ], [0.01452589, 0.03420826, 0.03053686, 0.005     ], [0.01467555, 0.03419897, 0.03047774, 0.00500425], [0.01477774, 0.03420008, 0.03051642, 0.00500117], [0.01388437, 0.03419654, 0.03056434, 0.00500262], [0.01446436, 0.03420859, 0.03045356, 0.005     ], [0.01421906, 0.03421256, 0.03059543, 0.00500271], [0.01415439, 0.03421661, 0.03056547, 0.00500007], [0.01405612, 0.03422047, 0.03046815, 0.005     ], [0.01385477, 0.03422191, 0.03035095, 0.005     ], [0.01459947, 0.03420911, 0.03054094, 0.005     ], [0.01441665, 0.03421415, 0.03053596, 0.00500159], [0.01453834, 0.03420738, 0.03050406, 0.005     ], [0.01384762, 0.03415787, 0.03051163, 0.005     ], [0.01455346, 0.03420861, 0.03043762, 0.00500357], [0.01400515, 0.03417621, 0.03053236, 0.005     ], [0.03349718, 0.01713756, 0.02004741, 0.01744945], [0.01462646, 0.03421267, 0.03052614, 0.005     ], [0.01473869, 0.03423181, 0.03047792, 0.005     ], [0.01446535, 0.03422246, 0.03055126, 0.00500171], [0.01456316, 0.03422908, 0.03041858, 0.005     ], [0.01479222, 0.03421125, 0.03045739, 0.005     ], [0.03327083, 0.02475   , 0.02791667, 0.01975   ], [0.01435375, 0.03421505, 0.03055463, 0.005     ], [0.01413658, 0.03423454, 0.0304435 , 0.00500069], [0.01458645, 0.03422134, 0.03048338, 0.005     ], [0.01818751, 0.03120257, 0.02327482, 0.03296452], [0.0345   , 0.0293222, 0.0315   , 0.005    ], [0.03327083, 0.03125   , 0.026125  , 0.01975   ], [0.01494601, 0.03391972, 0.03095561, 0.00500733]]
    single_bayes_y = [(-x[0] * N0LV * 1e-9) for x in [[-1.2656724808997788], [-1.2667910872626682], [-1.4393270684622568], [-1.136326412594267], [-1.3240491047556227], [-1.2774212061595358], [-1.4453291654258265], [-1.058409493609007], [-1.4355682550189903], [-1.3566532301068548], [-1.4631440830807647], [-1.414780281754958], [-1.4710200373270308], [-1.4747398479710652], [-1.4724114636841308], [-1.4618341735689553], [-1.4754698265313406], [-1.473408423617416], [-1.1154852972496274], [-1.4307254461746146], [-1.4702387403389634], [-1.448857707912533], [-1.4750567794449925], [-1.4756881436121116], [-1.4579931826791874], [-1.4653448920866905], [-1.4757126775299998], [-1.43002186988491], [-1.4756652551724252], [-1.474240436878943], [-1.4754950640375148], [-1.4758332755906531], [-1.4757520775445783], [-1.475805151762125], [-1.4757868789968713], [-1.4758055018990712], [-1.4758382740264029], [-1.4758418321198576], [-1.4758372960357897], [-1.4758320098043913], [-1.4758206814890031], [-1.4758398271035602], [-1.4758438410986998], [-1.4758456077798043], [-1.4758343451992524], [-1.4758535029189195], [-1.4757940275901626], [-1.4758351797820457], [-1.4758375536580275], [-1.475841743284041], [-1.4758415944343337], [-1.4758059612407561], [-1.1404225780777115], [-1.4042164761514648], [-1.4758301968221421], [-1.469176992472415], [-1.401250144669892], [-1.4758449835279523], [-1.4758390374815313], [-1.4758495226031565], [-1.4758477750026084], [-1.4224240608992853], [-1.4758404287969737], [-1.4758304298018983], [-1.4758454669825256], [-1.4758519923528692], [-1.4758517807871003], [-1.4758416262014993], [-1.4758643234720523], [-1.4758434492526624], [-1.4758599324371138], [-1.47585455980859], [-1.401481109901222], [-1.4757831249362214], [-1.475843840566522], [-1.4727297741445518], [-1.4758436441767382], [-1.4758457392737432], [-1.4758606473670872], [-1.4758448079006061], [-1.4758601712386827], [-1.4758626759700215], [-1.4758443772860654], [-1.4758442872119804], [-1.475856962171024], [-1.4758476698677692], [-1.4758466751513006], [-1.4758377837238483], [-1.475840150884263], [-1.4758460101612436], [-1.4758443162968544], [-1.4758123642844567], [-1.4758561694297438], [-1.4758357913388207], [-1.4758427301943768], [-1.4758443357206517], [-1.475854876213278], [-1.4758442851744127], [-1.2997393087351605], [-1.4758631209703212], [-1.4758521178829236], [-1.4758452204909809], [-1.4758555579407233], [-1.4758403002567086], [-1.4193695991387019], [-1.4758372454658213], [-1.4758431379328552], [-1.4758618369742231], [-1.4191912442115844], [-1.4509073759765834], [-1.4368044828091082], [-1.475702841690567]]]

    # Bayes with surrogate - get_kpls_regressor
    # LoudspeakerSingleProblem-457652 - run (159) - BayesOpt: elapsed time: 309.9739203453064 s
    # LoudspeakerSingleProblem-457652 - single (283) - surrogate.predict_counter, 63
    # LoudspeakerSingleProblem-457652 - single (284) - surrogate.eval_counter, 57
    single_bayes_kpls_x = [[0.022343430632267642, 0.02189392748904622, 0.025491069237262735, 0.03241341001552258], [0.015530237170877318, 0.02944716550301958, 0.02138513291251894, 0.011189261925049253], [0.017402018315408566, 0.025209540192824086, 0.023989312870811925, 0.009633725163640947], [0.005318652750639346, 0.031471647294488383, 0.015478141730371816, 0.031113698862159436], [0.031269667905341685, 0.023155493066263896, 0.028078744677143358, 0.024668559822068243], [0.03435134568966622, 0.020567241402089563, 0.011467808515914136, 0.026683048666209436], [0.01251823682626986, 0.03310719610589872, 0.013559874374783738, 0.02216046833279241], [0.008546818322426663, 0.027663452788810813, 0.017336638845139654, 0.005613422260001947], [0.023472272232267236, 0.01519798752020182, 0.03120794142239868, 0.018121291448997153], [0.027607726200271673, 0.01787321539219883, 0.018938865299026254, 0.01633161200123198], [0.01558542, 0.0290679 , 0.02783646, 0.00694057], [0.02241285, 0.02417381, 0.03086709, 0.01072899], [0.01936407, 0.02242837, 0.01515348, 0.01879125], [0.02708246, 0.02499134, 0.02068532, 0.02127579], [0.0309444 , 0.02797725, 0.01691816, 0.02569519], [0.03341852, 0.01587253, 0.02313084, 0.01537579], [0.0138028 , 0.03276812, 0.01170746, 0.01094966], [0.0250958 , 0.01847565, 0.01369656, 0.03295637], [0.00856315, 0.01920978, 0.02188662, 0.02519601], [0.00786661, 0.03192915, 0.02605553, 0.03134348], [0.0345, 0.0345, 0.0315, 0.0345], [0.00752046, 0.0345    , 0.0315    , 0.0345    ], [0.01601727, 0.03449991, 0.02582435, 0.03449983], [0.0344983 , 0.02736679, 0.03149987, 0.01678288], [0.0344995 , 0.02446455, 0.03149988, 0.03449957], [0.01161859, 0.03449977, 0.03149948, 0.02586841], [0.005     , 0.034496  , 0.0314975 , 0.02513161], [0.023565  , 0.03449991, 0.03149947, 0.02373286], [0.02106976, 0.032397  , 0.03149968, 0.02513512], [0.03449995, 0.0345    , 0.0315    , 0.01792415], [0.03449896, 0.03449979, 0.02866506, 0.02415472], [0.01779313, 0.03449998, 0.03149951, 0.02174443], [0.0345, 0.015 , 0.0315, 0.005 ], [0.01321325, 0.0344979 , 0.03149281, 0.02058109], [0.00974004, 0.03449967, 0.02830687, 0.02345152], [0.01130134, 0.03449699, 0.03149917, 0.0174047 ], [0.00596215, 0.03449841, 0.03148721, 0.01893968], [0.01596491, 0.03449929, 0.03149928, 0.01913865], [0.01200755, 0.0345    , 0.03149801, 0.01976055], [0.0344994 , 0.015002  , 0.0315    , 0.02025577], [0.01813509, 0.0345    , 0.03128993, 0.01210974], [0.01474564, 0.03449019, 0.03137295, 0.02238699], [0.01182965, 0.03447879, 0.03134174, 0.005     ], [0.0308588 , 0.03449364, 0.03132434, 0.0050214 ], [0.02520229, 0.02155809, 0.02574653, 0.03360149], [0.03067962, 0.02096464, 0.02864666, 0.03444508], [0.005     , 0.03443042, 0.03051109, 0.00500713], [0.00500238, 0.0345    , 0.03006324, 0.01186473], [0.02567531, 0.03394164, 0.03028536, 0.005     ], [0.00500268, 0.03043534, 0.03025309, 0.0186878 ], [0.005     , 0.0345    , 0.0315    , 0.00500923], [0.02334849, 0.03446961, 0.03137579, 0.00500564], [0.0280023 , 0.03449973, 0.03015886, 0.01108114], [0.03449935, 0.03449978, 0.02254655, 0.03449934], [0.0344995 , 0.03449969, 0.02841272, 0.00500282], [0.01638568, 0.0345    , 0.0315    , 0.00500603], [0.01272618, 0.0344934 , 0.0315    , 0.0050236 ], [0.00881522, 0.034489  , 0.03128236, 0.00500622], [0.0345, 0.0345, 0.01  , 0.005 ], [0.01135607, 0.03448804, 0.03096708, 0.00501298], [0.00500356, 0.03300737, 0.03149972, 0.01064417], [0.00500325, 0.02868491, 0.03149917, 0.03449943], [0.03449939, 0.02079746, 0.0314995 , 0.00500256], [0.03449931, 0.02887775, 0.02668088, 0.03449947], [0.00500303, 0.03195712, 0.0315    , 0.02343268], [0.00500353, 0.03385403, 0.02980007, 0.00691228], [0.01747397, 0.03393634, 0.03149987, 0.00643966], [0.01422768, 0.03427456, 0.03125855, 0.00507189], [0.01443704, 0.03427419, 0.03047075, 0.00515944], [0.03449808, 0.03269561, 0.03149966, 0.00879057], [0.005 , 0.015 , 0.0315, 0.005 ], [0.01614537, 0.03434261, 0.03034591, 0.0063937 ], [0.02133262, 0.03412976, 0.0306975 , 0.00814957], [0.01117023, 0.03447825, 0.02909298, 0.00564727], [0.00500311, 0.02870994, 0.01687321, 0.0344995 ], [0.01374914, 0.0344686 , 0.03069915, 0.00709029], [0.01581975, 0.03433852, 0.03069509, 0.00665981], [0.01421238, 0.03442871, 0.03060885, 0.00604318], [0.01214986, 0.03420995, 0.03047533, 0.00700076], [0.01491305, 0.03427028, 0.03062701, 0.00688579], [0.01403859, 0.03429885, 0.03064557, 0.00679494], [0.01418512, 0.03431059, 0.03068295, 0.00663554], [0.01614102, 0.03431383, 0.03068103, 0.00639584], [0.0187855 , 0.0342392 , 0.03040822, 0.0056683 ], [0.014921  , 0.03415891, 0.03053753, 0.00734579], [0.01560236, 0.03433749, 0.03033033, 0.00768087], [0.01586127, 0.03428206, 0.0306988 , 0.0074432 ], [0.01578252, 0.03425629, 0.03063593, 0.0073064 ], [0.01538458, 0.03406757, 0.03051526, 0.00859323], [0.01516815, 0.03408596, 0.03045913, 0.00733708], [0.01493508, 0.03417097, 0.03083233, 0.0081411 ], [0.01576321, 0.03424789, 0.03039925, 0.00814301], [0.03449917, 0.03210531, 0.0275488 , 0.01886997], [0.03449932, 0.02313849, 0.03149949, 0.01828336], [0.00500297, 0.03307044, 0.02938774, 0.01598027], [0.00500283, 0.02570144, 0.03149949, 0.0230466 ], [0.02416787, 0.03356089, 0.03008882, 0.01383478], [0.01526482, 0.03421355, 0.03021911, 0.00860075], [0.01531798, 0.03406554, 0.03063347, 0.00693393], [0.01560249, 0.03426142, 0.03098565, 0.00751038], [0.03449926, 0.01806079, 0.03149943, 0.01407021], [0.01184305, 0.03417042, 0.03039743, 0.005     ], [0.01532862, 0.0345    , 0.03032776, 0.005     ], [0.03449945, 0.03244417, 0.0260106 , 0.03449939], [0.01722046, 0.03419994, 0.03052129, 0.00690965], [0.01193884, 0.03419829, 0.03055902, 0.00749009], [0.01452801, 0.0345    , 0.03036779, 0.00617682], [0.02451936, 0.03375798, 0.02705632, 0.0177979 ], [0.0345, 0.015 , 0.01  , 0.005 ], [0.03449953, 0.02482098, 0.01000132, 0.0345    ], [0.02258787, 0.02719577, 0.03149936, 0.025227  ], [0.01882443, 0.03370712, 0.0315    , 0.01109907], [0.01549515, 0.03418883, 0.03059561, 0.005     ], [0.0145363 , 0.03426293, 0.03045849, 0.0061528 ], [0.02138103, 0.0345    , 0.0300662 , 0.005     ], [0.01800586, 0.0345    , 0.03014515, 0.00795298], [0.02035596, 0.03420217, 0.0308774 , 0.005     ], [0.0345    , 0.03329022, 0.03016341, 0.01530315], [0.01647846, 0.03376142, 0.03032015, 0.00703762], [0.01555657, 0.03420203, 0.03051744, 0.00610966]]
    single_bayes_kpls_y = [(-x[0] * N0LV * 1e-9) for x in [[-1.33126688234834], [-1.378428983622866], [-1.3725592214497198], [-1.375878832893289], [-1.4033787570332987], [-1.2740802646272966], [-1.319476162770135], [-1.2708757233393457], [-1.2965904550433232], [-1.2588705919506764], [-1.3281405875074495], [-1.3990842923975562], [-1.2172528034207457], [-1.345827102695184], [-1.3480559073547214], [-1.3371590676719656], [-1.152141250481863], [-1.2503019126132873], [-1.22324356416922], [-1.4362131968518104], [-1.4307254461746242], [-1.443824547232447], [-1.4344489327864995], [-1.4379779416940388], [-1.3954225337477886], [-1.46296819889977], [-1.4607160336462683], [-1.4633662102003968], [-1.445487374577576], [-1.4580371206512037], [-1.4512439045657712], [-1.4666774627091101], [-1.3862347285484151], [-1.4681982732040137], [-1.4556170997404914], [-1.4669847788832717], [-1.46672473124219], [-1.4677375441810812], [-1.4670015955677453], [-1.3787371117475127], [-1.4725279740027115], [-1.4667056737867594], [-1.4747504528635988], [-1.468687558456644], [-1.329860544711238], [-1.357462641911618], [-1.4725040748730323], [-1.4704440909870111], [-1.4731512543857963], [-1.4513792656537872], [-1.468928610062493], [-1.4740320497516761], [-1.4701828870598734], [-1.4067967268186379], [-1.4534481763386375], [-1.4741605618862728], [-1.4740320497516761], [-1.4727561898293804], [-1.0153630804112472], [-1.4746274510487476], [-1.4682481514372685], [-1.4089330160170452], [-1.412261316140777], [-1.4091170543306786], [-1.4593171319811982], [-1.4734366484546049], [-1.4746274510487476], [-1.4750527376895128], [-1.4758182536428903], [-1.462634367779167], [-1.115485297632374], [-1.4759883682991963], [-1.4747125083769006], [-1.4730964191419926], [-1.3519747838520457], [-1.4758182536428903], [-1.4759033109710433], [-1.475648138986584], [-1.4757331963147373], [-1.4759033109710433], [-1.4759883682991963], [-1.4758182536428903], [-1.4761584829555026], [-1.4752228523458188], [-1.4760734256273493], [-1.4759033109710433], [-1.4761584829555026], [-1.4762435402836556], [-1.4760734256273493], [-1.4759883682991963], [-1.4758182536428903], [-1.4760734256273493], [-1.454298749620168], [-1.417503890988978], [-1.4681355201012682], [-1.389858754918688], [-1.4715853456440169], [-1.4746015692607501], [-1.4750763158950733], [-1.4747471582286094], [-1.3964214176787433], [-1.474965541680398], [-1.4753896486737266], [-1.4213888015082445], [-1.4750984707380084], [-1.474658538856869], [-1.4750415011418896], [-1.4568697995893845], [-0.9610926748646113], [-1.3093573716986537], [-1.4229829629855741], [-1.4716264903523248], [-1.475538402619148], [-1.4752187398853702], [-1.4744091600549771], [-1.474310391325032], [-1.4749623767028357], [-1.4636159321425115], [-1.4746838586773663], [-1.4753200191673592]]]

    # COMSOL - Monte Carlo
    single_comsol_monte_carlo_y = [(x * 1e-1) for x in [2.4194373073265827	,2.4336164855064153	,2.321956506029703	,2.255407070419544	,2.368029879308222	,2.297351618667698	,2.3125515722468832	,2.302763456896952	,2.232375870242566	,2.342002624402161	,2.2033484426331227	,2.3644257186035222	,2.291681031640667	,2.435654366360054	,2.367299221002449	,2.310424482927342	,2.3529321595894053	,2.256087301456675	,2.4062838687335875	,2.100517381797827	,2.088180111706956	,2.358306620396386	,2.185954033034899	,2.187748202196359	,2.3484963673782007	,2.1582880234467843	,2.1884970073817622	,2.35337902245614	,2.3904200947328107	,2.4216218440732815	,2.246147116832287	,2.4276832843812857	,2.3877339394796575	,2.4296833581893575	,2.2531144057385126	,2.098689285351291	,2.211243226790937	,2.28442656438158	,2.3291537172426646	,2.2680768532021185	,2.341031459096751	,2.397425264698224	,2.280174200979342	,2.4470727184294025	,2.376738138763974	,2.422876346713807	,2.1069865921384228	,2.199466529611379	,2.2367326925694986	,2.3345831540187474	,2.230208807522932	,2.374922418118069	,2.346101901047889	,2.3003880787388247	,2.3127361812702913	,2.2680942838826703	,2.1275773226726633	,2.4208986823967202	,2.2743792426878504	,2.108087520608504	,2.396282903063105	,2.2171002997703972	,2.34885955651122	,2.2466232186249813	,2.3953066456296663	,2.3615003851042724	,2.269175370756033	,2.4225792634011403	,2.3206673137399894	,2.0772720728328484	,2.2887116902922395	,2.169192671979239	,2.1243078712276233	,2.3509336506317355	,2.3493210833194254	,2.3177183187523753	,2.2847912427439003	,2.261198798948291	,2.3538357645395065	,2.2187624408516564	,2.4390798074908973	,2.300933014242326	,2.257319194053616	,2.149079810761452	,2.3119337428297344	,2.3584115279956	,2.214879122572029	,2.371207515202898	,2.324510820939045	,2.366132080534961	,2.321306430314481	,2.2601703144591836	,2.444769235545359	,2.245686109940724	,2.4038077097165567	,2.3982024351770637	,2.1200209753957306	,2.324878165294974	,2.249141088141035	,2.3003917820278814	,2.4304694802045	,2.1655625590103136	,2.3557601748847223	,2.327410660851444	,2.4175239886126025	,2.1279220520215296	,2.2307601832453385	,2.4314731375460616	,2.398531294372601	,2.2487679394476037	,2.431314842033159	,2.171781252382866	,2.2494916925780024]]
    # COMSOL - BOBYQA
    single_comsol_bobyqa_y = [(x) for x in [0.24167, 0.24198, 0.24157, 0.24205, 0.24181, 0.24229, 0.24122, 0.24093, 0.24194, 0.24437, 0.24447, 0.24469, 0.24382, 0.24469]]

    import pylab as pl
    from matplotlib import rc
    rc('text', usetex=True)
    rc('font', size=16)

    single_bayes_n = range(len(single_bayes_y))
    pl.figure(figsize=[10, 4])
    pl.plot(single_bayes_n, single_bayes_y, color='blue', marker='o', markersize=3, linestyle='none', label="Agros - BOBYQA - full model")
    single_bayes_kpls_n = range(len(single_bayes_kpls_y))
    pl.plot(single_bayes_kpls_n, single_bayes_kpls_y, color='red', marker='o', markersize=3, linestyle='none', label="Agros - BOBYQA - full model + surrogate regression")
    single_comsol_monte_carlo_n = range(len(single_comsol_monte_carlo_y))
    pl.plot(single_comsol_monte_carlo_n, single_comsol_monte_carlo_y, color='green', marker='o', markersize=2, linestyle='none', label="COMSOL - Monte Carlo")
    single_comsol_bobyqa_n = range(len(single_comsol_bobyqa_y))
    pl.plot(single_comsol_bobyqa_n, single_comsol_bobyqa_y, color='magenta', marker='o', markersize=2, linestyle='none', label="COMSOL - BOBYQA")
    pl.grid()
    pl.xlabel("$N~(\mathrm{-})$")
    pl.ylabel("$BL~(\mathrm{T \cdot m})$")
    pl.xlim(-0.02, 80)
    pl.legend()
    pl.tight_layout()
    pl.savefig("/tmp/single_bayes.pdf")

def results_multi():
    # N0LV = 1.0 / (1.04615e-06 * 1e-3) * 100 / 6e-4
    N0LV = 1.0 / 1e-3 * 100 * 1.0 / 6e-4
    # LoudspeakerMultiProblem-254079 - run (174) - NSGA_II: elapsed time: 11828.238897800446 s
    # LoudspeakerMultiProblem-254079 - nsga2 (246) - surrogate.predict_counter: 0
    # LoudspeakerMultiProblem-254079 - nsga2 (247) - surrogate.eval_counter: 2020
    multi_nsga_x = [[0.013566703391, 0.023298766579999998, 0.014884344625999999, 0.009134216524], [0.005885132367, 0.029020592137999998, 0.015528235901, 0.011329731154], [0.025210766994, 0.022168071557999998, 0.010348872391, 0.008873417447], [0.011687725563, 0.030789035991, 0.013376162149999999, 0.012167874976], [0.017857859311, 0.03292530461, 0.013163057381, 0.005178978537], [0.01592465764, 0.033937537817, 0.010860664003999999, 0.008956320431], [0.016498761384, 0.030129834334999998, 0.013146066323, 0.015331411890999999], [0.008759395918, 0.016503167059, 0.020002955436, 0.025924240742], [0.026069026010999998, 0.022142380996, 0.021465612906, 0.0061237356429999995], [0.025918374916, 0.025048936009999998, 0.020259977922, 0.005069682719], [0.021631769432, 0.024363668354, 0.022388562639, 0.009204791859], [0.018560466261, 0.021843405851999998, 0.021829556384, 0.015432210364999999], [0.011045945086, 0.016421164506, 0.019441837067999998, 0.031925029612], [0.008229366865, 0.019731914678, 0.022400806744, 0.030815707982], [0.010396310062, 0.024383213258, 0.02000618282, 0.033666381416], [0.019063333599, 0.015520052597, 0.027468479655, 0.029094921491], [0.025065108342, 0.018023724038, 0.031487052689999996, 0.020470547621], [0.027380089079, 0.027131922828, 0.026266958511, 0.019882962663999998], [0.016253567801, 0.027912183109, 0.021914156253, 0.032208921312], [0.031497340027, 0.021364596731, 0.021618451893, 0.026755894877999998], [0.011797729622127103, 0.023298766579999998, 0.014884344625999999, 0.008274221003045607], [0.013566703391, 0.023298766579999998, 0.014884344625999999, 0.009134216524], [0.027380089079, 0.027131922828, 0.026266958511, 0.019882962663999998], [0.027865765320253165, 0.03025499570224346, 0.026266958511, 0.019882962663999998], [0.005885132367, 0.029020592137999998, 0.015528235901, 0.011329731154], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.010067962747332626], [0.021631769432, 0.024363668354, 0.022388562639, 0.009204791859], [0.021631769432, 0.024363668354, 0.022388562639, 0.010904813692769898], [0.025918374916, 0.025048936009999998, 0.020259977922, 0.005069682719], [0.026069026010999998, 0.022142380996, 0.021465612906, 0.0061237356429999995], [0.025918374916, 0.025048936009999998, 0.020259977922, 0.005069682719], [0.02599506191072659, 0.025048936009999998, 0.020259977922, 0.005069682719], [0.025210766994, 0.022168071557999998, 0.010348872391, 0.008873417447], [0.018560466261, 0.021843405851999998, 0.021829556384, 0.015432210364999999], [0.016498761384, 0.030129834334999998, 0.013146066323, 0.015331411890999999], [0.011687725563, 0.030789035991, 0.013610072863690465, 0.02422552126459037], [0.027380089079, 0.027131922828, 0.026266958511, 0.02127169636294283], [0.016253567801, 0.027912183109, 0.021914156253, 0.032208921312], [0.016253567801, 0.027912183109, 0.021914156253, 0.032208921312], [0.025065108342, 0.018023724038, 0.031487052689999996, 0.020470547621], [0.011797729622127103, 0.023298766579999998, 0.014884344625999999, 0.008274221003045607], [0.027380089079, 0.027131922828, 0.026266958511, 0.019882962663999998], [0.027865765320253165, 0.03025499570224346, 0.026266958511, 0.019882962663999998], [0.027865765320253165, 0.03025499570224346, 0.026266958511, 0.019882962663999998], [0.005885132367, 0.029020592137999998, 0.015528235901, 0.011329731154], [0.005885132367, 0.029020592137999998, 0.021187448621655613, 0.011329731154], [0.021631769432, 0.030979679820081554, 0.022388562639, 0.00817793014749357], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.010067962747332626], [0.011836539377640546, 0.023298766579999998, 0.015261307925685122, 0.008274221003045607], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.010067962747332626], [0.005885132367, 0.029020592137999998, 0.016689872400627, 0.011329731154], [0.005885132367, 0.029020592137999998, 0.02009173182150452, 0.011755453223410364], [0.013566703391, 0.023298766579999998, 0.014884344625999999, 0.009134216524], [0.022904711348821574, 0.023221441597364237, 0.031487052689999996, 0.020470547621], [0.027380089079, 0.027131922828, 0.026266958511, 0.02127169636294283], [0.014720437785334443, 0.0289014670430579, 0.02178162625374522, 0.032208921312], [0.021631769432, 0.024363668354, 0.022388562639, 0.009204791859], [0.021631769432, 0.024363668354, 0.022388562639, 0.009204791859], [0.025918374916, 0.025048936009999998, 0.020259977922, 0.005069682719], [0.026069026010999998, 0.022142380996, 0.021465612906, 0.0061237356429999995], [0.011797729622127103, 0.023298766579999998, 0.014884344625999999, 0.008274221003045607], [0.027865765320253165, 0.03025499570224346, 0.026266958511, 0.019330747181077204], [0.005885132367, 0.029020592137999998, 0.021187448621655613, 0.011329731154], [0.005885132367, 0.029020592137999998, 0.015528235901, 0.011329731154], [0.021631769432, 0.030979679820081554, 0.022388562639, 0.00817793014749357], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.010067962747332626], [0.021631769432, 0.030979679820081554, 0.022388562639, 0.00817793014749357], [0.005885132367, 0.029020592137999998, 0.015528235901, 0.011329731154], [0.005885132367, 0.029020592137999998, 0.02009173182150452, 0.011755453223410364], [0.005842011669205989, 0.027923498710553024, 0.015528235901, 0.011726486733972247], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.010067962747332626], [0.025918374916, 0.03403731514777877, 0.023762341307804268, 0.009124565214062895], [0.005885132367, 0.029020592137999998, 0.016689872400627, 0.011329731154], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.010067962747332626], [0.011836539377640546, 0.023298766579999998, 0.015261307925685122, 0.008274221003045607], [0.005832126552756788, 0.028802895782705416, 0.02228628261051206, 0.011329731154], [0.005885132367, 0.029020592137999998, 0.02009173182150452, 0.011755453223410364], [0.013566703391, 0.023298766579999998, 0.014884344625999999, 0.009134216524], [0.027380089079, 0.027131922828, 0.026266958511, 0.019882962663999998], [0.027865765320253165, 0.03025499570224346, 0.026266958511, 0.019882962663999998], [0.011797729622127103, 0.023298766579999998, 0.014884344625999999, 0.008274221003045607], [0.027865765320253165, 0.03025499570224346, 0.026266958511, 0.017616142742732512], [0.005885132367, 0.029020592137999998, 0.021187448621655613, 0.011329731154], [0.005885132367, 0.029020592137999998, 0.015528235901, 0.011329731154], [0.021631769432, 0.030979679820081554, 0.022388562639, 0.00817793014749357], [0.005842011669205989, 0.027923498710553024, 0.015528235901, 0.011726486733972247], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.010067962747332626], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.010067962747332626], [0.021631769432, 0.030979679820081554, 0.022388562639, 0.00817793014749357], [0.005885132367, 0.029020592137999998, 0.015528235901, 0.011329731154], [0.005885132367, 0.029020592137999998, 0.02009173182150452, 0.011755453223410364], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.010067962747332626], [0.005832126552756788, 0.028802895782705416, 0.02228628261051206, 0.011329731154], [0.025918374916, 0.03403731514777877, 0.023762341307804268, 0.009124565214062895], [0.005885132367, 0.029020592137999998, 0.016689872400627, 0.011329731154], [0.025918374916, 0.03403731514777877, 0.02385338158597872, 0.009200605499479406], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.010067962747332626], [0.025918374916, 0.03403731514777877, 0.023762341307804268, 0.009124565214062895], [0.005885132367, 0.029020592137999998, 0.016047018231016688, 0.011329731154], [0.005885132367, 0.029020592137999998, 0.016689872400627, 0.011329731154], [0.011797729622127103, 0.023298766579999998, 0.014884344625999999, 0.008274221003045607], [0.027865765320253165, 0.03025499570224346, 0.026266958511, 0.017616142742732512], [0.027865765320253165, 0.03284106898246328, 0.026266958511, 0.017616142742732512], [0.005832126552756788, 0.028802895782705416, 0.02228628261051206, 0.011329731154], [0.005885132367, 0.029020592137999998, 0.021187448621655613, 0.011329731154], [0.021631769432, 0.030979679820081554, 0.022388562639, 0.00817793014749357], [0.005842011669205989, 0.027923498710553024, 0.015528235901, 0.011726486733972247], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.01185839409880759], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.010067962747332626], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.010067962747332626], [0.027229588526991373, 0.03403731514777877, 0.02385338158597872, 0.00500224548388097], [0.021631769432, 0.030979679820081554, 0.022388562639, 0.00817793014749357], [0.005885132367, 0.029020592137999998, 0.02009173182150452, 0.011755453223410364], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.010067962747332626], [0.025918374916, 0.03403731514777877, 0.023762341307804268, 0.009124565214062895], [0.025918374916, 0.03403731514777877, 0.02385338158597872, 0.009504484776676956], [0.005842011669205989, 0.027923498710553024, 0.0159995273103773, 0.011726486733972247], [0.005885132367, 0.029020592137999998, 0.016689872400627, 0.011329731154], [0.025918374916, 0.03403731514777877, 0.02385338158597872, 0.009200605499479406], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.010067962747332626], [0.011797729622127103, 0.023098243852321135, 0.014884344625999999, 0.008274221003045607], [0.011797729622127103, 0.023298766579999998, 0.014884344625999999, 0.008274221003045607], [0.027865765320253165, 0.03025499570224346, 0.026266958511, 0.017616142742732512], [0.027865765320253165, 0.03284106898246328, 0.026266958511, 0.017616142742732512], [0.005884626754317762, 0.02896857350797836, 0.0159995273103773, 0.011726486733972247], [0.006130346580002135, 0.029288938922685605, 0.022303662661445185, 0.011329731154], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.01185839409880759], [0.005832126552756788, 0.028802895782705416, 0.02228628261051206, 0.011329731154], [0.02106037756761926, 0.03193355628248827, 0.022388562639, 0.010067962747332626], [0.005885132367, 0.029020592137999998, 0.021187448621655613, 0.011329731154], [0.005842011669205989, 0.027923498710553024, 0.015528235901, 0.011726486733972247], [0.021631769432, 0.030979679820081554, 0.022388562639, 0.00817793014749357], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.010067962747332626], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.010067962747332626], [0.005842011669205989, 0.027923498710553024, 0.0159995273103773, 0.011726486733972247], [0.005885132367, 0.029020592137999998, 0.02009173182150452, 0.011755453223410364], [0.021631769432, 0.030979679820081554, 0.022388562639, 0.00817793014749357], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.010067962747332626], [0.021631769432, 0.030979679820081554, 0.022388562639, 0.00817793014749357], [0.025918374916, 0.03403731514777877, 0.023762341307804268, 0.009124565214062895], [0.011797729622127103, 0.0211521745302615, 0.01599186179820934, 0.008276021449724743], [0.011797729622127103, 0.023098243852321135, 0.014884344625999999, 0.008274221003045607], [0.011797729622127103, 0.023298766579999998, 0.014884344625999999, 0.008274221003045607], [0.027865765320253165, 0.03025499570224346, 0.026266958511, 0.017616142742732512], [0.027865765320253165, 0.03284106898246328, 0.026266958511, 0.017616142742732512], [0.006130346580002135, 0.029288938922685605, 0.022303662661445185, 0.011329731154], [0.005885132367, 0.029020592137999998, 0.021187448621655613, 0.011329731154], [0.005832126552756788, 0.028802895782705416, 0.02228628261051206, 0.011329731154], [0.02106037756761926, 0.03193355628248827, 0.022388562639, 0.010067962747332626], [0.005884626754317762, 0.02896857350797836, 0.0159995273103773, 0.011726486733972247], [0.005885132367, 0.029020592137999998, 0.021187448621655613, 0.011329731154], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.01185839409880759], [0.005832126552756788, 0.028802895782705416, 0.02228628261051206, 0.011329731154], [0.005842011669205989, 0.027923498710553024, 0.015528235901, 0.011726486733972247], [0.021631769432, 0.030979679820081554, 0.022388562639, 0.00817793014749357], [0.005842011669205989, 0.027923498710553024, 0.0159995273103773, 0.011726486733972247], [0.025918374916, 0.03403731514777877, 0.023762341307804268, 0.009124565214062895], [0.021631769432, 0.030979679820081554, 0.022388562639, 0.00817793014749357], [0.021631769432, 0.030979679820081554, 0.022388562639, 0.00817793014749357], [0.021631769432, 0.030979679820081554, 0.022388562639, 0.00817793014749357], [0.011797729622127103, 0.0211521745302615, 0.01599186179820934, 0.008276021449724743], [0.027865765320253165, 0.03025499570224346, 0.026266958511, 0.017616142742732512], [0.027865765320253165, 0.03284106898246328, 0.026266958511, 0.017616142742732512], [0.006130346580002135, 0.029288938922685605, 0.022303662661445185, 0.011329731154], [0.02106037756761926, 0.033919398083310216, 0.022388562639, 0.01185839409880759], [0.005885132367, 0.029020592137999998, 0.021187448621655613, 0.011329731154], [0.005832126552756788, 0.028802895782705416, 0.02228628261051206, 0.011329731154], [0.005842011669205989, 0.026811092585102503, 0.0159995273103773, 0.011726486733972247], [0.005884626754317762, 0.02896857350797836, 0.0159995273103773, 0.011726486733972247], [0.005885132367, 0.029020592137999998, 0.021187448621655613, 0.011329731154], [0.005884626754317762, 0.02896857350797836, 0.0159995273103773, 0.011726486733972247], [0.025918374916, 0.03403731514777877, 0.023762341307804268, 0.009124565214062895], [0.005832126552756788, 0.028802895782705416, 0.02228628261051206, 0.011329731154], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.021631769432, 0.032820999257901884, 0.022388562639, 0.00817793014749357], [0.01183703960636509, 0.0211521745302615, 0.0164763782119552, 0.007814721705852947], [0.00614135060467614, 0.028799431622867695, 0.022454176895790826, 0.011329731154], [0.005832126552756788, 0.028802895782705416, 0.02231641947885232, 0.011329731154], [0.005832545731495254, 0.028802895782705416, 0.02228628261051206, 0.011329731154], [0.005832126552756788, 0.028802895782705416, 0.02228628261051206, 0.011329731154], [0.011797729622127103, 0.0211521745302615, 0.01599186179820934, 0.008276021449724743], [0.027865765320253165, 0.03284106898246328, 0.026266958511, 0.017616142742732512], [0.027865765320253165, 0.03284106898246328, 0.026266958511, 0.017616142742732512], [0.006130346580002135, 0.029288938922685605, 0.022303662661445185, 0.011329731154], [0.0059606950143158595, 0.03301372413651443, 0.02228628261051206, 0.011053185608675884], [0.005842011669205989, 0.026811092585102503, 0.0159995273103773, 0.011726486733972247], [0.005885132367, 0.029020592137999998, 0.021187448621655613, 0.011329731154], [0.005884626754317762, 0.02896857350797836, 0.0159995273103773, 0.011726486733972247], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.005884626754317762, 0.02896857350797836, 0.0158764282336725, 0.011726486733972247], [0.005885132367, 0.029020592137999998, 0.021187448621655613, 0.011329731154], [0.005842011669205989, 0.02460223998371875, 0.01676416730923022, 0.01208943238934826], [0.00614135060467614, 0.028799431622867695, 0.022454176895790826, 0.011329731154], [0.005884626754317762, 0.02896857350797836, 0.0159995273103773, 0.011726486733972247], [0.00614135060467614, 0.028831744305387434, 0.022454176895790826, 0.011329731154], [0.005885132367, 0.029020592137999998, 0.021187448621655613, 0.011329731154], [0.0052723709153796915, 0.028799431622867695, 0.022454176895790826, 0.011329731154], [0.005884626754317762, 0.03147107353997999, 0.026607750554734088, 0.0160903922592919], [0.005885132367, 0.029142385831889592, 0.021187448621655613, 0.011329731154], [0.011797729622127103, 0.0211521745302615, 0.01599186179820934, 0.008276021449724743], [0.027865765320253165, 0.03284106898246328, 0.026266958511, 0.017616142742732512], [0.027865765320253165, 0.03284106898246328, 0.026266958511, 0.017616142742732512], [0.027865765320253165, 0.03344086592509921, 0.026266958511, 0.019592834252715022], [0.005884626754317762, 0.03147107353997999, 0.026607750554734088, 0.0160903922592919], [0.005884626754317762, 0.02896857350797836, 0.0158764282336725, 0.011726486733972247], [0.006130346580002135, 0.029288938922685605, 0.022303662661445185, 0.011329731154], [0.0059606950143158595, 0.03301372413651443, 0.02228628261051206, 0.011053185608675884], [0.00614135060467614, 0.030942735893227324, 0.022454176895790826, 0.011329731154], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.005842011669205989, 0.02460223998371875, 0.01676416730923022, 0.01208943238934826], [0.005842011669205989, 0.026811092585102503, 0.0159995273103773, 0.011726486733972247], [0.005885132367, 0.029020592137999998, 0.021187448621655613, 0.011329731154], [0.005884626754317762, 0.02896857350797836, 0.0159995273103773, 0.011726486733972247], [0.027865765320253165, 0.03284106898246328, 0.026266958511, 0.017616142742732512], [0.00614135060467614, 0.02887357029945859, 0.022454176895790826, 0.011329731154], [0.005884626754317762, 0.03147107353997999, 0.026607750554734088, 0.0160903922592919], [0.00614135060467614, 0.028799431622867695, 0.022454176895790826, 0.011329731154], [0.005885132367, 0.029020592137999998, 0.021187448621655613, 0.011329731154], [0.005884626754317762, 0.02896857350797836, 0.0159995273103773, 0.011726486733972247], [0.011797729622127103, 0.0211521745302615, 0.01599186179820934, 0.008276021449724743], [0.027865765320253165, 0.03284106898246328, 0.026266958511, 0.017616142742732512], [0.027865765320253165, 0.03284106898246328, 0.026266958511, 0.017616142742732512], [0.02768909447316994, 0.03284106898246328, 0.027061575295744756, 0.017616142742732512], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.005884626754317762, 0.02896857350797836, 0.0158764282336725, 0.011726486733972247], [0.00614135060467614, 0.030942735893227324, 0.022454176895790826, 0.011329731154], [0.027865765320253165, 0.03284106898246328, 0.026266958511, 0.017616142742732512], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.005842011669205989, 0.026811092585102503, 0.0159995273103773, 0.011726486733972247], [0.00614135060467614, 0.02887357029945859, 0.022454176895790826, 0.011329731154], [0.005884626754317762, 0.03147107353997999, 0.026607750554734088, 0.015141795253011216], [0.005842011669205989, 0.02460223998371875, 0.01676416730923022, 0.01208943238934826], [0.006130346580002135, 0.029288938922685605, 0.022303662661445185, 0.011329731154], [0.008919550765794107, 0.02898182439802671, 0.02656900569913856, 0.011726486733972247], [0.005842011669205989, 0.026811092585102503, 0.0159995273103773, 0.011726486733972247], [0.006663849630664917, 0.031366630395214846, 0.02228628261051206, 0.011053185608675884], [0.006130346580002135, 0.029288938922685605, 0.022446193355951445, 0.011329731154], [0.005885132367, 0.029020592137999998, 0.021187448621655613, 0.011329731154], [0.005884626754317762, 0.02896857350797836, 0.0159995273103773, 0.011726486733972247], [0.011797729622127103, 0.0211521745302615, 0.01599186179820934, 0.008276021449724743], [0.026481548159307602, 0.03284106898246328, 0.027061575295744756, 0.017616142742732512], [0.006130346580002135, 0.03362859377192008, 0.0248647772796683, 0.011329731154], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.005884626754317762, 0.02896857350797836, 0.0158764282336725, 0.011726486733972247], [0.005842011669205989, 0.026811092585102503, 0.0159995273103773, 0.011726486733972247], [0.005842011669205989, 0.026811092585102503, 0.0159995273103773, 0.011726486733972247], [0.005842011669205989, 0.02460223998371875, 0.01676416730923022, 0.01208943238934826], [0.008919550765794107, 0.02898182439802671, 0.02656900569913856, 0.011726486733972247], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.005722164209458369, 0.029512206212687925, 0.0158764282336725, 0.011726486733972247], [0.006061838452605032, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.00614135060467614, 0.02887357029945859, 0.0217687508865932, 0.011329731154], [0.006663849630664917, 0.031366630395214846, 0.02228628261051206, 0.011053185608675884], [0.005884626754317762, 0.02896857350797836, 0.0159995273103773, 0.011726486733972247], [0.006663849630664917, 0.031366630395214846, 0.02228628261051206, 0.010674506777137717], [0.006130346580002135, 0.029288938922685605, 0.022446193355951445, 0.011329731154], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.005635027790191864, 0.02922886077596469, 0.021187448621655613, 0.011329731154], [0.009225263747613881, 0.021094107195895863, 0.01599186179820934, 0.008446858641255368], [0.011797729622127103, 0.0211521745302615, 0.01599186179820934, 0.008276021449724743], [0.026481548159307602, 0.03284106898246328, 0.027061575295744756, 0.017616142742732512], [0.026481548159307602, 0.03314842851423333, 0.027061575295744756, 0.017616142742732512], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.006663849630664917, 0.031366630395214846, 0.02228628261051206, 0.011053185608675884], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.005842011669205989, 0.026811092585102503, 0.0159995273103773, 0.011726486733972247], [0.005842011669205989, 0.02460223998371875, 0.01676416730923022, 0.01208943238934826], [0.005842011669205989, 0.026811092585102503, 0.0159995273103773, 0.011726486733972247], [0.005722164209458369, 0.029512206212687925, 0.0158764282336725, 0.011726486733972247], [0.00614135060467614, 0.02887357029945859, 0.0217687508865932, 0.011329731154], [0.006130346580002135, 0.0305806200601356, 0.022446193355951445, 0.011329731154], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.006061838452605032, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.005884626754317762, 0.02881741611893251, 0.0159995273103773, 0.011726486733972247], [0.011615600164291758, 0.031366630395214846, 0.02228628261051206, 0.010674506777137717], [0.005679909017294417, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.009225263747613881, 0.021094107195895863, 0.01599186179820934, 0.008446858641255368], [0.011797729622127103, 0.0211521745302615, 0.01599186179820934, 0.008276021449724743], [0.026481548159307602, 0.03284106898246328, 0.027061575295744756, 0.017616142742732512], [0.026481548159307602, 0.03314842851423333, 0.027061575295744756, 0.017616142742732512], [0.026481548159307602, 0.03314842851423333, 0.027061575295744756, 0.017616142742732512], [0.011751772760366123, 0.031248356835240788, 0.026607750554734088, 0.0160903922592919], [0.005959695280746272, 0.02881741611893251, 0.0159995273103773, 0.011726486733972247], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.005842011669205989, 0.026811092585102503, 0.0159995273103773, 0.011726486733972247], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.005842011669205989, 0.026811092585102503, 0.0159995273103773, 0.011726486733972247], [0.005842011669205989, 0.02460223998371875, 0.01676416730923022, 0.01208943238934826], [0.006663849630664917, 0.031366630395214846, 0.02228628261051206, 0.011053185608675884], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.006663849630664917, 0.031366630395214846, 0.02228628261051206, 0.011053185608675884], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.005884626754317762, 0.02881741611893251, 0.0159995273103773, 0.011726486733972247], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.005679909017294417, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.009225263747613881, 0.021094107195895863, 0.01599186179820934, 0.008446858641255368], [0.011797729622127103, 0.0211521745302615, 0.01599186179820934, 0.008276021449724743], [0.026481548159307602, 0.03284106898246328, 0.027061575295744756, 0.017616142742732512], [0.026481548159307602, 0.03314842851423333, 0.027061575295744756, 0.017616142742732512], [0.026481548159307602, 0.03314842851423333, 0.027061575295744756, 0.017616142742732512], [0.026481548159307602, 0.03444089134738728, 0.027839491886923834, 0.017616142742732512], [0.026481548159307602, 0.03284106898246328, 0.02694053342946806, 0.017616142742732512], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.006041499892570327, 0.031776306981364855, 0.026941845894986747, 0.0160903922592919], [0.005959695280746272, 0.02881741611893251, 0.0159995273103773, 0.011726486733972247], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.005842011669205989, 0.026811092585102503, 0.0159995273103773, 0.011726486733972247], [0.005842011669205989, 0.02460223998371875, 0.01676416730923022, 0.01208943238934826], [0.005842011669205989, 0.026811092585102503, 0.0159995273103773, 0.011726486733972247], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.007224166173153271, 0.026811092585102503, 0.0159995273103773, 0.010417419149512282], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.009225263747613881, 0.021181994378879892, 0.01599186179820934, 0.008446858641255368], [0.009585709628670034, 0.02460223998371875, 0.02106887293983658, 0.01208943238934826], [0.009225263747613881, 0.021094107195895863, 0.01599186179820934, 0.008446858641255368], [0.011797729622127103, 0.0211521745302615, 0.01599186179820934, 0.008276021449724743], [0.026481548159307602, 0.03444089134738728, 0.027839491886923834, 0.017616142742732512], [0.006042856971184859, 0.0298250736354349, 0.026603913287688154, 0.016020188590790786], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.006041499892570327, 0.031776306981364855, 0.026941845894986747, 0.0160903922592919], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.026481548159307602, 0.03284106898246328, 0.02790945165747545, 0.017616142742732512], [0.009585709628670034, 0.02460223998371875, 0.02106887293983658, 0.01208943238934826], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.005842011669205989, 0.026811092585102503, 0.0159995273103773, 0.011726486733972247], [0.009225263747613881, 0.021181994378879892, 0.01599186179820934, 0.008446858641255368], [0.005959695280746272, 0.02881741611893251, 0.0159995273103773, 0.011726486733972247], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.005842011669205989, 0.026811092585102503, 0.0159995273103773, 0.011726486733972247], [0.005959695280746272, 0.028044688154969757, 0.0159995273103773, 0.011726486733972247], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.005842011669205989, 0.02460223998371875, 0.01676416730923022, 0.01208943238934826], [0.0077362150049944504, 0.02129066808527367, 0.01528204344843302, 0.008446858641255368], [0.009225263747613881, 0.021094107195895863, 0.01599186179820934, 0.008446858641255368], [0.011797729622127103, 0.0211521745302615, 0.01599186179820934, 0.008276021449724743], [0.026481548159307602, 0.03444089134738728, 0.027839491886923834, 0.017616142742732512], [0.026481548159307602, 0.03284106898246328, 0.02790945165747545, 0.017616142742732512], [0.006041499892570327, 0.031776306981364855, 0.026941845894986747, 0.0160903922592919], [0.006042856971184859, 0.0298250736354349, 0.026603913287688154, 0.016020188590790786], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.009585709628670034, 0.02460223998371875, 0.02106887293983658, 0.01208943238934826], [0.005842011669205989, 0.02460223998371875, 0.01676416730923022, 0.01208943238934826], [0.005842011669205989, 0.026811092585102503, 0.0159995273103773, 0.011726486733972247], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.009225263747613881, 0.021181994378879892, 0.01599186179820934, 0.008446858641255368], [0.005044551802509602, 0.027850409193116703, 0.01676416730923022, 0.01208943238934826], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.010575762190614384], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.005153243252895062, 0.030959132586452554, 0.024381264224689704, 0.011329731154], [0.005842011669205989, 0.027939248231505447, 0.0159995273103773, 0.011726486733972247], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.0077362150049944504, 0.02129066808527367, 0.01528204344843302, 0.008446858641255368], [0.026481548159307602, 0.03444089134738728, 0.027839491886923834, 0.017616142742732512], [0.006041499892570327, 0.031776306981364855, 0.026941845894986747, 0.0160903922592919], [0.009585709628670034, 0.02460223998371875, 0.02106887293983658, 0.01208943238934826], [0.026481548159307602, 0.03284106898246328, 0.02790945165747545, 0.017616142742732512], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.010575762190614384], [0.006042856971184859, 0.0298250736354349, 0.026603913287688154, 0.016020188590790786], [0.005044551802509602, 0.027850409193116703, 0.01676416730923022, 0.01208943238934826], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.005842011669205989, 0.02460223998371875, 0.01676416730923022, 0.01208943238934826], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.005944247398951958, 0.027939248231505447, 0.0159995273103773, 0.01003692920459686], [0.005044551802509602, 0.027850409193116703, 0.01676416730923022, 0.013168357227211196], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.009162488128506084, 0.021094107195895863, 0.016173504432798743, 0.007700190014575918], [0.026481548159307602, 0.03284106898246328, 0.02790945165747545, 0.017616142742732512], [0.005842011669205989, 0.026811092585102503, 0.016571892352702875, 0.011726486733972247], [0.005153243252895062, 0.030959132586452554, 0.024381264224689704, 0.011329731154], [0.0077362150049944504, 0.02129066808527367, 0.01528204344843302, 0.008446858641255368], [0.024274589710544987, 0.03444089134738728, 0.027839491886923834, 0.017616142742732512], [0.006041499892570327, 0.031776306981364855, 0.026941845894986747, 0.0160903922592919], [0.006041499892570327, 0.031776306981364855, 0.026941845894986747, 0.0160903922592919], [0.009585709628670034, 0.02460223998371875, 0.02106887293983658, 0.01208943238934826], [0.006042856971184859, 0.0298250736354349, 0.026603913287688154, 0.016020188590790786], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.005044551802509602, 0.027850409193116703, 0.01676416730923022, 0.013168357227211196], [0.006042856971184859, 0.0298250736354349, 0.026603913287688154, 0.016020188590790786], [0.009162488128506084, 0.021094107195895863, 0.016173504432798743, 0.007700190014575918], [0.0077362150049944504, 0.02796653358692045, 0.016005367919048316, 0.008446858641255368], [0.005153243252895062, 0.030959132586452554, 0.0219580925335683, 0.010575762190614384], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.010575762190614384], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.005153243252895062, 0.030959132586452554, 0.024381264224689704, 0.011329731154], [0.005944247398951958, 0.02611703794414079, 0.0159995273103773, 0.01003692920459686], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.007700190014575918], [0.0077362150049944504, 0.02129066808527367, 0.01528204344843302, 0.008446858641255368], [0.024274589710544987, 0.03444089134738728, 0.027839491886923834, 0.017616142742732512], [0.024274589710544987, 0.03444089134738728, 0.0283905979226595, 0.017616142742732512], [0.007704070929705906, 0.031961255684330316, 0.026941845894986747, 0.0160903922592919], [0.006041499892570327, 0.031776306981364855, 0.026941845894986747, 0.0160903922592919], [0.006041499892570327, 0.031776306981364855, 0.026941845894986747, 0.0160903922592919], [0.005153243252895062, 0.030959132586452554, 0.0219580925335683, 0.010575762190614384], [0.009162488128506084, 0.021094107195895863, 0.016173504432798743, 0.007700190014575918], [0.005044551802509602, 0.027850409193116703, 0.01676416730923022, 0.013168357227211196], [0.024274589710544987, 0.03444089134738728, 0.027839491886923834, 0.017616142742732512], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.005044551802509602, 0.030829715465105302, 0.017215197898440872, 0.012132558537461036], [0.005944247398951958, 0.02611703794414079, 0.0159995273103773, 0.01003692920459686], [0.006042856971184859, 0.0316034624219892, 0.026607750554734088, 0.0160903922592919], [0.006042856971184859, 0.0298250736354349, 0.026603913287688154, 0.016020188590790786], [0.006042856971184859, 0.0298250736354349, 0.026603913287688154, 0.016020188590790786], [0.0077362150049944504, 0.02796653358692045, 0.016005367919048316, 0.008446858641255368], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.006042856971184859, 0.0298250736354349, 0.027379787014627722, 0.016020188590790786], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.007700190014575918], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.007700190014575918], [0.0077362150049944504, 0.02129066808527367, 0.01528204344843302, 0.008446858641255368], [0.024274589710544987, 0.03444089134738728, 0.027839491886923834, 0.017616142742732512], [0.024274589710544987, 0.03444089134738728, 0.0283905979226595, 0.017616142742732512], [0.024274589710544987, 0.03444089134738728, 0.027647988619046306, 0.017616142742732512], [0.009162488128506084, 0.031651269210197744, 0.02763465260703337, 0.007697683219821857], [0.009162488128506084, 0.021094107195895863, 0.016173504432798743, 0.007700190014575918], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.005044551802509602, 0.030829715465105302, 0.017215197898440872, 0.012132558537461036], [0.005944247398951958, 0.02611703794414079, 0.0159995273103773, 0.01003692920459686], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.024274589710544987, 0.03444089134738728, 0.027839491886923834, 0.017616142742732512], [0.007761616669596546, 0.02611703794414079, 0.021623706598455215, 0.01003692920459686], [0.00901441275990559, 0.021094107195895863, 0.016173504432798743, 0.007700190014575918], [0.005153243252895062, 0.030959132586452554, 0.0219580925335683, 0.010575762190614384], [0.024274589710544987, 0.03444089134738728, 0.027839491886923834, 0.017616142742732512], [0.009162488128506084, 0.021146744931969323, 0.014251197310590184, 0.008445186006598313], [0.0077362150049944504, 0.02796653358692045, 0.016005367919048316, 0.008446858641255368], [0.005044551802509602, 0.030829715465105302, 0.017215197898440872, 0.012132558537461036], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.005014754400817891], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.007700190014575918], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.007700190014575918], [0.0077362150049944504, 0.02129066808527367, 0.01528204344843302, 0.008446858641255368], [0.024274589710544987, 0.03444089134738728, 0.027839491886923834, 0.017616142742732512], [0.024274589710544987, 0.03444089134738728, 0.0283905979226595, 0.017616142742732512], [0.024274589710544987, 0.03444089134738728, 0.027647988619046306, 0.017616142742732512], [0.009162488128506084, 0.031651269210197744, 0.02763465260703337, 0.007697683219821857], [0.0050271814774283715, 0.030829715465105302, 0.017215197898440872, 0.011836703743713239], [0.009162488128506084, 0.021094107195895863, 0.016173504432798743, 0.007700190014575918], [0.009162488128506084, 0.031651269210197744, 0.02763465260703337, 0.007697683219821857], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.007761616669596546, 0.02611703794414079, 0.021623706598455215, 0.01003692920459686], [0.005331063869203465, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.009162488128506084, 0.031651269210197744, 0.025574709378920212, 0.007697683219821857], [0.005044551802509602, 0.030829715465105302, 0.017215197898440872, 0.012132558537461036], [0.024274589710544987, 0.03444089134738728, 0.027647988619046306, 0.017616142742732512], [0.005944247398951958, 0.02611703794414079, 0.01600323203319273, 0.009847664824011896], [0.00901441275990559, 0.021094107195895863, 0.016173504432798743, 0.007700190014575918], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.005014754400817891], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.007700190014575918], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.007700190014575918], [0.0077362150049944504, 0.02129066808527367, 0.01528204344843302, 0.008446858641255368], [0.024274589710544987, 0.03439325659667182, 0.0283905979226595, 0.016854976878393074], [0.009162488128506084, 0.031651269210197744, 0.02763465260703337, 0.007697683219821857], [0.0216815238284495, 0.03444089134738728, 0.027647988619046306, 0.017616142742732512], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.009162488128506084, 0.031651269210197744, 0.025574709378920212, 0.007697683219821857], [0.009162488128506084, 0.021094107195895863, 0.016173504432798743, 0.007700190014575918], [0.005944247398951958, 0.02611703794414079, 0.01600323203319273, 0.009847664824011896], [0.009162488128506084, 0.031651269210197744, 0.02763465260703337, 0.007697683219821857], [0.005331063869203465, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.005331063869203465, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.007680610755318182, 0.02611703794414079, 0.02107582935193922, 0.01003692920459686], [0.007761616669596546, 0.02611703794414079, 0.021623706598455215, 0.01003692920459686], [0.00901441275990559, 0.021094107195895863, 0.016173504432798743, 0.007700190014575918], [0.009162488128506084, 0.032104110034916346, 0.02763465260703337, 0.007697683219821857], [0.0051562378592429885, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.005014754400817891], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.005014754400817891], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.007700190014575918], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.007700190014575918], [0.0077362150049944504, 0.02129066808527367, 0.01528204344843302, 0.008446858641255368], [0.024274589710544987, 0.03439325659667182, 0.0283905979226595, 0.016854976878393074], [0.024274589710544987, 0.03439325659667182, 0.028613621804920435, 0.017007422476574934], [0.0216815238284495, 0.03444089134738728, 0.027647988619046306, 0.017616142742732512], [0.009162488128506084, 0.031651269210197744, 0.02763465260703337, 0.007697683219821857], [0.0216815238284495, 0.03444089134738728, 0.027647988619046306, 0.017616142742732512], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.009162488128506084, 0.032104110034916346, 0.02763465260703337, 0.007697683219821857], [0.009162488128506084, 0.031651269210197744, 0.025574709378920212, 0.007697683219821857], [0.005049063891441333, 0.02611703794414079, 0.01617586201228082, 0.009847664824011896], [0.0051562378592429885, 0.031629173050340395, 0.02228628261051206, 0.011165891520219105], [0.009162488128506084, 0.031651269210197744, 0.02763465260703337, 0.007697683219821857], [0.007761616669596546, 0.026929406573678406, 0.021238684800099463, 0.01003692920459686], [0.007761616669596546, 0.02611703794414079, 0.021623706598455215, 0.01003692920459686], [0.007685603842730202, 0.02129066808527367, 0.018179193865885863, 0.008446858641255368], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.005004543899614723], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.007700190014575918], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.007700190014575918], [0.0077362150049944504, 0.02129066808527367, 0.01528204344843302, 0.008446858641255368], [0.024274589710544987, 0.03439325659667182, 0.0283905979226595, 0.016854976878393074], [0.024274589710544987, 0.03439325659667182, 0.028613621804920435, 0.017007422476574934], [0.024274589710544987, 0.03439325659667182, 0.030878835412620813, 0.01634715205649849], [0.0216815238284495, 0.03444089134738728, 0.027647988619046306, 0.017616142742732512], [0.009162488128506084, 0.032104110034916346, 0.02763465260703337, 0.007697683219821857], [0.0216815238284495, 0.03444089134738728, 0.027647988619046306, 0.017616142742732512], [0.007761616669596546, 0.026929406573678406, 0.021238684800099463, 0.01003692920459686], [0.009162488128506084, 0.031651269210197744, 0.02763465260703337, 0.007697683219821857], [0.009162488128506084, 0.031651269210197744, 0.02763465260703337, 0.007697683219821857], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.009162488128506084, 0.032104110034916346, 0.02763465260703337, 0.007697683219821857], [0.009162488128506084, 0.031651269210197744, 0.025574709378920212, 0.007697683219821857], [0.0216815238284495, 0.03444089134738728, 0.027647988619046306, 0.017616142742732512], [0.005153243252895062, 0.030959132586452554, 0.022623403173694715, 0.011329731154], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.0059056830672078765, 0.024310629569745664, 0.01892924004715692, 0.01208943238934826], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.005004543899614723], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.007700190014575918], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.007700190014575918], [0.0077362150049944504, 0.02129066808527367, 0.01528204344843302, 0.008446858641255368], [0.0216815238284495, 0.03444089134738728, 0.030606856544257755, 0.017616142742732512], [0.020306531598219844, 0.032104110034916346, 0.027647883065869766, 0.00828903312439872], [0.009162488128506084, 0.031651269210197744, 0.02763465260703337, 0.007697683219821857], [0.009162488128506084, 0.032104110034916346, 0.02763465260703337, 0.007645184296228064], [0.0216815238284495, 0.03444089134738728, 0.027647988619046306, 0.017616142742732512], [0.009162488128506084, 0.031651269210197744, 0.025574709378920212, 0.007697683219821857], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.007761616669596546, 0.026929406573678406, 0.021238684800099463, 0.01003692920459686], [0.0216815238284495, 0.03444089134738728, 0.027647988619046306, 0.017616142742732512], [0.005153243252895062, 0.030959132586452554, 0.022623403173694715, 0.011329731154], [0.0216815238284495, 0.03444089134738728, 0.027647988619046306, 0.017616142742732512], [0.007761616669596546, 0.02749665592427898, 0.020512649176257846, 0.010057920019009552], [0.009162488128506084, 0.031651269210197744, 0.02763465260703337, 0.007697683219821857], [0.005153243252895062, 0.030959132586452554, 0.022623403173694715, 0.011329731154], [0.0216815238284495, 0.03444089134738728, 0.027647988619046306, 0.017616142742732512], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.005004543899614723], [0.0077362150049944504, 0.02129066808527367, 0.01528204344843302, 0.008446858641255368], [0.0216815238284495, 0.03444089134738728, 0.030606856544257755, 0.017616142742732512], [0.020306531598219844, 0.032104110034916346, 0.027647883065869766, 0.00828903312439872], [0.009162488128506084, 0.03219254123215279, 0.02763465260703337, 0.008079402120233711], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.009162488128506084, 0.031651269210197744, 0.025574709378920212, 0.007697683219821857], [0.009162488128506084, 0.031651269210197744, 0.02763465260703337, 0.007697683219821857], [0.009162488128506084, 0.032104110034916346, 0.02763465260703337, 0.007645184296228064], [0.005326703275842472, 0.030959132586452554, 0.022623403173694715, 0.011329731154], [0.0216815238284495, 0.034412110707668746, 0.027647988619046306, 0.01674544545737601], [0.005153243252895062, 0.030959132586452554, 0.022623403173694715, 0.011329731154], [0.009162488128506084, 0.031651269210197744, 0.02763465260703337, 0.007697683219821857], [0.0077362150049944504, 0.02124471109016991, 0.015575247076898488, 0.008708419259611515], [0.0059056830672078765, 0.02466452693177953, 0.016183973104048045, 0.01208943238934826], [0.005153243252895062, 0.030959132586452554, 0.022623403173694715, 0.011329731154], [0.007761616669596546, 0.02749665592427898, 0.020512649176257846, 0.010057920019009552], [0.007761616669596546, 0.02749665592427898, 0.020512649176257846, 0.010057920019009552], [0.009162488128506084, 0.032104110034916346, 0.02763465260703337, 0.007645184296228064], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.005004543899614723], [0.0077362150049944504, 0.02129066808527367, 0.01528204344843302, 0.008446858641255368], [0.0216815238284495, 0.03444089134738728, 0.030606856544257755, 0.017616142742732512], [0.0216815238284495, 0.03444089134738728, 0.030606856544257755, 0.017616142742732512], [0.007738952881498976, 0.030959132586452554, 0.022623403173694715, 0.011329731154], [0.0216815238284495, 0.034412110707668746, 0.027647988619046306, 0.01674544545737601], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.009162488128506084, 0.031651269210197744, 0.025449082517335304, 0.007697683219821857], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.005326703275842472, 0.030959132586452554, 0.022623403173694715, 0.011329731154], [0.009162488128506084, 0.031651269210197744, 0.025574709378920212, 0.007697683219821857], [0.006937139804091146, 0.030278126109319698, 0.020512649176257846, 0.010075263779498474], [0.005326703275842472, 0.030959132586452554, 0.022623403173694715, 0.011329731154], [0.0216815238284495, 0.034412110707668746, 0.027647988619046306, 0.01674544545737601], [0.0077362150049944504, 0.02124471109016991, 0.015575247076898488, 0.008708419259611515], [0.007761616669596546, 0.02749665592427898, 0.021041058319305514, 0.008476001856686972], [0.009162488128506084, 0.031651269210197744, 0.025574709378920212, 0.007697683219821857], [0.0059056830672078765, 0.02451792920287719, 0.016183973104048045, 0.01208943238934826], [0.009162488128506084, 0.031651269210197744, 0.02763465260703337, 0.006440648740315814], [0.005153243252895062, 0.030959132586452554, 0.022623403173694715, 0.011329731154], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.005004543899614723], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.005004543899614723], [0.0077362150049944504, 0.02129066808527367, 0.01528204344843302, 0.008446858641255368], [0.0216815238284495, 0.03444089134738728, 0.030606856544257755, 0.017616142742732512], [0.0216815238284495, 0.03444089134738728, 0.030606856544257755, 0.017616142742732512], [0.0216815238284495, 0.03444089134738728, 0.030606856544257755, 0.017616142742732512], [0.009162488128506084, 0.031651269210197744, 0.02763465260703337, 0.006440648740315814], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.007738952881498976, 0.030959132586452554, 0.022623403173694715, 0.011329731154], [0.007761616669596546, 0.02749665592427898, 0.021041058319305514, 0.008476001856686972], [0.005153243252895062, 0.030959132586452554, 0.02228628261051206, 0.011329731154], [0.006350241627137465, 0.02749665592427898, 0.021041058319305514, 0.008476001856686972], [0.0077362150049944504, 0.02124471109016991, 0.015575247076898488, 0.008708419259611515], [0.009162488128506084, 0.031651269210197744, 0.0254344300051963, 0.007697823490991891], [0.009162488128506084, 0.031651269210197744, 0.025449082517335304, 0.007697683219821857], [0.005326703275842472, 0.030959132586452554, 0.022790418540034153, 0.011329731154], [0.005326703275842472, 0.030959132586452554, 0.022623403173694715, 0.011329731154], [0.009162488128506084, 0.031651269210197744, 0.025574709378920212, 0.007697683219821857], [0.005326703275842472, 0.030959132586452554, 0.022623403173694715, 0.011329731154], [0.0077362150049944504, 0.024453741906331303, 0.01528204344843302, 0.008397067288606539], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.005004543899614723], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.005004543899614723], [0.009162488128506084, 0.021146744931969323, 0.014268967953760375, 0.005004543899614723], [0.0077362150049944504, 0.02129066808527367, 0.01528204344843302, 0.008446858641255368], [0.0216815238284495, 0.03444089134738728, 0.030606856544257755, 0.017616142742732512], [0.0216815238284495, 0.03444089134738728, 0.030606856544257755, 0.017616142742732512], [0.0216815238284495, 0.03444089134738728, 0.030606856544257755, 0.017616142742732512], [0.0216815238284495, 0.03444089134738728, 0.030606856544257755, 0.017616142742732512], [0.009162488128506084, 0.031651269210197744, 0.03026228190368752, 0.006440648740315814], [0.005326703275842472, 0.030959132586452554, 0.022790418540034153, 0.011329731154], [0.007738952881498976, 0.030959132586452554, 0.022623403173694715, 0.011329731154], [0.009162488128506084, 0.031651269210197744, 0.02763465260703337, 0.006440648740315814], [0.006350241627137465, 0.02749665592427898, 0.021041058319305514, 0.008476001856686972], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.009162488128506084, 0.031651269210197744, 0.0254344300051963, 0.007697823490991891], [0.0077362150049944504, 0.02124471109016991, 0.01602460889332924, 0.008708419259611515], [0.009162488128506084, 0.031651269210197744, 0.025449082517335304, 0.007697683219821857], [0.0077362150049944504, 0.02124471109016991, 0.015575247076898488, 0.008708419259611515], [0.007738952881498976, 0.030959132586452554, 0.022623403173694715, 0.011329731154], [0.0077362150049944504, 0.024453741906331303, 0.01528204344843302, 0.008397067288606539], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.0216815238284495, 0.03390508912065952, 0.030606856544257755, 0.017616142742732512], [0.006350241627137465, 0.028953504042049407, 0.021041058319305514, 0.008476001856686972], [0.0077362150049944504, 0.024453741906331303, 0.01528204344843302, 0.008397067288606539], [0.005326703275842472, 0.030959132586452554, 0.022790418540034153, 0.011329731154], [0.006350241627137465, 0.02749665592427898, 0.021041058319305514, 0.008476001856686972], [0.009162488128506084, 0.031651269210197744, 0.0254344300051963, 0.007697823490991891], [0.0216815238284495, 0.03444089134738728, 0.030606856544257755, 0.017213861069641272], [0.006350241627137465, 0.02749665592427898, 0.021041058319305514, 0.008476001856686972], [0.0077362150049944504, 0.024453741906331303, 0.01528204344843302, 0.008398974659829384], [0.0077362150049944504, 0.02124471109016991, 0.015575247076898488, 0.008708419259611515], [0.0077362150049944504, 0.02129066808527367, 0.01528204344843302, 0.008446858641255368], [0.007477197731746519, 0.021457555245634807, 0.01602460889332924, 0.008708419259611515], [0.009940787124213162, 0.03106228790475094, 0.0254344300051963, 0.005138327888625149], [0.009162488128506084, 0.031651269210197744, 0.0254344300051963, 0.007697823490991891], [0.009162488128506084, 0.031651269210197744, 0.025449082517335304, 0.007697683219821857], [0.009162488128506084, 0.031651269210197744, 0.025449082517335304, 0.007697683219821857], [0.009162488128506084, 0.031651269210197744, 0.02763465260703337, 0.006416493946015444], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0216815238284495, 0.031870026648633074, 0.030606856544257755, 0.017616142742732512], [0.008670806814652234, 0.031651269210197744, 0.027782715723674183, 0.006416493946015444], [0.005326703275842472, 0.030959132586452554, 0.022790418540034153, 0.011329731154], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.0077362150049944504, 0.024453741906331303, 0.01528204344843302, 0.008397067288606539], [0.009162488128506084, 0.031651269210197744, 0.025449082517335304, 0.007697683219821857], [0.0066268674329929955, 0.023545153340764586, 0.01528204344843302, 0.008398974659829384], [0.006350241627137465, 0.02749665592427898, 0.021041058319305514, 0.008476001856686972], [0.0216815238284495, 0.03390508912065952, 0.030606856544257755, 0.017616142742732512], [0.0216815238284495, 0.03390508912065952, 0.030606856544257755, 0.017616142742732512], [0.006350241627137465, 0.02749665592427898, 0.021041058319305514, 0.008476001856686972], [0.007477197731746519, 0.021457555245634807, 0.01602460889332924, 0.008708419259611515], [0.009162488128506084, 0.031651269210197744, 0.025449082517335304, 0.007697683219821857], [0.009162488128506084, 0.031651269210197744, 0.025449082517335304, 0.007697683219821857], [0.009162488128506084, 0.031651269210197744, 0.025449082517335304, 0.005610334614981503], [0.009940787124213162, 0.03106228790475094, 0.0254344300051963, 0.005138327888625149], [0.0077362150049944504, 0.024453741906331303, 0.01528204344843302, 0.008398974659829384], [0.007477197731746519, 0.021457555245634807, 0.01602460889332924, 0.008708419259611515], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.009543135421931708], [0.021755367720050694, 0.031230771866997804, 0.02975480848381225, 0.01783911155675748], [0.0216815238284495, 0.031870026648633074, 0.030606856544257755, 0.017616142742732512], [0.008670806814652234, 0.031651269210197744, 0.027782715723674183, 0.006416493946015444], [0.009940787124213162, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.006350241627137465, 0.02749665592427898, 0.021041058319305514, 0.008476001856686972], [0.0077362150049944504, 0.024453741906331303, 0.01528204344843302, 0.008397067288606539], [0.009162488128506084, 0.031651269210197744, 0.025449082517335304, 0.007697683219821857], [0.007477197731746519, 0.021457555245634807, 0.01602460889332924, 0.008708419259611515], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.0077362150049944504, 0.024453741906331303, 0.01528204344843302, 0.008398974659829384], [0.006350241627137465, 0.02749665592427898, 0.021041058319305514, 0.008476001856686972], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.012289901614841163], [0.0077362150049944504, 0.024453741906331303, 0.01543576298067926, 0.008396175518520645], [0.006350241627137465, 0.02749665592427898, 0.021041058319305514, 0.008476001856686972], [0.0216815238284495, 0.03390508912065952, 0.030606856544257755, 0.017616142742732512], [0.0216815238284495, 0.031870026648633074, 0.030606856544257755, 0.017616142742732512], [0.007477197731746519, 0.021457555245634807, 0.01602460889332924, 0.008708419259611515], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.009543135421931708], [0.021755367720050694, 0.031230771866997804, 0.02975480848381225, 0.01783911155675748], [0.006350241627137465, 0.02749665592427898, 0.022393437619126075, 0.00928948473972783], [0.008670806814652234, 0.031651269210197744, 0.027782715723674183, 0.006416493946015444], [0.0059056830672078765, 0.024714329790255996, 0.01676416730923022, 0.012289901614841163], [0.0075321237856212095, 0.024453741906331303, 0.01528204344843302, 0.006447981470947457], [0.009940787124213162, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.0077362150049944504, 0.024453741906331303, 0.01528204344843302, 0.008397067288606539], [0.006350241627137465, 0.02749665592427898, 0.021041058319305514, 0.008476001856686972], [0.0077362150049944504, 0.03146852368072103, 0.01528204344843302, 0.008397067288606539], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.01208943238934826], [0.009162488128506084, 0.031651269210197744, 0.025449082517335304, 0.007697683219821857], [0.0077362150049944504, 0.024453741906331303, 0.01528204344843302, 0.008398974659829384], [0.006350241627137465, 0.02749665592427898, 0.021041058319305514, 0.008476001856686972], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.0077362150049944504, 0.024453741906331303, 0.01528204344843302, 0.008397067288606539], [0.0059056830672078765, 0.02466452693177953, 0.01676416730923022, 0.012289901614841163], [0.0216815238284495, 0.031870026648633074, 0.030606856544257755, 0.016695286507122604], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0216815238284495, 0.031870026648633074, 0.030606856544257755, 0.016695286507122604], [0.005872580963578069, 0.024714329790255996, 0.017160945517279782, 0.011560420479070977], [0.006350241627137465, 0.02749665592427898, 0.022393437619126075, 0.00928948473972783], [0.009940787124213162, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.0077362150049944504, 0.024453741906331303, 0.01528204344843302, 0.008398974659829384], [0.006350241627137465, 0.02749665592427898, 0.021041058319305514, 0.008476001856686972], [0.0077362150049944504, 0.024453741906331303, 0.01528204344843302, 0.008397067288606539], [0.006350241627137465, 0.02749665592427898, 0.021041058319305514, 0.008476001856686972], [0.008690908419444617, 0.031651269210197744, 0.027783915027064624, 0.006161115448979039], [0.0077362150049944504, 0.024453741906331303, 0.01528204344843302, 0.008397067288606539], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.0075321237856212095, 0.024453741906331303, 0.01528204344843302, 0.006447981470947457], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.008670806814652234, 0.031634879908522776, 0.028689135640684954, 0.006416829191526845], [0.008516756637463783, 0.024453741906331303, 0.01528204344843302, 0.008399029795924868], [0.006350241627137465, 0.02749665592427898, 0.021041058319305514, 0.008476001856686972], [0.009940787124213162, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.0059056830672078765, 0.02466452693177953, 0.01795544854529066, 0.01208943238934826], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0216815238284495, 0.031870026648633074, 0.029802797527219087, 0.016695286507122604], [0.008670806814652234, 0.031634879908522776, 0.028689135640684954, 0.006416829191526845], [0.0216815238284495, 0.031870026648633074, 0.030606856544257755, 0.016695286507122604], [0.005872580963578069, 0.024714329790255996, 0.017160945517279782, 0.011560420479070977], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.006350241627137465, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.006350241627137465, 0.02749665592427898, 0.022393437619126075, 0.00928948473972783], [0.008516756637463783, 0.024453741906331303, 0.01528204344843302, 0.008399029795924868], [0.0077362150049944504, 0.024453741906331303, 0.01528204344843302, 0.008398974659829384], [0.009940787124213162, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.0077362150049944504, 0.024453741906331303, 0.01528204344843302, 0.008397067288606539], [0.008690908419444617, 0.031651269210197744, 0.027783915027064624, 0.006161115448979039], [0.0077362150049944504, 0.024453741906331303, 0.01528204344843302, 0.008397067288606539], [0.0059056830672078765, 0.02466452693177953, 0.01795544854529066, 0.01208943238934826], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.009940787124213162, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.006853056174775034], [0.009940787124213162, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.008670806814652234, 0.031634879908522776, 0.028689135640684954, 0.006416829191526845], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.006350241627137465, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.0059056830672078765, 0.025639298902611324, 0.01795544854529066, 0.01208943238934826], [0.005872580963578069, 0.024714329790255996, 0.017160945517279782, 0.011560420479070977], [0.008516756637463783, 0.024453741906331303, 0.01528204344843302, 0.008399029795924868], [0.0077362150049944504, 0.024453741906331303, 0.01528204344843302, 0.008398974659829384], [0.005872580963578069, 0.024714329790255996, 0.017160945517279782, 0.011560420479070977], [0.009940787124213162, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.009940787124213162, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.0077362150049944504, 0.024453741906331303, 0.01528204344843302, 0.008397067288606539], [0.008690908419444617, 0.031651269210197744, 0.027783915027064624, 0.006161115448979039], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.009940787124213162, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.006350241627137465, 0.02749665592427898, 0.022703946984749365, 0.008093620292395057], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.0055562060053424055], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.005565959977632532], [0.00625774909227207, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.008670806814652234, 0.031634879908522776, 0.028689135640684954, 0.006416829191526845], [0.005872580963578069, 0.024714329790255996, 0.017160945517279782, 0.011560420479070977], [0.006350241627137465, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.007325133897488294, 0.02749665592427898, 0.023727358011224723, 0.00907970461962664], [0.009386208216946421, 0.03105174489615866, 0.023024224347414544, 0.005138327888625149], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.0077362150049944504, 0.024453741906331303, 0.01528204344843302, 0.008398974659829384], [0.008516756637463783, 0.024453741906331303, 0.01528204344843302, 0.008399029795924868], [0.008690908419444617, 0.031651269210197744, 0.027783915027064624, 0.006161115448979039], [0.009940787124213162, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.005872580963578069, 0.024714329790255996, 0.017160945517279782, 0.011560420479070977], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.009256184439506397, 0.031651269210197744, 0.027783915027064624, 0.006161115448979039], [0.009940787124213162, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.00625774909227207, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.0077362150049944504, 0.024424159035996998, 0.01528204344843302, 0.009270994706696442], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.0055562060053424055], [0.0077362150049944504, 0.02539532494366452, 0.015480910528047124, 0.008397067288606539], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.005872580963578069, 0.024714329790255996, 0.017160945517279782, 0.011560420479070977], [0.008670806814652234, 0.031634879908522776, 0.028689135640684954, 0.006416829191526845], [0.0077362150049944504, 0.030387303891569756, 0.01528204344843302, 0.009270994706696442], [0.008690908419444617, 0.031651269210197744, 0.027783915027064624, 0.006161115448979039], [0.007325133897488294, 0.02749665592427898, 0.023727358011224723, 0.00907970461962664], [0.008661616053446248, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.006350241627137465, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.008516756637463783, 0.024453741906331303, 0.01528204344843302, 0.008399029795924868], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.009386208216946421, 0.03105174489615866, 0.023024224347414544, 0.005138327888625149], [0.009940787124213162, 0.02987105241158176, 0.02329666415148, 0.0051789814502383276], [0.009256184439506397, 0.031651269210197744, 0.027783915027064624, 0.006161115448979039], [0.006824201540631962, 0.02749665592427898, 0.023727358011224723, 0.00907970461962664], [0.008516756637463783, 0.024453741906331303, 0.01528204344843302, 0.008348979947811756], [0.008670806814652234, 0.031634879908522776, 0.029928559247206454, 0.006416829191526845], [0.005872580963578069, 0.024714329790255996, 0.017160945517279782, 0.011560420479070977], [0.009940787124213162, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.0055562060053424055], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0077362150049944504, 0.030387303891569756, 0.01528204344843302, 0.009270994706696442], [0.005872580963578069, 0.024714329790255996, 0.017160945517279782, 0.011560420479070977], [0.008670806814652234, 0.031634879908522776, 0.028689135640684954, 0.006416829191526845], [0.007325133897488294, 0.02749665592427898, 0.023727358011224723, 0.00907970461962664], [0.008690908419444617, 0.031651269210197744, 0.027783915027064624, 0.006161115448979039], [0.008661616053446248, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.006350241627137465, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.009940787124213162, 0.02987105241158176, 0.02329666415148, 0.0051789814502383276], [0.008516756637463783, 0.024453741906331303, 0.01528204344843302, 0.008399029795924868], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.006824201540631962, 0.02749665592427898, 0.023727358011224723, 0.00907970461962664], [0.008516756637463783, 0.024453741906331303, 0.01528204344843302, 0.008348979947811756], [0.009256184439506397, 0.031651269210197744, 0.027783915027064624, 0.006161115448979039], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.0055562060053424055], [0.005872580963578069, 0.024714329790255996, 0.017160945517279782, 0.011560420479070977], [0.008653959254797432, 0.020420613489075014, 0.014268967953760375, 0.005375981499090798], [0.008670806814652234, 0.031634879908522776, 0.029928559247206454, 0.006416829191526845], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.008670806814652234, 0.031634879908522776, 0.028689135640684954, 0.006416829191526845], [0.005872580963578069, 0.024714329790255996, 0.017160945517279782, 0.011560420479070977], [0.007325133897488294, 0.02749665592427898, 0.023727358011224723, 0.00907970461962664], [0.008661616053446248, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.008653959254797432, 0.020420613489075014, 0.014268967953760375, 0.005375981499090798], [0.006350241627137465, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.0077362150049944504, 0.030387303891569756, 0.01528204344843302, 0.009270994706696442], [0.006824201540631962, 0.02749665592427898, 0.023727358011224723, 0.00907970461962664], [0.008670806814652234, 0.031634879908522776, 0.029928559247206454, 0.006416829191526845], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.009563853447362455, 0.031634879908522776, 0.029928559247206454, 0.006416829191526845], [0.008516756637463783, 0.024453741906331303, 0.01528204344843302, 0.008348979947811756], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.0055562060053424055], [0.008670806814652234, 0.031634879908522776, 0.029928559247206454, 0.006416829191526845], [0.0084921306625718, 0.024453741906331303, 0.01528204344843302, 0.008348979947811756], [0.008653959254797432, 0.021146744931969323, 0.014259361652797653, 0.005375981499090798], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.008670806814652234, 0.031634879908522776, 0.028689135640684954, 0.006416829191526845], [0.007325133897488294, 0.02749665592427898, 0.023727358011224723, 0.00907970461962664], [0.008653959254797432, 0.020420613489075014, 0.014268967953760375, 0.005375981499090798], [0.005872580963578069, 0.024714329790255996, 0.017160945517279782, 0.011560420479070977], [0.008661616053446248, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.0077362150049944504, 0.030387303891569756, 0.01528204344843302, 0.009270994706696442], [0.006350241627137465, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.006824201540631962, 0.02749665592427898, 0.023727358011224723, 0.00907970461962664], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.008670806814652234, 0.031634879908522776, 0.028689135640684954, 0.006416829191526845], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.007046154121780212], [0.008661616053446248, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.0055562060053424055], [0.006824201540631962, 0.0275217278276228, 0.023727358011224723, 0.00907970461962664], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.007325133897488294, 0.02749665592427898, 0.023727358011224723, 0.00907970461962664], [0.008670806814652234, 0.031634879908522776, 0.028689135640684954, 0.006416829191526845], [0.008661616053446248, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.006824201540631962, 0.02749665592427898, 0.023727358011224723, 0.00907970461962664], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.0069053330960330605, 0.021068382410792886, 0.012756016158662821, 0.005625804106325206], [0.008661616053446248, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.006345400626098766, 0.020352872934501987, 0.02295717423538388, 0.005375451497642175], [0.008670806814652234, 0.031634879908522776, 0.028689135640684954, 0.006416829191526845], [0.006656005835223213, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.0070215660404767885, 0.02749665592427898, 0.023727358011224723, 0.00907970461962664], [0.008670806814652234, 0.031634879908522776, 0.028689135640684954, 0.006416829191526845], [0.006824201540631962, 0.0275217278276228, 0.023727358011224723, 0.00907970461962664], [0.006350241627137465, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.006345400626098766, 0.021068382410792886, 0.01016476334865582, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006345400626098766, 0.020352872934501987, 0.02295717423538388, 0.005375451497642175], [0.008670806814652234, 0.031634879908522776, 0.028689135640684954, 0.006416829191526845], [0.006824201540631962, 0.02749665592427898, 0.023727358011224723, 0.00907970461962664], [0.006656005835223213, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.008661616053446248, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.0069053330960330605, 0.021068382410792886, 0.012756016158662821, 0.005625804106325206], [0.008661616053446248, 0.02738830493105654, 0.023727358011224723, 0.008093620292395057], [0.006325003396419089, 0.020676170077314412, 0.023577210513746612, 0.005375451497642175], [0.008661616053446248, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.0070215660404767885, 0.02749665592427898, 0.023727358011224723, 0.00907970461962664], [0.008661182439682667, 0.031591335862574434, 0.023727358011224723, 0.00907970461962664], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005134401672508565], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006255528533129185, 0.021068382410792886, 0.01016476334865582, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.01016476334865582, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.008410969798420584, 0.021095430400269795, 0.024203118903959535, 0.005375451497642175], [0.006656005835223213, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.008661616053446248, 0.02738830493105654, 0.023727358011224723, 0.008093620292395057], [0.006159488205254294, 0.021352754264249693, 0.0229409959924416, 0.005375451497642175], [0.007023264488938916, 0.02749665592427898, 0.023727358011224723, 0.00907970461962664], [0.008670806814652234, 0.031634879908522776, 0.028689135640684954, 0.006416829191526845], [0.008661616053446248, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.0069053330960330605, 0.021068382410792886, 0.012756016158662821, 0.005625804106325206], [0.006345400626098766, 0.020352872934501987, 0.02295717423538388, 0.005375451497642175], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005134401672508565], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005138327888625149], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005134401672508565], [0.008661616053446248, 0.02749665592427898, 0.023727358011224723, 0.008093620292395057], [0.006255528533129185, 0.021068382410792886, 0.01016476334865582, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.01016476334865582, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.008653959254797432, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.008410969798420584, 0.021095430400269795, 0.024203118903959535, 0.005375451497642175], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005134401672508565], [0.0069053330960330605, 0.021068382410792886, 0.012756016158662821, 0.005625804106325206], [0.006340670168503612, 0.021352754264249693, 0.0229409959924416, 0.005375451497642175], [0.008405847765845766, 0.027002283622378363, 0.02415852812891787, 0.005375451497642175], [0.006656005835223213, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.006159488205254294, 0.021352754264249693, 0.0229409959924416, 0.005375451497642175], [0.008670806814652234, 0.031634879908522776, 0.028689135640684954, 0.006416829191526845], [0.009386208216946421, 0.03210565728919472, 0.02329666415148, 0.005617867921690691], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005134401672508565], [0.006345400626098766, 0.020352872934501987, 0.02295717423538388, 0.005375451497642175], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005134401672508565], [0.006345400626098766, 0.021068382410792886, 0.01016476334865582, 0.005375451497642175], [0.006255528533129185, 0.021068382410792886, 0.01016476334865582, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.01016476334865582, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.008410969798420584, 0.021095430400269795, 0.024203118903959535, 0.005375451497642175], [0.0074664156863967985, 0.03106228790475094, 0.02329666415148, 0.005134401672508565], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005134401672508565], [0.0069053330960330605, 0.021068382410792886, 0.012756016158662821, 0.005625804106325206], [0.007156523001116461, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.006645560369104161, 0.021454071121940723, 0.0229409959924416, 0.005375940434306164], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005134401672508565], [0.006159488205254294, 0.021352754264249693, 0.0229409959924416, 0.005375451497642175], [0.006340670168503612, 0.021352754264249693, 0.0229409959924416, 0.005375451497642175], [0.0069053330960330605, 0.027367780430029007, 0.024298921418318173, 0.005625804106325206], [0.009386208216946421, 0.03210565728919472, 0.02329666415148, 0.005617867921690691], [0.006656005835223213, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.008670806814652234, 0.031634879908522776, 0.028745990407563555, 0.006416829191526845], [0.009386208216946421, 0.03106228790475094, 0.02329666415148, 0.005134401672508565], [0.006255528533129185, 0.021068382410792886, 0.01016476334865582, 0.005375451497642175], [0.006255528533129185, 0.021068382410792886, 0.01016476334865582, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.01016476334865582, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0074664156863967985, 0.03106228790475094, 0.02329666415148, 0.005134401672508565], [0.0069053330960330605, 0.021068382410792886, 0.012756016158662821, 0.005625804106325206], [0.008410969798420584, 0.021095430400269795, 0.024203118903959535, 0.005375451497642175], [0.0069053330960330605, 0.027367780430029007, 0.024298921418318173, 0.005625804106325206], [0.008670806814652234, 0.031634879908522776, 0.028745990407563555, 0.006416829191526845], [0.009386208216946421, 0.03210565728919472, 0.02329666415148, 0.005617867921690691], [0.006960109189233871, 0.032409438383303096, 0.02329666415148, 0.005617867921690691], [0.006912141710656658, 0.021454071121940723, 0.0229409959924416, 0.005375940434306164], [0.006656005835223213, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.006645560369104161, 0.021454071121940723, 0.0229409959924416, 0.005375940434306164], [0.0057401277323327025, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006340670168503612, 0.021352754264249693, 0.0229409959924416, 0.005375451497642175], [0.006159488205254294, 0.021352754264249693, 0.0229409959924416, 0.005375451497642175], [0.006344023667146063, 0.021068382410792886, 0.010103094812394669, 0.005262863122765026], [0.006255528533129185, 0.021068382410792886, 0.01016476334865582, 0.005375451497642175], [0.006255528533129185, 0.021068382410792886, 0.01016476334865582, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.01016476334865582, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0074664156863967985, 0.03106228790475094, 0.02329666415148, 0.005134401672508565], [0.0069053330960330605, 0.027367780430029007, 0.024298921418318173, 0.005625804106325206], [0.006960109189233871, 0.032409438383303096, 0.02329666415148, 0.005617867921690691], [0.0069053330960330605, 0.021068382410792886, 0.012756016158662821, 0.005625804106325206], [0.006912141710656658, 0.021454071121940723, 0.0229409959924416, 0.005375940434306164], [0.009386208216946421, 0.03210565728919472, 0.02329666415148, 0.005319455594540661], [0.0069053330960330605, 0.021068382410792886, 0.012756016158662821, 0.005625804106325206], [0.0069053330960330605, 0.027367780430029007, 0.024298921418318173, 0.005625804106325206], [0.006656005835223213, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.008670806814652234, 0.031634879908522776, 0.028745990407563555, 0.006416829191526845], [0.006645560369104161, 0.021454071121940723, 0.0229409959924416, 0.005375940434306164], [0.006159488205254294, 0.021352754264249693, 0.0229409959924416, 0.005375451497642175], [0.006342041523338799, 0.020743756434333478, 0.01016476334865582, 0.005198952718470577], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.008670806814652234, 0.031634879908522776, 0.028745990407563555, 0.006416829191526845], [0.0074664156863967985, 0.03106228790475094, 0.02329666415148, 0.005134401672508565], [0.0069053330960330605, 0.027367780430029007, 0.024298921418318173, 0.005625804106325206], [0.006928558123684635, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.0069053330960330605, 0.027367780430029007, 0.024298921418318173, 0.005625804106325206], [0.006656005835223213, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.006645560369104161, 0.021454071121940723, 0.023177865942496212, 0.010655838131797223], [0.0069053330960330605, 0.027367780430029007, 0.024298921418318173, 0.005625804106325206], [0.006645560369104161, 0.021454071121940723, 0.0229409959924416, 0.005375940434306164], [0.006912141710656658, 0.021454071121940723, 0.0229409959924416, 0.005375940434306164], [0.006159488205254294, 0.021352754264249693, 0.021902302010867205, 0.005379809743049886], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.0074664156863967985, 0.03106228790475094, 0.02329666415148, 0.005134401672508565], [0.005685951116719402, 0.021068382410792886, 0.014319811629393584, 0.005827876509134161], [0.006656005835223213, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.006342041523338799, 0.020495345360719373, 0.010270079454881398, 0.005108908205101578], [0.006342041523338799, 0.020743756434333478, 0.01016476334865582, 0.005198952718470577], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005375451497642175], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0074664156863967985, 0.03106228790475094, 0.02329666415148, 0.005134401672508565], [0.008670806814652234, 0.031634879908522776, 0.028745990407563555, 0.006416829191526845], [0.01012622646764141, 0.03106228790475094, 0.02329666415148, 0.005000233850180588], [0.0069053330960330605, 0.0269590534025459, 0.024298921418318173, 0.005625804106325206], [0.0074664156863967985, 0.030318937752080652, 0.023474058381103273, 0.005134401672508565], [0.006928558123684635, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.0069053330960330605, 0.027367780430029007, 0.024298921418318173, 0.006323353537586679], [0.0074664156863967985, 0.03106228790475094, 0.02329666415148, 0.005134401672508565], [0.006656005835223213, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.0069053330960330605, 0.027367780430029007, 0.024298921418318173, 0.005625804106325206], [0.006645560369104161, 0.021454071121940723, 0.0229409959924416, 0.005375940434306164], [0.006656005835223213, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.0069053330960330605, 0.027367780430029007, 0.024298921418318173, 0.005625804106325206], [0.006159488205254294, 0.021352754264249693, 0.021902302010867205, 0.005379809743049886], [0.006342041523338799, 0.020495345360719373, 0.010270079454881398, 0.005108908205101578], [0.006342041523338799, 0.020743756434333478, 0.01016476334865582, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0074664156863967985, 0.030393000219608574, 0.023474058381103273, 0.0050158117237382276], [0.0069053330960330605, 0.027367780430029007, 0.024298921418318173, 0.006323353537586679], [0.006645560369104161, 0.021454071121940723, 0.0229409959924416, 0.005375940434306164], [0.0074664156863967985, 0.03106228790475094, 0.0245279797126548, 0.006052139989237614], [0.0069053330960330605, 0.0269590534025459, 0.024298921418318173, 0.005625804106325206], [0.0069053330960330605, 0.0269590534025459, 0.024298921418318173, 0.005625804106325206], [0.006342041523338799, 0.020496227051305167, 0.013934861385417066, 0.005108908205101578], [0.0069286578956512706, 0.0269590534025459, 0.024298921418318173, 0.005625804106325206], [0.006656005835223213, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.006928558123684635, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.006656005835223213, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.006910945696476765, 0.032154407219981494, 0.028745990407563555, 0.006416829191526845], [0.0069053330960330605, 0.027367780430029007, 0.024298921418318173, 0.005625804106325206], [0.0063441702810197965, 0.021352754264249693, 0.02197357249204316, 0.005379809743049886], [0.0069053330960330605, 0.027367780430029007, 0.024298921418318173, 0.005625804106325206], [0.0074664156863967985, 0.030318937752080652, 0.02373756929849754, 0.005134401672508565], [0.006345400626098766, 0.021068382410792886, 0.012756016158662821, 0.005143596222086018], [0.006342041523338799, 0.020495345360719373, 0.010270079454881398, 0.005108908205101578], [0.006342041523338799, 0.020743756434333478, 0.01016476334865582, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0069053330960330605, 0.0269590534025459, 0.024298921418318173, 0.005625804106325206], [0.006910945696476765, 0.032154407219981494, 0.028745990407563555, 0.006416829191526845], [0.006645560369104161, 0.021454071121940723, 0.0229409959924416, 0.005375940434306164], [0.0074664156863967985, 0.030393000219608574, 0.023474058381103273, 0.0050158117237382276], [0.0074664156863967985, 0.030393000219608574, 0.023474058381103273, 0.005001996670224202], [0.00679290781904557, 0.027463053407923096, 0.024298921418318173, 0.006323353537586679], [0.006928558123684635, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.0074664156863967985, 0.03106228790475094, 0.0245279797126548, 0.006052139989237614], [0.006342041523338799, 0.020496227051305167, 0.013934861385417066, 0.005108908205101578], [0.0069053330960330605, 0.027367780430029007, 0.024298921418318173, 0.006323353537586679], [0.0069053330960330605, 0.0269590534025459, 0.024298921418318173, 0.005625804106325206], [0.0063680847784430405, 0.026960213678442643, 0.024298921418318173, 0.005625804106325206], [0.006656005835223213, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.006656005835223213, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.006342041523338799, 0.02027547940340143, 0.010796562573817243, 0.005198952718470577], [0.006656005835223213, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.0069286578956512706, 0.0269590534025459, 0.024298921418318173, 0.005625804106325206], [0.006342041523338799, 0.020502078449191972, 0.010270079454881398, 0.005029109393670729], [0.006342041523338799, 0.020495345360719373, 0.010270079454881398, 0.005108908205101578], [0.006342041523338799, 0.020743756434333478, 0.01016476334865582, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006910945696476765, 0.032154407219981494, 0.028745990407563555, 0.006416829191526845], [0.006645560369104161, 0.021454071121940723, 0.0229409959924416, 0.005375940434306164], [0.0074664156863967985, 0.030393000219608574, 0.023474058381103273, 0.005001996670224202], [0.0063680847784430405, 0.026960213678442643, 0.024298921418318173, 0.005625804106325206], [0.006342041523338799, 0.03049865911167613, 0.024217830142638157, 0.005198952718470577], [0.0069053330960330605, 0.0269590534025459, 0.024298921418318173, 0.005625804106325206], [0.006928558123684635, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.005018464918782205, 0.0269590534025459, 0.024419229397004038, 0.00527586319118001], [0.0074664156863967985, 0.03106228790475094, 0.0245279797126548, 0.005645376421175011], [0.0067953409212352, 0.027463053407923096, 0.024298921418318173, 0.006323353537586679], [0.00679290781904557, 0.027463053407923096, 0.024298921418318173, 0.006323353537586679], [0.006342041523338799, 0.020496227051305167, 0.013934861385417066, 0.005108908205101578], [0.006656005835223213, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.0069053330960330605, 0.027367780430029007, 0.024298921418318173, 0.006323353537586679], [0.0069053330960330605, 0.0269590534025459, 0.024298921418318173, 0.005625804106325206], [0.007215084782087199, 0.020150749296177148, 0.01427100775242876, 0.005108908205101578], [0.005906908819289854, 0.020502078449191972, 0.010270079454881398, 0.00512705053251211], [0.006342041523338799, 0.020495345360719373, 0.010270079454881398, 0.005108908205101578], [0.006342041523338799, 0.020743756434333478, 0.01016476334865582, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006910945696476765, 0.032154407219981494, 0.028745990407563555, 0.006416829191526845], [0.0063680847784430405, 0.026960213678442643, 0.024298921418318173, 0.005625804106325206], [0.005018464918782205, 0.0269590534025459, 0.024419229397004038, 0.00527586319118001], [0.0074664156863967985, 0.03106228790475094, 0.024585303405413476, 0.005645376421175011], [0.0074664156863967985, 0.030393000219608574, 0.023474058381103273, 0.005001996670224202], [0.006645560369104161, 0.021454071121940723, 0.0229409959924416, 0.005375940434306164], [0.007215084782087199, 0.020150749296177148, 0.01427100775242876, 0.005108908205101578], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.006342041523338799, 0.020496227051305167, 0.013934861385417066, 0.005108908205101578], [0.0069053330960330605, 0.0269590534025459, 0.024298921418318173, 0.005625804106325206], [0.0074664156863967985, 0.03106228790475094, 0.0245279797126548, 0.005645376421175011], [0.0069053330960330605, 0.0269590534025459, 0.024298921418318173, 0.005625804106325206], [0.006928558123684635, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.00679290781904557, 0.027463053407923096, 0.024298921418318173, 0.006323353537586679], [0.006342041523338799, 0.020743756434333478, 0.013907671261664584, 0.006431666686039272], [0.006342041523338799, 0.020329265440811806, 0.013934861385417066, 0.005108908205101578], [0.006342041523338799, 0.02010457035965816, 0.010144760695373003, 0.005198952718470577], [0.005906908819289854, 0.020502078449191972, 0.010270079454881398, 0.00512705053251211], [0.006342041523338799, 0.020495345360719373, 0.010270079454881398, 0.005108908205101578], [0.006342041523338799, 0.020743756434333478, 0.01016476334865582, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.00703915231247869, 0.030393000219608574, 0.023474058381103273, 0.005001996670224202], [0.006910945696476765, 0.032154407219981494, 0.028745990407563555, 0.006416829191526845], [0.0074664156863967985, 0.03106228790475094, 0.0245279797126548, 0.005645376421175011], [0.005018464918782205, 0.0269590534025459, 0.024419229397004038, 0.00527586319118001], [0.0050079845553487, 0.0269590534025459, 0.02607423523377811, 0.00527586319118001], [0.0075662093062044454, 0.03106228790475094, 0.02563838477984795, 0.005645376421175011], [0.006342041523338799, 0.020496227051305167, 0.013934861385417066, 0.005108908205101578], [0.006645560369104161, 0.021454071121940723, 0.0229409959924416, 0.005375940434306164], [0.0074664156863967985, 0.030393000219608574, 0.023474058381103273, 0.005001996670224202], [0.0074664156863967985, 0.03106228790475094, 0.024585303405413476, 0.005645376421175011], [0.006928558123684635, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.009162488128506084, 0.030855970849877564, 0.028093383647777456, 0.006440648740315814], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.006342041523338799, 0.020329265440811806, 0.014564474230307899, 0.005620462987376269], [0.007215084782087199, 0.020150749296177148, 0.01427100775242876, 0.005108908205101578], [0.006342041523338799, 0.02010457035965816, 0.010144760695373003, 0.005198952718470577], [0.005906908819289854, 0.020502078449191972, 0.010270079454881398, 0.00512705053251211], [0.006342041523338799, 0.020495345360719373, 0.010270079454881398, 0.005108908205101578], [0.006342041523338799, 0.020743756434333478, 0.01016476334865582, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0050079845553487, 0.0269590534025459, 0.02607423523377811, 0.00527586319118001], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.0074664156863967985, 0.03106228790475094, 0.024585303405413476, 0.005645376421175011], [0.006928558123684635, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.009162488128506084, 0.030855970849877564, 0.028093383647777456, 0.006440648740315814], [0.0074664156863967985, 0.030725286247869358, 0.0245279797126548, 0.005459617238106811], [0.006910945696476765, 0.032154407219981494, 0.028745990407563555, 0.006416829191526845], [0.0074664156863967985, 0.03106228790475094, 0.0245279797126548, 0.005645376421175011], [0.005000496242188544, 0.027767534632921476, 0.02607423523377811, 0.005008159421325727], [0.006342041523338799, 0.020496227051305167, 0.013934861385417066, 0.005108908205101578], [0.006342041523338799, 0.020496227051305167, 0.013934861385417066, 0.005108908205101578], [0.0075662093062044454, 0.03106228790475094, 0.02563838477984795, 0.007061440537226805], [0.005007057520516356, 0.026539756986571756, 0.024419229397004038, 0.00527586319118001], [0.005331840444957584, 0.02130159205731372, 0.014268967953760375, 0.005375981499090798], [0.006645560369104161, 0.021454071121940723, 0.0229409959924416, 0.005375940434306164], [0.006342041523338799, 0.02010457035965816, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.02010457035965816, 0.010144760695373003, 0.005198952718470577], [0.005906908819289854, 0.020502078449191972, 0.010270079454881398, 0.00512705053251211], [0.006342041523338799, 0.020495345360719373, 0.010270079454881398, 0.005108908205101578], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.006928558123684635, 0.021146744931969323, 0.014268967953760375, 0.005375981499090798], [0.0074664156863967985, 0.030725286247869358, 0.0245279797126548, 0.005459617238106811], [0.009162488128506084, 0.030855970849877564, 0.028093383647777456, 0.006440648740315814], [0.005000496242188544, 0.027767534632921476, 0.02607423523377811, 0.005008159421325727], [0.005331840444957584, 0.02130159205731372, 0.014268967953760375, 0.005375981499090798], [0.005007057520516356, 0.026539756986571756, 0.024419229397004038, 0.00527586319118001], [0.0074664156863967985, 0.03106228790475094, 0.026679142472300032, 0.005601342030528544], [0.0074664156863967985, 0.03106228790475094, 0.024585303405413476, 0.005645376421175011], [0.006342041523338799, 0.020496227051305167, 0.013934861385417066, 0.005108908205101578], [0.0074664156863967985, 0.03106228790475094, 0.0245279797126548, 0.005645376421175011], [0.006910945696476765, 0.032154407219981494, 0.028745990407563555, 0.006416829191526845], [0.006342041523338799, 0.020496227051305167, 0.013934861385417066, 0.005108908205101578], [0.009162488128506084, 0.03107476695023952, 0.028093383647777456, 0.006440648740315814], [0.0050079845553487, 0.0269590534025459, 0.02607423523377811, 0.00527586319118001], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.02010457035965816, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.02010457035965816, 0.010144760695373003, 0.005198952718470577], [0.005906908819289854, 0.020502078449191972, 0.010270079454881398, 0.00512705053251211], [0.006342041523338799, 0.020495345360719373, 0.010270079454881398, 0.005108908205101578], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.009162488128506084, 0.030855970849877564, 0.028093383647777456, 0.006440648740315814], [0.0074664156863967985, 0.030725286247869358, 0.0245279797126548, 0.005459617238106811], [0.006910945696476765, 0.032154407219981494, 0.028745990407563555, 0.006416829191526845], [0.0074664156863967985, 0.03106228790475094, 0.026679142472300032, 0.005601342030528544], [0.005331840444957584, 0.02130159205731372, 0.014268967953760375, 0.005375981499090798], [0.006342041523338799, 0.020496227051305167, 0.013934861385417066, 0.005108908205101578], [0.0074664156863967985, 0.03106228790475094, 0.0245279797126548, 0.005645376421175011], [0.0074664156863967985, 0.03106228790475094, 0.024585303405413476, 0.005645376421175011], [0.006342041523338799, 0.020496227051305167, 0.013934861385417066, 0.005089417261332304], [0.0074664156863967985, 0.03106228790475094, 0.026679142472300032, 0.0050684815172732465], [0.005007057520516356, 0.026539756986571756, 0.024419229397004038, 0.00527586319118001], [0.005000496242188544, 0.027767534632921476, 0.02607423523377811, 0.005008159421325727], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.02010457035965816, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.02010457035965816, 0.010144760695373003, 0.005198952718470577], [0.005906908819289854, 0.020502078449191972, 0.010270079454881398, 0.00512705053251211], [0.006342041523338799, 0.020495345360719373, 0.010270079454881398, 0.005108908205101578], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0074664156863967985, 0.030725286247869358, 0.0245279797126548, 0.005459617238106811], [0.005007057520516356, 0.026608892227706037, 0.024419229397004038, 0.00527586319118001], [0.006910945696476765, 0.032154407219981494, 0.028745990407563555, 0.006416829191526845], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.006476755026702735, 0.019153922508031416, 0.027664597299866298, 0.005185766799924772], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.005007057520516356, 0.026539756986571756, 0.024419229397004038, 0.00527586319118001], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.0074664156863967985, 0.03106228790475094, 0.026679142472300032, 0.005601342030528544], [0.0074664156863967985, 0.03106228790475094, 0.024585303405413476, 0.005645376421175011], [0.0074664156863967985, 0.03106228790475094, 0.0245279797126548, 0.005645376421175011], [0.005331840444957584, 0.02130159205731372, 0.014268967953760375, 0.008801007604672733], [0.006342041523338799, 0.020496227051305167, 0.013934861385417066, 0.005089417261332304], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.02010457035965816, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.02010457035965816, 0.010144760695373003, 0.005198952718470577], [0.005906908819289854, 0.020502078449191972, 0.010270079454881398, 0.00512705053251211], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0074664156863967985, 0.030725286247869358, 0.0245279797126548, 0.005459617238106811], [0.0074664156863967985, 0.03106228790475094, 0.026679142472300032, 0.005601342030528544], [0.005007057520516356, 0.026608892227706037, 0.024419229397004038, 0.00527586319118001], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.0074664156863967985, 0.03211131717972788, 0.026755286849006667, 0.005601342030528544], [0.006476755026702735, 0.019153922508031416, 0.027664597299866298, 0.005185766799924772], [0.005556282212684292, 0.028945956080943292, 0.024298921418318173, 0.006532261017986073], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.006910945696476765, 0.032154407219981494, 0.028745990407563555, 0.006416829191526845], [0.005007057520516356, 0.026539756986571756, 0.024419229397004038, 0.00527586319118001], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.005331840444957584, 0.02151083742855445, 0.014368503827956642, 0.008922920227231814], [0.0074664156863967985, 0.03106228790475094, 0.0245279797126548, 0.005424103095345458], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.02010457035965816, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.02010457035965816, 0.010144760695373003, 0.005198952718470577], [0.005906908819289854, 0.020502078449191972, 0.010270079454881398, 0.00512705053251211], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0074664156863967985, 0.030725286247869358, 0.0245279797126548, 0.005459617238106811], [0.008008980052257232, 0.03106228790475094, 0.026679142472300032, 0.005601342030528544], [0.0074664156863967985, 0.03106228790475094, 0.026679142472300032, 0.005601342030528544], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.005007057520516356, 0.026539756986571756, 0.024419229397004038, 0.00527586319118001], [0.006864703931352239, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.005556282212684292, 0.028945956080943292, 0.024298921418318173, 0.006532261017986073], [0.006491532783493051, 0.032154407219981494, 0.029158632267528378, 0.006416829191526845], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.005007057520516356, 0.026608892227706037, 0.024419229397004038, 0.00527586319118001], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.005331840444957584, 0.02151083742855445, 0.014368503827956642, 0.008922920227231814], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.005007057520516356, 0.02664258719440132, 0.024419229397004038, 0.00527586319118001], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.02010457035965816, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.02010457035965816, 0.010144760695373003, 0.005198952718470577], [0.005906908819289854, 0.020502078449191972, 0.010270079454881398, 0.00512705053251211], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0074664156863967985, 0.030725286247869358, 0.0245279797126548, 0.005459617238106811], [0.0074664156863967985, 0.03106228790475094, 0.026679142472300032, 0.005601342030528544], [0.006491532783493051, 0.032154407219981494, 0.029158632267528378, 0.005846406061995457], [0.005556282212684292, 0.028945956080943292, 0.024298921418318173, 0.006532261017986073], [0.005007057520516356, 0.026539756986571756, 0.024419229397004038, 0.00527586319118001], [0.006864703931352239, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.005556282212684292, 0.028945956080943292, 0.02442254482169733, 0.006532261017986073], [0.006342041523338799, 0.02010457035965816, 0.01075837733745357, 0.005192625189069395], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.005007057520516356, 0.02664258719440132, 0.024419229397004038, 0.005383096258147974], [0.005331840444957584, 0.02151083742855445, 0.014368503827956642, 0.008922920227231814], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.005007057520516356, 0.026608892227706037, 0.024419229397004038, 0.00527586319118001], [0.006342041523338799, 0.01679646957501133, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.02010457035965816, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.02010457035965816, 0.010144760695373003, 0.005198952718470577], [0.005906908819289854, 0.020502078449191972, 0.010270079454881398, 0.00512705053251211], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0074664156863967985, 0.03106228790475094, 0.026679142472300032, 0.005601342030528544], [0.006491532783493051, 0.032154407219981494, 0.029158632267528378, 0.005846406061995457], [0.0074664156863967985, 0.030725286247869358, 0.0245279797126548, 0.005459617238106811], [0.005556282212684292, 0.03054532387739037, 0.024298921418318173, 0.006532261017986073], [0.005556282212684292, 0.028945956080943292, 0.02442254482169733, 0.006532261017986073], [0.005007057520516356, 0.02664258719440132, 0.024419229397004038, 0.005383096258147974], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.005007057520516356, 0.026539756986571756, 0.024419229397004038, 0.00527586319118001], [0.0050335504619870765, 0.02894692213357084, 0.024419229397004038, 0.005274537443771711], [0.005007057520516356, 0.026539756986571756, 0.024419229397004038, 0.00527586319118001], [0.005556282212684292, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.006342041523338799, 0.02010457035965816, 0.01075837733745357, 0.005192625189069395], [0.005553740434367353, 0.03049153154975022, 0.024298921418318173, 0.005625804106325206], [0.006342041523338799, 0.016617182203032634, 0.010003381110572798, 0.005198952718470577], [0.006342041523338799, 0.01679646957501133, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.005906908819289854, 0.020502078449191972, 0.010270079454881398, 0.00512705053251211], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006491532783493051, 0.032154407219981494, 0.029158632267528378, 0.005846406061995457], [0.0074664156863967985, 0.030725286247869358, 0.0245279797126548, 0.005459617238106811], [0.006306531095596868, 0.03103900258936777, 0.024298921418318173, 0.005829756300924865], [0.0050335504619870765, 0.02894692213357084, 0.024419229397004038, 0.005274537443771711], [0.005556282212684292, 0.03097705725076671, 0.024301891900890524, 0.005625804106325206], [0.0074664156863967985, 0.03224081019148592, 0.026679142472300032, 0.005601394192627106], [0.005556282212684292, 0.028945956080943292, 0.02442254482169733, 0.006532261017986073], [0.005007057520516356, 0.02664258719440132, 0.024419229397004038, 0.005383096258147974], [0.005395547098166909, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.005560687541302301, 0.03056928449899924, 0.026174694176604438, 0.005274537443771711], [0.005906908819289854, 0.020502078449191972, 0.010270079454881398, 0.00512705053251211], [0.005007057520516356, 0.026539756986571756, 0.024419229397004038, 0.00527586319118001], [0.006342041523338799, 0.02010457035965816, 0.01075837733745357, 0.005192625189069395], [0.005007057520516356, 0.026539756986571756, 0.024419229397004038, 0.00527586319118001], [0.006342041523338799, 0.016617182203032634, 0.010003381110572798, 0.005198952718470577], [0.006342041523338799, 0.01679646957501133, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006306531095596868, 0.03103900258936777, 0.024298921418318173, 0.005829756300924865], [0.0074664156863967985, 0.03224081019148592, 0.026679142472300032, 0.005601394192627106], [0.006491532783493051, 0.032154407219981494, 0.030943412613024076, 0.005846406061995457], [0.005395547098166909, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.006342041523338799, 0.02010457035965816, 0.01075837733745357, 0.005192625189069395], [0.006491532783493051, 0.032154407219981494, 0.029158632267528378, 0.005846406061995457], [0.0074664156863967985, 0.03209536538006277, 0.027476743752422483, 0.005601394192627106], [0.005556282212684292, 0.028945956080943292, 0.02442254482169733, 0.006532261017986073], [0.0050335504619870765, 0.02894692213357084, 0.024419229397004038, 0.005274537443771711], [0.005556282212684292, 0.03097705725076671, 0.024301891900890524, 0.005625804106325206], [0.006342041523338799, 0.02654659776360939, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.017406316829493625, 0.012217540808199728, 0.00505757076194898], [0.005556282212684292, 0.03097705725076671, 0.024301891900890524, 0.005625804106325206], [0.005033196947395795, 0.02894692213357084, 0.024419229397004038, 0.006016128510642959], [0.005007057520516356, 0.02664258719440132, 0.024419229397004038, 0.005383096258147974], [0.006342041523338799, 0.016617182203032634, 0.010003381110572798, 0.005198952718470577], [0.006342041523338799, 0.01679646957501133, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006491532783493051, 0.032154407219981494, 0.030943412613024076, 0.005846406061995457], [0.0074664156863967985, 0.03224081019148592, 0.026679142472300032, 0.005601394192627106], [0.006306531095596868, 0.03103900258936777, 0.024298921418318173, 0.005829756300924865], [0.006377816228509599, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.006342041523338799, 0.02010457035965816, 0.01075837733745357, 0.005192625189069395], [0.005007057520516356, 0.02664258719440132, 0.024419229397004038, 0.005383096258147974], [0.005395547098166909, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.006306531095596868, 0.03103900258936777, 0.024298921418318173, 0.005829756300924865], [0.006342041523338799, 0.024413430565548302, 0.01038401697215871, 0.005198952718470577], [0.005556282212684292, 0.03097705725076671, 0.024301891900890524, 0.005625804106325206], [0.006491532783493051, 0.032154407219981494, 0.029158632267528378, 0.005846406061995457], [0.0050335504619870765, 0.02894692213357084, 0.024419229397004038, 0.005274537443771711], [0.005396436196264339, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.0074664156863967985, 0.03209536538006277, 0.027476743752422483, 0.005601394192627106], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005077893584830725], [0.0058932221521298955, 0.01602214457495366, 0.010003381110572798, 0.005198952718470577], [0.006342041523338799, 0.01679646957501133, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006377816228509599, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.006342041523338799, 0.024413430565548302, 0.01038401697215871, 0.005198952718470577], [0.006491532783493051, 0.032154407219981494, 0.030943412613024076, 0.005846406061995457], [0.0074664156863967985, 0.03224081019148592, 0.026679142472300032, 0.005601394192627106], [0.006342041523338799, 0.02010457035965816, 0.01075837733745357, 0.005192625189069395], [0.006491532783493051, 0.032154407219981494, 0.030943412613024076, 0.005846406061995457], [0.0074664156863967985, 0.03209536538006277, 0.027476743752422483, 0.005601394192627106], [0.005396436196264339, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.006306531095596868, 0.03103900258936777, 0.024298921418318173, 0.005829756300924865], [0.005395547098166909, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.006306531095596868, 0.03103900258936777, 0.024298921418318173, 0.005829756300924865], [0.00602046550904368, 0.01679646957501133, 0.010019742202058316, 0.005198952718470577], [0.005556282212684292, 0.03097705725076671, 0.024301891900890524, 0.005625804106325206], [0.006324278742081681, 0.03097705725076671, 0.024301891900890524, 0.005625804106325206], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005077893584830725], [0.0058932221521298955, 0.015990843628093925, 0.010019905316710223, 0.005198952718470577], [0.0058932221521298955, 0.01602214457495366, 0.010003381110572798, 0.005198952718470577], [0.006342041523338799, 0.01679646957501133, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0074664156863967985, 0.03209536538006277, 0.027476743752422483, 0.005601394192627106], [0.006377816228509599, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.006491532783493051, 0.032154407219981494, 0.030943412613024076, 0.005846406061995457], [0.006342041523338799, 0.024413430565548302, 0.01038401697215871, 0.005198952718470577], [0.005396436196264339, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.0074664156863967985, 0.032271751607132304, 0.027476743752422483, 0.005603051642286915], [0.006491532783493051, 0.032154407219981494, 0.030943412613024076, 0.005846406061995457], [0.005556282212684292, 0.03097705725076671, 0.024301891900890524, 0.005625804106325206], [0.006324278742081681, 0.03097705725076671, 0.024301891900890524, 0.005625804106325206], [0.005395547098166909, 0.032092897717224865, 0.027454162903091408, 0.005381525112014932], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006377816228509599, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.006342041523338799, 0.024777722090640136, 0.01038401697215871, 0.005198952718470577], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005077893584830725], [0.0058932221521298955, 0.015990843628093925, 0.010019905316710223, 0.005198952718470577], [0.0058932221521298955, 0.01602214457495366, 0.010003381110572798, 0.005198952718470577], [0.006342041523338799, 0.01679646957501133, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006491532783493051, 0.032154407219981494, 0.030943412613024076, 0.005846406061995457], [0.0074664156863967985, 0.03209536538006277, 0.027476743752422483, 0.005601394192627106], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005077893584830725], [0.006342041523338799, 0.01935555945233449, 0.010019742202058316, 0.005198952718470577], [0.005395547098166909, 0.032092897717224865, 0.027454162903091408, 0.005381525112014932], [0.006377816228509599, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.006377816228509599, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.005556282212684292, 0.03097705725076671, 0.024301891900890524, 0.005625804106325206], [0.005396436196264339, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.009111195850412647, 0.018952444784987314, 0.01034328135854901, 0.006424979287658931], [0.0058932221521298955, 0.0158911474146002, 0.027536753249540892, 0.0052066125225243235], [0.0074664156863967985, 0.032271751607132304, 0.027476743752422483, 0.005603051642286915], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.016519622013112478, 0.010019742202058316, 0.005198952718470577], [0.0058932221521298955, 0.015990843628093925, 0.010019905316710223, 0.005198952718470577], [0.0058932221521298955, 0.015990843628093925, 0.010019905316710223, 0.005198952718470577], [0.0058932221521298955, 0.01602214457495366, 0.010003381110572798, 0.005198952718470577], [0.006342041523338799, 0.01679646957501133, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006491532783493051, 0.032154407219981494, 0.030943412613024076, 0.005846406061995457], [0.006342041523338799, 0.019364985854054606, 0.010019742202058316, 0.005198952718470577], [0.0074664156863967985, 0.03209536538006277, 0.027476743752422483, 0.005601394192627106], [0.006374362420622133, 0.026819391888284496, 0.024419229397004038, 0.005077893584830725], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005077893584830725], [0.005864326847021778, 0.032092897717224865, 0.027454162903091408, 0.005381525112014932], [0.006342041523338799, 0.01935555945233449, 0.010019742202058316, 0.005198952718470577], [0.006377816228509599, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.005396436196264339, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.006299045674135657, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.006377816228509599, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.006342041523338799, 0.016519622013112478, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.0058932221521298955, 0.015990843628093925, 0.010019905316710223, 0.005198952718470577], [0.0058932221521298955, 0.015990843628093925, 0.010019905316710223, 0.005198952718470577], [0.0058932221521298955, 0.01602214457495366, 0.010003381110572798, 0.005198952718470577], [0.006342041523338799, 0.01679646957501133, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0074664156863967985, 0.03209536538006277, 0.027476743752422483, 0.005601394192627106], [0.006374362420622133, 0.026819391888284496, 0.024419229397004038, 0.005077893584830725], [0.005864326847021778, 0.032092897717224865, 0.027454162903091408, 0.005381525112014932], [0.006491532783493051, 0.032154407219981494, 0.030943412613024076, 0.005846406061995457], [0.006342041523338799, 0.019364985854054606, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.016519622013112478, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.022864327211426415, 0.010079109369076595, 0.005198952718470577], [0.006299045674135657, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.005396436196264339, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005077893584830725], [0.006342041523338799, 0.019569488127203525, 0.010019742202058316, 0.005247038021994612], [0.006342041523338799, 0.01935555945233449, 0.010019742202058316, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.0058932221521298955, 0.015990843628093925, 0.010019905316710223, 0.005198952718470577], [0.0058932221521298955, 0.015990843628093925, 0.010019905316710223, 0.005198952718470577], [0.0058932221521298955, 0.01602214457495366, 0.010003381110572798, 0.005198952718470577], [0.006342041523338799, 0.01679646957501133, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006491532783493051, 0.032154407219981494, 0.030943412613024076, 0.005846406061995457], [0.006374362420622133, 0.026819391888284496, 0.024419229397004038, 0.005077893584830725], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005077893584830725], [0.006299045674135657, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.0074664156863967985, 0.03209536538006277, 0.027476743752422483, 0.005601394192627106], [0.0059238919035131885, 0.01944059390927788, 0.023536416336095954, 0.006509465603991173], [0.0074664156863967985, 0.03209536538006277, 0.027476743752422483, 0.005601394192627106], [0.0074664156863967985, 0.03186252820185349, 0.02743192702882102, 0.005601394192627106], [0.006342041523338799, 0.016519622013112478, 0.010019742202058316, 0.005198952718470577], [0.005864326847021778, 0.032092897717224865, 0.027454162903091408, 0.005381525112014932], [0.006342041523338799, 0.019364985854054606, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019364985854054606, 0.010019742202058316, 0.006134534661543615], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.0058932221521298955, 0.015990843628093925, 0.010019905316710223, 0.005198952718470577], [0.0058932221521298955, 0.015990843628093925, 0.010019905316710223, 0.005198952718470577], [0.0058932221521298955, 0.01602214457495366, 0.010003381110572798, 0.005198952718470577], [0.006342041523338799, 0.01679646957501133, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006374362420622133, 0.026819391888284496, 0.024419229397004038, 0.005077893584830725], [0.006342041523338799, 0.019364985854054606, 0.010019742202058316, 0.006134534661543615], [0.0074664156863967985, 0.03186252820185349, 0.02743192702882102, 0.005601394192627106], [0.006491532783493051, 0.032154407219981494, 0.030943412613024076, 0.005846406061995457], [0.0059238919035131885, 0.01944059390927788, 0.023536416336095954, 0.006509465603991173], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005077893584830725], [0.006299045674135657, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.005864326847021778, 0.032092897717224865, 0.027454162903091408, 0.005381525112014932], [0.0074664156863967985, 0.03209536538006277, 0.027476743752422483, 0.005601394192627106], [0.0074664156863967985, 0.03209536538006277, 0.027476743752422483, 0.005601394192627106], [0.006342041523338799, 0.018890069097873382, 0.010006322483534157, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.0058932221521298955, 0.015990843628093925, 0.010019905316710223, 0.005198952718470577], [0.0058932221521298955, 0.015990843628093925, 0.010019905316710223, 0.005198952718470577], [0.0058932221521298955, 0.01602214457495366, 0.010003381110572798, 0.005198952718470577], [0.006342041523338799, 0.01679646957501133, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0059238919035131885, 0.01944059390927788, 0.023536416336095954, 0.006509465603991173], [0.006491532783493051, 0.032154407219981494, 0.030943412613024076, 0.005846406061995457], [0.0074664156863967985, 0.03186252820185349, 0.02743192702882102, 0.005601394192627106], [0.006299045674135657, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005077893584830725], [0.009951379526527315, 0.01944059390927788, 0.02430671293285077, 0.006509465603991173], [0.005245703620893705, 0.01679646957501133, 0.010006316269806631, 0.005198952718470577], [0.006374362420622133, 0.026819391888284496, 0.024419229397004038, 0.005077893584830725], [0.0059238919035131885, 0.01944059390927788, 0.023536416336095954, 0.006509465603991173], [0.005677655782410327, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.006342041523338799, 0.018890069097873382, 0.010006322483534157, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.0058932221521298955, 0.01602214457495366, 0.010003381110572798, 0.005198952718470577], [0.006342041523338799, 0.01679646957501133, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006491532783493051, 0.032154407219981494, 0.030943412613024076, 0.005846406061995457], [0.009951379526527315, 0.01944059390927788, 0.02430671293285077, 0.006509465603991173], [0.0074664156863967985, 0.03186252820185349, 0.02743192702882102, 0.005601394192627106], [0.006299045674135657, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.0059238919035131885, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005077893584830725], [0.006374362420622133, 0.026819391888284496, 0.024419229397004038, 0.005077893584830725], [0.0074664156863967985, 0.03154963980328353, 0.02743192702882102, 0.005601394192627106], [0.0058932221521298955, 0.019223114707816367, 0.010920951662763084, 0.005198952718470577], [0.006342041523338799, 0.018890069097873382, 0.010006322483534157, 0.005198952718470577], [0.005245703620893705, 0.01679646957501133, 0.010006316269806631, 0.005198952718470577], [0.005677655782410327, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.0058932221521298955, 0.01602214457495366, 0.010003381110572798, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.005867723848283135, 0.019223114707816367, 0.010920951662763084, 0.006804830704394255], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005077893584830725], [0.006188199032652241, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.009951379526527315, 0.01944059390927788, 0.02430671293285077, 0.006509465603991173], [0.006491532783493051, 0.032154407219981494, 0.030943412613024076, 0.005846406061995457], [0.0058932221521298955, 0.01602214457495366, 0.010003381110572798, 0.005198952718470577], [0.009951379526527315, 0.01944059390927788, 0.02349613449099225, 0.006509465603991173], [0.006342041523338799, 0.018890069097873382, 0.010006322483534157, 0.005198952718470577], [0.0074664156863967985, 0.03154963980328353, 0.02743192702882102, 0.005601394192627106], [0.0074664156863967985, 0.03186252820185349, 0.02743192702882102, 0.005601394192627106], [0.007146667387148644, 0.03190253986099357, 0.028093383647777456, 0.006440648740315814], [0.005245703620893705, 0.01679646957501133, 0.010006316269806631, 0.005198952718470577], [0.006299045674135657, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.0058932221521298955, 0.01602214457495366, 0.010003381110572798, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.009951379526527315, 0.01944059390927788, 0.02349613449099225, 0.006509465603991173], [0.006342041523338799, 0.018890069097873382, 0.02674155669647058, 0.0055997960427886795], [0.006491532783493051, 0.032154407219981494, 0.030943412613024076, 0.005846406061995457], [0.006188199032652241, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.0074664156863967985, 0.03154963980328353, 0.02743192702882102, 0.005601394192627106], [0.0058932221521298955, 0.01602214457495366, 0.010003381110572798, 0.005198952718470577], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005077893584830725], [0.009951379526527315, 0.01944059390927788, 0.02430671293285077, 0.006509465603991173], [0.006299045674135657, 0.03173403360958757, 0.026691223477177624, 0.005381525112014932], [0.006363293341727147, 0.018712032606066877, 0.010019742202058316, 0.005198952718470577], [0.005245703620893705, 0.01679646957501133, 0.010006316269806631, 0.005198952718470577], [0.006188199032652241, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.0051846115761800884, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.0058932221521298955, 0.01602214457495366, 0.010003381110572798, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.009951379526527315, 0.01944059390927788, 0.02349613449099225, 0.006509465603991173], [0.0074664156863967985, 0.03154963980328353, 0.02743192702882102, 0.005601394192627106], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005107643702521533], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005077893584830725], [0.006342041523338799, 0.018890069097873382, 0.02674155669647058, 0.0055997960427886795], [0.005369121067801104, 0.017162792989505515, 0.010019228860957269, 0.005198952718470577], [0.006491532783493051, 0.032154407219981494, 0.030943412613024076, 0.005790143735848345], [0.006342041523338799, 0.018890069097873382, 0.02674155669647058, 0.0055997960427886795], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006307973455061285], [0.006188199032652241, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.0058932221521298955, 0.01602214457495366, 0.010003381110572798, 0.005198952718470577], [0.0051846115761800884, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.0058932221521298955, 0.01602214457495366, 0.010003381110572798, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.006342041523338799, 0.019153922508031416, 0.010019742202058316, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0075708893180679955, 0.017162792989505515, 0.010019228860957269, 0.00512311655622703], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005077893584830725], [0.006342041523338799, 0.018890069097873382, 0.02674155669647058, 0.005110486946255112], [0.005369121067801104, 0.017162792989505515, 0.010019228860957269, 0.005198952718470577], [0.009233466747927273, 0.02741280105877409, 0.028025101653815433, 0.005108290394046696], [0.006491532783493051, 0.032154407219981494, 0.030943412613024076, 0.005790143735848345], [0.0074664156863967985, 0.03154963980328353, 0.02743192702882102, 0.005601394192627106], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.00512695072784651], [0.009951379526527315, 0.01944059390927788, 0.02349613449099225, 0.006509465603991173], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005107643702521533], [0.006188199032652241, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.005595526103283799, 0.015138962953187752, 0.010019228860957269, 0.005021292928140091], [0.0051846115761800884, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.0058932221521298955, 0.01602214457495366, 0.010003381110572798, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006342041523338799, 0.018890069097873382, 0.02674155669647058, 0.005110486946255112], [0.009233466747927273, 0.02741280105877409, 0.028025101653815433, 0.005108290394046696], [0.006491532783493051, 0.032154407219981494, 0.030943412613024076, 0.005790143735848345], [0.0075708893180679955, 0.017162792989505515, 0.010019228860957269, 0.00512311655622703], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.00512695072784651], [0.005369121067801104, 0.017162792989505515, 0.010019228860957269, 0.005198952718470577], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005077893584830725], [0.009168045872227422, 0.02741280105877409, 0.028025101653815433, 0.005108290394046696], [0.006188199032652241, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.00578470412082191, 0.032154407219981494, 0.030943412613024076, 0.005790143735848345], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005107643702521533], [0.0058932221521298955, 0.016446104909045946, 0.012596786986852612, 0.005198952718470577], [0.005595526103283799, 0.015138962953187752, 0.010019228860957269, 0.005021292928140091], [0.0051846115761800884, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.0058932221521298955, 0.01602214457495366, 0.010003381110572798, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006491532783493051, 0.032154407219981494, 0.030943412613024076, 0.005790143735848345], [0.006342041523338799, 0.018890069097873382, 0.02674155669647058, 0.005110486946255112], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.00512695072784651], [0.009168045872227422, 0.02741280105877409, 0.028025101653815433, 0.005108290394046696], [0.005369121067801104, 0.017162792989505515, 0.010019228860957269, 0.005198952718470577], [0.009233466747927273, 0.02741280105877409, 0.028025101653815433, 0.005108290394046696], [0.011340465729100913, 0.032154407219981494, 0.030943412613024076, 0.005790143735848345], [0.006374362420622133, 0.028726691784986734, 0.024250612626068844, 0.005077893584830725], [0.00578470412082191, 0.032154407219981494, 0.030943412613024076, 0.005790143735848345], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005077893584830725], [0.005618026871721269, 0.031028121790117356, 0.010019228860957269, 0.0064276240056413605], [0.006188199032652241, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.005595526103283799, 0.015138962953187752, 0.010019228860957269, 0.005021292928140091], [0.0051846115761800884, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.0058932221521298955, 0.01602214457495366, 0.010003381110572798, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006491532783493051, 0.032154407219981494, 0.030943412613024076, 0.005790143735848345], [0.009168045872227422, 0.02741280105877409, 0.028025101653815433, 0.005108290394046696], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.00512695072784651], [0.0058932221521298955, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.006374362420622133, 0.028726691784986734, 0.024250612626068844, 0.005077893584830725], [0.006342041523338799, 0.018890069097873382, 0.02674155669647058, 0.005110486946255112], [0.009233466747927273, 0.02741280105877409, 0.028025101653815433, 0.005108290394046696], [0.00578470412082191, 0.032154407219981494, 0.030943412613024076, 0.005790143735848345], [0.005369121067801104, 0.017162792989505515, 0.010019228860957269, 0.005198952718470577], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005077893584830725], [0.006191451601503053, 0.032154407219981494, 0.030943412613024076, 0.005768508881847319], [0.006188199032652241, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.005595526103283799, 0.015138962953187752, 0.010019228860957269, 0.005021292928140091], [0.0051846115761800884, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0058932221521298955, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.00512695072784651], [0.009168045872227422, 0.02741280105877409, 0.028025101653815433, 0.005108290394046696], [0.006374362420622133, 0.028726691784986734, 0.024250612626068844, 0.005077893584830725], [0.006342041523338799, 0.03224106236577426, 0.03053605427732613, 0.005050614712316471], [0.006188199032652241, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.006342041523338799, 0.018890069097873382, 0.02674155669647058, 0.005110486946255112], [0.0058932221521298955, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.005077893584830725], [0.005369121067801104, 0.017162792989505515, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.006374362420622133, 0.028726691784986734, 0.02581012322511635, 0.005077893584830725], [0.005595526103283799, 0.015138962953187752, 0.010019228860957269, 0.005021292928140091], [0.0051846115761800884, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.009168045872227422, 0.02741280105877409, 0.028025101653815433, 0.005108290394046696], [0.0058932221521298955, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.006342041523338799, 0.03224106236577426, 0.03053605427732613, 0.005050614712316471], [0.006374362420622133, 0.028726691784986734, 0.02581012322511635, 0.005077893584830725], [0.00880018835116222, 0.032217524432047666, 0.03125638758609301, 0.006440648740315814], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.00512695072784651], [0.006374362420622133, 0.028726691784986734, 0.024250612626068844, 0.005077893584830725], [0.006374362420622133, 0.028726691784986734, 0.024250612626068844, 0.005077893584830725], [0.0058932221521298955, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.006188199032652241, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.005369121067801104, 0.017162792989505515, 0.010019228860957269, 0.005198952718470577], [0.006188199032652241, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005080117468257812, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.0051846115761800884, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006342041523338799, 0.03224106236577426, 0.03053605427732613, 0.005050614712316471], [0.009168045872227422, 0.02741280105877409, 0.028025101653815433, 0.005108290394046696], [0.006877966355576547, 0.028726691784986734, 0.02581012322511635, 0.005081789867670985], [0.006187297545280034, 0.02001074489132886, 0.02428515951978368, 0.006509465603991173], [0.006374362420622133, 0.028726691784986734, 0.02581012322511635, 0.005077893584830725], [0.0058932221521298955, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.006374362420622133, 0.028726691784986734, 0.024250612626068844, 0.005077893584830725], [0.006374362420622133, 0.028726691784986734, 0.023877923277726105, 0.00501608649559957], [0.006374362420622133, 0.028726691784986734, 0.024419229397004038, 0.00512695072784651], [0.005959521835390086, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.006374362420622133, 0.028726691784986734, 0.024250612626068844, 0.005077893584830725], [0.0058932221521298955, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005080117468257812, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.0051846115761800884, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006342041523338799, 0.03224106236577426, 0.03053605427732613, 0.005050614712316471], [0.006187297545280034, 0.02001074489132886, 0.02428515951978368, 0.006509465603991173], [0.006374362420622133, 0.028726691784986734, 0.024250612626068844, 0.005077893584830725], [0.005561965207174634, 0.028234451479742198, 0.02462496015032389, 0.005198952718470577], [0.006374362420622133, 0.028726691784986734, 0.02581012322511635, 0.005077893584830725], [0.0058932221521298955, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.006384031402437705, 0.028631044334832063, 0.026310918509977743, 0.006509465603991173], [0.006374362420622133, 0.028726691784986734, 0.024250612626068844, 0.005077893584830725], [0.0058932221521298955, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.005595526103283799, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015377195447811234, 0.012884390286656555, 0.005198952718470577], [0.005959521835390086, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.006639511780462624, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.005080117468257812, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005080117468257812, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.0051846115761800884, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006639511780462624, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006187297545280034, 0.02001074489132886, 0.02428515951978368, 0.006509465603991173], [0.0058932221521298955, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.005561965207174634, 0.028234451479742198, 0.024359892278287158, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.006342041523338799, 0.03224106236577426, 0.03053605427732613, 0.005050614712316471], [0.005561965207174634, 0.028234451479742198, 0.02462496015032389, 0.005198952718470577], [0.006187297545280034, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006384031402437705, 0.028631044334832063, 0.026310918509977743, 0.006509465603991173], [0.0058932221521298955, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.006374362420622133, 0.028726691784986734, 0.024250612626068844, 0.005077893584830725], [0.005595526103283799, 0.015377195447811234, 0.012884390286656555, 0.005198952718470577], [0.006374362420622133, 0.028726691784986734, 0.024250612626068844, 0.005077893584830725], [0.005959521835390086, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.005080117468257812, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005080117468257812, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.0051846115761800884, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006187297545280034, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006342041523338799, 0.032544146382856444, 0.03053605427732613, 0.0050504559175926405], [0.006639511780462624, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006187297545280034, 0.02001074489132886, 0.02428515951978368, 0.006509465603991173], [0.005959521835390086, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.006384031402437705, 0.028631044334832063, 0.026310918509977743, 0.006509465603991173], [0.006639511780462624, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0058932221521298955, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.00874818400167653, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.0058932221521298955, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.005561965207174634, 0.028234451479742198, 0.02462496015032389, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.006639511780462624, 0.03134551226037156, 0.028093383647777456, 0.006440648740315814], [0.005835878830284347, 0.028726691784986734, 0.024250612626068844, 0.005077893584830725], [0.005080117468257812, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005080117468257812, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.0051846115761800884, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006342041523338799, 0.032544146382856444, 0.03053605427732613, 0.0050504559175926405], [0.00874818400167653, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.007223666742838813, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006639511780462624, 0.0318964083520743, 0.028093383647777456, 0.006440648740315814], [0.0058932221521298955, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.006187297545280034, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006384031402437705, 0.028631044334832063, 0.026310918509977743, 0.006509465603991173], [0.006187297545280034, 0.02001074489132886, 0.02428515951978368, 0.006509465603991173], [0.006639511780462624, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.005959521835390086, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.005959521835390086, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.006187297545280034, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006639511780462624, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.005835878830284347, 0.028726691784986734, 0.024250612626068844, 0.005077893584830725], [0.005080117468257812, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005080117468257812, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005080117468257812, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.0051846115761800884, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.00874818400167653, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006639511780462624, 0.0318964083520743, 0.028093383647777456, 0.006440648740315814], [0.006384031402437705, 0.028631044334832063, 0.026310918509977743, 0.006509465603991173], [0.007223666742838813, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006342041523338799, 0.032544146382856444, 0.03053605427732613, 0.0050504559175926405], [0.0058932221521298955, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.00874818400167653, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006187297545280034, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.007223666742838813, 0.02857422885561665, 0.028118377384904553, 0.006509465603991173], [0.005814825602571267, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.005959521835390086, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.006187297545280034, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006639511780462624, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0050282369007730995, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.00874818400167653, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006384031402437705, 0.028631044334832063, 0.026310918509977743, 0.006509465603991173], [0.006639511780462624, 0.031651269210197744, 0.028404121355019553, 0.006440648740315814], [0.006639511780462624, 0.0318964083520743, 0.028093383647777456, 0.006440648740315814], [0.0051846115761800884, 0.02859514018536135, 0.02580273650959637, 0.005198952718470577], [0.007223666742838813, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006187297545280034, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.0058932221521298955, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.007223666742838813, 0.02857422885561665, 0.028118377384904553, 0.006509465603991173], [0.006342041523338799, 0.032544146382856444, 0.03053605427732613, 0.0050504559175926405], [0.005814825602571267, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.005959521835390086, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.006384031402437705, 0.028631044334832063, 0.026310918509977743, 0.006509465603991173], [0.006639511780462624, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.005080117468257812, 0.015377195447811234, 0.010085898118116367, 0.005127368637549481], [0.006187297545280034, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.0050282369007730995, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.007223666742838813, 0.02857422885561665, 0.028118377384904553, 0.006509465603991173], [0.0051846115761800884, 0.02859514018536135, 0.02580273650959637, 0.005198952718470577], [0.006384031402437705, 0.028631044334832063, 0.026310918509977743, 0.005697960202630839], [0.007223666742838813, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006639511780462624, 0.0318964083520743, 0.028093383647777456, 0.006440648740315814], [0.00874818400167653, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006187297545280034, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006639511780462624, 0.0318964083520743, 0.028093383647777456, 0.006440648740315814], [0.0058932221521298955, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.005080117468257812, 0.015377195447811234, 0.010085898118116367, 0.005127368637549481], [0.006748681659678107, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006342041523338799, 0.032544146382856444, 0.03053605427732613, 0.0050504559175926405], [0.006384031402437705, 0.029787352738001215, 0.026310918509977743, 0.006509465603991173], [0.00646758336340416, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.005959521835390086, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.005814825602571267, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.0050282369007730995, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0051846115761800884, 0.02859514018536135, 0.02580273650959637, 0.005198952718470577], [0.006384031402437705, 0.028631044334832063, 0.026310918509977743, 0.005697960202630839], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006639511780462624, 0.0318964083520743, 0.028093383647777456, 0.006440648740315814], [0.006342041523338799, 0.032544146382856444, 0.03053605427732613, 0.0050504559175926405], [0.0058932221521298955, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.007223666742838813, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006639511780462624, 0.0318964083520743, 0.028093383647777456, 0.006440648740315814], [0.007223666742838813, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006639511780462624, 0.0318964083520743, 0.028093383647777456, 0.006440648740315814], [0.005814825602571267, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.006748681659678107, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006187297545280034, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.005959521835390086, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.0051846115761800884, 0.02859514018536135, 0.0260408860549808, 0.005198952718470577], [0.006187297545280034, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.0050282369007730995, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006384031402437705, 0.028631044334832063, 0.02745219139826682, 0.005697960202630839], [0.007223666742838813, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006342041523338799, 0.032544146382856444, 0.03053605427732613, 0.0050504559175926405], [0.0051846115761800884, 0.02859514018536135, 0.02580273650959637, 0.005198952718470577], [0.006639511780462624, 0.0318964083520743, 0.028093383647777456, 0.00692958647283161], [0.005567579439943247, 0.02001074489132886, 0.026214414170064062, 0.006509465603991173], [0.006384031402437705, 0.028631044334832063, 0.026310918509977743, 0.005697960202630839], [0.0058932221521298955, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.006639511780462624, 0.0318964083520743, 0.028093383647777456, 0.006440648740315814], [0.005814825602571267, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.0051846115761800884, 0.02859514018536135, 0.0260408860549808, 0.005198952718470577], [0.006384031402437705, 0.028465166525924998, 0.026310918509977743, 0.005697960202630839], [0.006748681659678107, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.007223666742838813, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006748681659678107, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006639511780462624, 0.0318964083520743, 0.028093383647777456, 0.006440648740315814], [0.0050282369007730995, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.0058932221521298955, 0.01602214457495366, 0.02430122300069735, 0.005166826696731704], [0.006384031402437705, 0.028631044334832063, 0.02745219139826682, 0.005697960202630839], [0.007223666742838813, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006639511780462624, 0.0318964083520743, 0.028093383647777456, 0.005270524867219299], [0.006639511780462624, 0.0318964083520743, 0.026663504653936118, 0.006160546782880607], [0.006384031402437705, 0.028631044334832063, 0.026310918509977743, 0.005697960202630839], [0.0051846115761800884, 0.02859514018536135, 0.0260408860549808, 0.005198952718470577], [0.006748681659678107, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006342041523338799, 0.032544146382856444, 0.03053605427732613, 0.0050504559175926405], [0.005567579439943247, 0.02001074489132886, 0.026214414170064062, 0.006509465603991173], [0.0051846115761800884, 0.02859514018536135, 0.02657491803687519, 0.005198952718470577], [0.006748681659678107, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.005814825602571267, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.007223666742838813, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.005814825602571267, 0.01944059390927788, 0.023955639493087655, 0.006509465603991173], [0.008670391806960494, 0.01602214457495366, 0.02430122300069735, 0.005103439656448715], [0.0050282369007730995, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006639511780462624, 0.0318964083520743, 0.028093383647777456, 0.005270524867219299], [0.007223666742838813, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.0051846115761800884, 0.02859514018536135, 0.0260408860549808, 0.005198952718470577], [0.006342041523338799, 0.032544146382856444, 0.03053605427732613, 0.0050504559175926405], [0.006384031402437705, 0.028631044334832063, 0.02745219139826682, 0.005697960202630839], [0.005595526103283799, 0.015939638383512072, 0.0251383346208637, 0.005198952718470577], [0.005595526103283799, 0.01917211114036606, 0.012675983555643415, 0.005552634153488642], [0.006384031402437705, 0.028631044334832063, 0.026310918509977743, 0.005697960202630839], [0.005814825602571267, 0.01944059390927788, 0.023989824485045042, 0.006509465603991173], [0.0051846115761800884, 0.02859514018536135, 0.02657491803687519, 0.005198952718470577], [0.006748681659678107, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006748681659678107, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006639511780462624, 0.0318964083520743, 0.026663504653936118, 0.006160546782880607], [0.008670391806960494, 0.01602214457495366, 0.02430122300069735, 0.005103439656448715], [0.0057448806469450366, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.005595526103283799, 0.0158634849759992, 0.010951424083173542, 0.007621729576744483], [0.0050282369007730995, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006342041523338799, 0.032544146382856444, 0.03053605427732613, 0.0050504559175926405], [0.007223666742838813, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006639511780462624, 0.0318964083520743, 0.026663504653936118, 0.005386541893343448], [0.0051846115761800884, 0.02859514018536135, 0.0260408860549808, 0.005198952718470577], [0.00675061659848341, 0.028631044334832063, 0.02857704499558852, 0.006091508036089105], [0.008670391806960494, 0.01602214457495366, 0.02430122300069735, 0.005103439656448715], [0.006748681659678107, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.005595526103283799, 0.015939638383512072, 0.0251383346208637, 0.005198952718470577], [0.005595526103283799, 0.01917211114036606, 0.012675983555643415, 0.005552634153488642], [0.005595526103283799, 0.0158634849759992, 0.010951424083173542, 0.007621729576744483], [0.006748681659678107, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.005191693155763629, 0.028594425119226864, 0.027394321254812903, 0.005198952718470577], [0.0057448806469450366, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.0051846115761800884, 0.02859514018536135, 0.02657491803687519, 0.005198952718470577], [0.006639511780462624, 0.0318964083520743, 0.028093383647777456, 0.005270524867219299], [0.0051846115761800884, 0.02859514018536135, 0.02657491803687519, 0.005198952718470577], [0.0050282369007730995, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015939638383512072, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006342041523338799, 0.032544146382856444, 0.03053605427732613, 0.0050504559175926405], [0.006639511780462624, 0.0318964083520743, 0.028093383647777456, 0.005270524867219299], [0.00675061659848341, 0.028631044334832063, 0.02857704499558852, 0.006091508036089105], [0.005595526103283799, 0.015939638383512072, 0.0251383346208637, 0.005198952718470577], [0.008670391806960494, 0.01602214457495366, 0.02430122300069735, 0.005103439656448715], [0.005191693155763629, 0.028594425119226864, 0.027394321254812903, 0.005198952718470577], [0.006748681659678107, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.005015797228865461, 0.027341194645033724, 0.02657491803687519, 0.0051740168192639996], [0.0057448806469450366, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006639511780462624, 0.0318964083520743, 0.026663504653936118, 0.005386541893343448], [0.007223666742838813, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.005595526103283799, 0.0158634849759992, 0.010951424083173542, 0.007820447135013129], [0.006748681659678107, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.0059507225630965254, 0.019266589558422566, 0.0251383346208637, 0.005198952718470577], [0.005595526103283799, 0.0158634849759992, 0.010951424083173542, 0.007621729576744483], [0.005595526103283799, 0.01917211114036606, 0.012675983555643415, 0.005552634153488642], [0.0050282369007730995, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.0050282369007730995, 0.015377195447811234, 0.010019228860957269, 0.005198952718470577], [0.005595526103283799, 0.015994783595507267, 0.010019228860957269, 0.005198952718470577], [0.009162488128506084, 0.031651269210197744, 0.028093383647777456, 0.006440648740315814], [0.006342041523338799, 0.032544146382856444, 0.03053605427732613, 0.0050504559175926405], [0.006639511780462624, 0.0318964083520743, 0.026663504653936118, 0.005386541893343448], [0.005595526103283799, 0.015939638383512072, 0.0251383346208637, 0.005198952718470577], [0.005191693155763629, 0.028594425119226864, 0.027394321254812903, 0.005198952718470577], [0.006639511780462624, 0.0318964083520743, 0.028093383647777456, 0.005270524867219299], [0.008670391806960494, 0.01602214457495366, 0.02430122300069735, 0.005103439656448715], [0.005015797228865461, 0.027341194645033724, 0.02657491803687519, 0.0051740168192639996], [0.006748681659678107, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.006748681659678107, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.007223666742838813, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173], [0.005595526103283799, 0.01917211114036606, 0.012675983555643415, 0.005552634153488642], [0.005627667836258931, 0.017308423187578328, 0.010330376760911492, 0.007621729576744483], [0.005983352717125751, 0.032544146382856444, 0.03053605427732613, 0.005045615432055871], [0.005015797228865461, 0.027341194645033724, 0.027378148847329316, 0.0051740168192639996], [0.005595526103283799, 0.0158634849759992, 0.010951424083173542, 0.007621729576744483], [0.0057448806469450366, 0.02001074489132886, 0.026611987825694388, 0.006509465603991173]]
    multi_nsga_y = [[(-x[0] * N0LV * 1e-9), x[1]] for x in [[-1.164747118720809, 0.46611404500831094], [-1.247488803792855, 0.47857260536063073], [-1.0268996955879839, 0.502292313178496], [-1.2061936716939847, 0.5128574245232302], [-1.1613485845124736, 0.5248933182283249], [-1.1030152740295338, 0.5271325564099195], [-1.2278952573297461, 0.5559477409209541], [-1.1325872355224247, 0.5605927279020553], [-1.319013217305417, 0.5621764750039568], [-1.3203424802809067, 0.5666623965120964], [-1.344870037772817, 0.5727403675607146], [-1.2970086206059028, 0.5752745446308016], [-1.1333350152206731, 0.6191664751547924], [-1.2354671687399423, 0.6381140217825313], [-1.3260895174351206, 0.6814707697270428], [-1.2228514523447793, 0.7011290409835728], [-1.3368832189412383, 0.7128903186028808], [-1.4172191302542165, 0.7394914535797328], [-1.3882505704792407, 0.7400897767170709], [-1.3516352567610677, 0.7473286829537321], [-1.1614511067593125, 0.4514708589795078], [-1.164747118720809, 0.46611404500831094], [-1.4172191302542165, 0.7394914535797328], [-1.4376025126027727, 0.7665971249094595], [-1.247488803792855, 0.47857260536063073], [-1.4079852340453387, 0.6433521495045543], [-1.344870037772817, 0.5727403675607146], [-1.345505198119182, 0.5837934342930887], [-1.3203424802809067, 0.5666623965120964], [-1.319013217305417, 0.5621764750039568], [-1.3203424802809067, 0.5666623965120964], [-1.3204672909634343, 0.5672037458360422], [-1.0268996955879839, 0.502292313178496], [-1.2970086206059028, 0.5752745446308016], [-1.2278952573297461, 0.5559477409209541], [-1.3301824222254612, 0.5986360683043009], [-1.4162681936948487, 0.7507823396425213], [-1.3882505704792407, 0.7400897767170709], [-1.3882505704792407, 0.7400897767170709], [-1.3368832189412383, 0.7128903186028808], [-1.1614511067593125, 0.4514708589795078], [-1.4172191302542165, 0.7394914535797328], [-1.4376025126027727, 0.7665971249094595], [-1.4376025126027727, 0.7665971249094595], [-1.247488803792855, 0.47857260536063073], [-1.3675657026886365, 0.5174186075481133], [-1.4005547317655267, 0.6129555863859376], [-1.4079852340453387, 0.6433521495045543], [-1.1714674945684687, 0.45402863973636925], [-1.4079852340453387, 0.6433521495045543], [-1.2762149711199609, 0.4862049188484356], [-1.3493886824810026, 0.5123574443621298], [-1.164747118720809, 0.46611404500831094], [-1.3811328563539222, 0.731878572470769], [-1.4162681936948487, 0.7507823396425213], [-1.3966776664439111, 0.7363771818949023], [-1.344870037772817, 0.5727403675607146], [-1.344870037772817, 0.5727403675607146], [-1.3203424802809067, 0.5666623965120964], [-1.319013217305417, 0.5621764750039568], [-1.1614511067593125, 0.4514708589795078], [-1.4376772642934093, 0.7621774600900425], [-1.3675657026886365, 0.5174186075481133], [-1.247488803792855, 0.47857260536063073], [-1.4005547317655267, 0.6129555863859376], [-1.4079852340453387, 0.6433521495045543], [-1.4005547317655267, 0.6129555863859376], [-1.247488803792855, 0.47857260536063073], [-1.3493886824810026, 0.5123574443621298], [-1.239855192333726, 0.4739802689688908], [-1.4079852340453387, 0.6433521495045543], [-1.4262064214528536, 0.6836871755610583], [-1.2762149711199609, 0.4862049188484356], [-1.4079852340453387, 0.6433521495045543], [-1.1714674945684687, 0.45402863973636925], [-1.3825144989953568, 0.5238109573082284], [-1.3493886824810026, 0.5123574443621298], [-1.164747118720809, 0.46611404500831094], [-1.4172191302542165, 0.7394914535797328], [-1.4376025126027727, 0.7665971249094595], [-1.1614511067593125, 0.4514708589795078], [-1.4385668092230506, 0.7487085748744502], [-1.3675657026886365, 0.5174186075481133], [-1.247488803792855, 0.47857260536063073], [-1.4005547317655267, 0.6129555863859376], [-1.239855192333726, 0.4739802689688908], [-1.4079852340453387, 0.6433521495045543], [-1.4079852340453387, 0.6433521495045543], [-1.4005547317655267, 0.6129555863859376], [-1.247488803792855, 0.47857260536063073], [-1.3493886824810026, 0.5123574443621298], [-1.4079852340453387, 0.6433521495045543], [-1.3825144989953568, 0.5238109573082284], [-1.4262064214528536, 0.6836871755610583], [-1.2762149711199609, 0.4862049188484356], [-1.4274262109956217, 0.6848566254723953], [-1.4079852340453387, 0.6433521495045543], [-1.4262064214528536, 0.6836871755610583], [-1.2603736992111754, 0.4819593741057098], [-1.2762149711199609, 0.4862049188484356], [-1.1614511067593125, 0.4514708589795078], [-1.4385668092230506, 0.7487085748744502], [-1.4482023634827195, 0.7690342139300231], [-1.3825144989953568, 0.5238109573082284], [-1.3675657026886365, 0.5174186075481133], [-1.4005547317655267, 0.6129555863859376], [-1.239855192333726, 0.4739802689688908], [-1.408712171767717, 0.655205586217366], [-1.4079852340453387, 0.6433521495045543], [-1.4079852340453387, 0.6433521495045543], [-1.425304904188545, 0.6688012248814976], [-1.4005547317655267, 0.6129555863859376], [-1.3493886824810026, 0.5123574443621298], [-1.4079852340453387, 0.6433521495045543], [-1.4262064214528536, 0.6836871755610583], [-1.4274848137906164, 0.6868335610369367], [-1.252071814893169, 0.4770677587111699], [-1.2762149711199609, 0.4862049188484356], [-1.4274262109956217, 0.6848566254723953], [-1.4079852340453387, 0.6433521495045543], [-1.1579944848511297, 0.45027050392035756], [-1.1614511067593125, 0.4514708589795078], [-1.4385668092230506, 0.7487085748744502], [-1.4482023634827195, 0.7690342139300231], [-1.260353613904907, 0.48380741703611413], [-1.387236767192932, 0.5285042130913551], [-1.408712171767717, 0.655205586217366], [-1.3825144989953568, 0.5238109573082284], [-1.4046417663104027, 0.6282038061835137], [-1.3675657026886365, 0.5174186075481133], [-1.239855192333726, 0.4739802689688908], [-1.4005547317655267, 0.6129555863859376], [-1.4079852340453387, 0.6433521495045543], [-1.4079852340453387, 0.6433521495045543], [-1.252071814893169, 0.4770677587111699], [-1.3493886824810026, 0.5123574443621298], [-1.4005547317655267, 0.6129555863859376], [-1.4079852340453387, 0.6433521495045543], [-1.4005547317655267, 0.6129555863859376], [-1.4262064214528536, 0.6836871755610583], [-1.15694879361813, 0.445828955504169], [-1.1579944848511297, 0.45027050392035756], [-1.1614511067593125, 0.4514708589795078], [-1.4385668092230506, 0.7487085748744502], [-1.4482023634827195, 0.7690342139300231], [-1.387236767192932, 0.5285042130913551], [-1.3675657026886365, 0.5174186075481133], [-1.3825144989953568, 0.5238109573082284], [-1.4046417663104027, 0.6282038061835137], [-1.260353613904907, 0.48380741703611413], [-1.3675657026886365, 0.5174186075481133], [-1.408712171767717, 0.655205586217366], [-1.3825144989953568, 0.5238109573082284], [-1.239855192333726, 0.4739802689688908], [-1.4005547317655267, 0.6129555863859376], [-1.252071814893169, 0.4770677587111699], [-1.4262064214528536, 0.6836871755610583], [-1.4005547317655267, 0.6129555863859376], [-1.4005547317655267, 0.6129555863859376], [-1.4005547317655267, 0.6129555863859376], [-1.15694879361813, 0.445828955504169], [-1.4385668092230506, 0.7487085748744502], [-1.4482023634827195, 0.7690342139300231], [-1.387236767192932, 0.5285042130913551], [-1.408712171767717, 0.655205586217366], [-1.3675657026886365, 0.5174186075481133], [-1.3825144989953568, 0.5238109573082284], [-1.2412416346367066, 0.4702572552664096], [-1.260353613904907, 0.48380741703611413], [-1.3675657026886365, 0.5174186075481133], [-1.260353613904907, 0.48380741703611413], [-1.4262064214528536, 0.6836871755610583], [-1.3825144989953568, 0.5238109573082284], [-1.3970815221847066, 0.534197616954201], [-1.4064622864791352, 0.6268455791975437], [-1.1680632535211664, 0.4464289110834171], [-1.3850675612162369, 0.5265727125403853], [-1.382934716156623, 0.5240333548661124], [-1.3825151375610067, 0.5238130416927174], [-1.3825144989953568, 0.5238109573082284], [-1.15694879361813, 0.445828955504169], [-1.4482023634827195, 0.7690342139300231], [-1.4482023634827195, 0.7690342139300231], [-1.387236767192932, 0.5285042130913551], [-1.4056355933553408, 0.5501276464786788], [-1.2412416346367066, 0.4702572552664096], [-1.3675657026886365, 0.5174186075481133], [-1.260353613904907, 0.48380741703611413], [-1.3970815221847066, 0.534197616954201], [-1.3970815221847066, 0.534197616954201], [-1.25720349877863, 0.48299817855323646], [-1.3675657026886365, 0.5174186075481133], [-1.231892918127118, 0.46453870658181917], [-1.3850675612162369, 0.5265727125403853], [-1.260353613904907, 0.48380741703611413], [-1.3853516734128908, 0.5267764456431637], [-1.3675657026886365, 0.5174186075481133], [-1.3837083431798283, 0.5222643140486318], [-1.4468153947973201, 0.6089008541745258], [-1.3686151131162387, 0.5181884420771876], [-1.15694879361813, 0.445828955504169], [-1.4482023634827195, 0.7690342139300231], [-1.4482023634827195, 0.7690342139300231], [-1.448257685775023, 0.7894192021721472], [-1.4468153947973201, 0.6089008541745258], [-1.25720349877863, 0.48299817855323646], [-1.387236767192932, 0.5285042130913551], [-1.4056355933553408, 0.5501276464786788], [-1.4002711725570938, 0.5403450518319243], [-1.3970815221847066, 0.534197616954201], [-1.231892918127118, 0.46453870658181917], [-1.2412416346367066, 0.4702572552664096], [-1.3675657026886365, 0.5174186075481133], [-1.260353613904907, 0.48380741703611413], [-1.4482023634827195, 0.7690342139300231], [-1.3857150423975622, 0.5270403379594434], [-1.4468153947973201, 0.6089008541745258], [-1.3850675612162369, 0.5265727125403853], [-1.3675657026886365, 0.5174186075481133], [-1.260353613904907, 0.48380741703611413], [-1.15694879361813, 0.445828955504169], [-1.4482023634827195, 0.7690342139300231], [-1.4482023634827195, 0.7690342139300231], [-1.454007316555517, 0.7743150040448582], [-1.447679785398125, 0.6105893549384375], [-1.25720349877863, 0.48299817855323646], [-1.4002711725570938, 0.5403450518319243], [-1.4482023634827195, 0.7690342139300231], [-1.3970815221847066, 0.534197616954201], [-1.2412416346367066, 0.4702572552664096], [-1.3857150423975622, 0.5270403379594434], [-1.4473693847453792, 0.6017823637205721], [-1.231892918127118, 0.46453870658181917], [-1.387236767192932, 0.5285042130913551], [-1.4331157153760241, 0.5764255974484144], [-1.2412416346367066, 0.4702572552664096], [-1.4006717975226572, 0.542762171487356], [-1.3890464682245909, 0.5295574009001751], [-1.3675657026886365, 0.5174186075481133], [-1.260353613904907, 0.48380741703611413], [-1.15694879361813, 0.445828955504169], [-1.4547568035758671, 0.765096840248102], [-1.4399551919484643, 0.5765042754206209], [-1.447679785398125, 0.6105893549384375], [-1.447679785398125, 0.6105893549384375], [-1.25720349877863, 0.48299817855323646], [-1.2412416346367066, 0.4702572552664096], [-1.2412416346367066, 0.4702572552664096], [-1.231892918127118, 0.46453870658181917], [-1.4331157153760241, 0.5764255974484144], [-1.3970815221847066, 0.534197616954201], [-1.2608176400130318, 0.48563033572728637], [-1.3980601102334684, 0.5388043075872061], [-1.3761732525540853, 0.5219991888079151], [-1.4006717975226572, 0.542762171487356], [-1.260353613904907, 0.48380741703611413], [-1.400520556061887, 0.5402625992209454], [-1.3890464682245909, 0.5295574009001751], [-1.3970815221847066, 0.534197616954201], [-1.368954869395652, 0.5174883826217295], [-1.1526892222565694, 0.43346233617110647], [-1.15694879361813, 0.445828955504169], [-1.4547568035758671, 0.765096840248102], [-1.455337660507204, 0.7675391023588642], [-1.447679785398125, 0.6105893549384375], [-1.447679785398125, 0.6105893549384375], [-1.4006717975226572, 0.542762171487356], [-1.447679785398125, 0.6105893549384375], [-1.2412416346367066, 0.4702572552664096], [-1.231892918127118, 0.46453870658181917], [-1.2412416346367066, 0.4702572552664096], [-1.2608176400130318, 0.48563033572728637], [-1.3761732525540853, 0.5219991888079151], [-1.3979673946074076, 0.5378660178286373], [-1.3970815221847066, 0.534197616954201], [-1.3980601102334684, 0.5388043075872061], [-1.2592577255858435, 0.48285524281001757], [-1.402366626657161, 0.5674840708963654], [-1.4473715998702659, 0.6087252567767171], [-1.3970815221847066, 0.534197616954201], [-1.1526892222565694, 0.43346233617110647], [-1.15694879361813, 0.445828955504169], [-1.4547568035758671, 0.765096840248102], [-1.455337660507204, 0.7675391023588642], [-1.455337660507204, 0.7675391023588642], [-1.4489620741455407, 0.6394334046485775], [-1.259323772415009, 0.4832293577345385], [-1.447679785398125, 0.6105893549384375], [-1.2412416346367066, 0.4702572552664096], [-1.447679785398125, 0.6105893549384375], [-1.447679785398125, 0.6105893549384375], [-1.2412416346367066, 0.4702572552664096], [-1.231892918127118, 0.46453870658181917], [-1.4006717975226572, 0.542762171487356], [-1.3970815221847066, 0.534197616954201], [-1.4006717975226572, 0.542762171487356], [-1.447679785398125, 0.6105893549384375], [-1.2592577255858435, 0.48285524281001757], [-1.3970815221847066, 0.534197616954201], [-1.4473715998702659, 0.6087252567767171], [-1.1526892222565694, 0.43346233617110647], [-1.15694879361813, 0.445828955504169], [-1.4547568035758671, 0.765096840248102], [-1.455337660507204, 0.7675391023588642], [-1.455337660507204, 0.7675391023588642], [-1.460712139466406, 0.7845109230880851], [-1.4539549969889936, 0.764079534683588], [-1.3970815221847066, 0.534197616954201], [-1.4507508310917885, 0.6144927168875075], [-1.259323772415009, 0.4832293577345385], [-1.447679785398125, 0.6105893549384375], [-1.447679785398125, 0.6105893549384375], [-1.2412416346367066, 0.4702572552664096], [-1.231892918127118, 0.46453870658181917], [-1.2412416346367066, 0.4702572552664096], [-1.447679785398125, 0.6105893549384375], [-1.2384875569394873, 0.46894474659557317], [-1.447679785398125, 0.6105893549384375], [-1.154323289386467, 0.4339548052092086], [-1.3191670465238248, 0.5131371227986213], [-1.1526892222565694, 0.43346233617110647], [-1.15694879361813, 0.445828955504169], [-1.460712139466406, 0.7845109230880851], [-1.435801378002047, 0.5984379055706345], [-1.447679785398125, 0.6105893549384375], [-1.4507508310917885, 0.6144927168875075], [-1.3970815221847066, 0.534197616954201], [-1.3970815221847066, 0.534197616954201], [-1.4597395868318794, 0.7722765640862461], [-1.3191670465238248, 0.5131371227986213], [-1.2329440136808374, 0.46520241919884137], [-1.2412416346367066, 0.4702572552664096], [-1.154323289386467, 0.4339548052092086], [-1.259323772415009, 0.4832293577345385], [-1.447679785398125, 0.6105893549384375], [-1.447679785398125, 0.6105893549384375], [-1.2412416346367066, 0.4702572552664096], [-1.2532495065675362, 0.47839927397776627], [-1.447679785398125, 0.6105893549384375], [-1.231892918127118, 0.46453870658181917], [-1.1358701449637882, 0.4228630155024416], [-1.1526892222565694, 0.43346233617110647], [-1.15694879361813, 0.445828955504169], [-1.460712139466406, 0.7845109230880851], [-1.4597395868318794, 0.7722765640862461], [-1.4507508310917885, 0.6144927168875075], [-1.435801378002047, 0.5984379055706345], [-1.447679785398125, 0.6105893549384375], [-1.3191670465238248, 0.5131371227986213], [-1.231892918127118, 0.46453870658181917], [-1.2412416346367066, 0.4702572552664096], [-1.3970815221847066, 0.534197616954201], [-1.154323289386467, 0.4339548052092086], [-1.269064824935693, 0.4801147311570932], [-1.396986504043963, 0.5292120805193679], [-1.3970815221847066, 0.534197616954201], [-1.4242527110943703, 0.5499403809251503], [-1.2522087465634533, 0.47716519819389097], [-1.2329440136808374, 0.46520241919884137], [-1.447679785398125, 0.6105893549384375], [-1.1358701449637882, 0.4228630155024416], [-1.460712139466406, 0.7845109230880851], [-1.4507508310917885, 0.6144927168875075], [-1.3191670465238248, 0.5131371227986213], [-1.4597395868318794, 0.7722765640862461], [-1.447679785398125, 0.6105893549384375], [-1.396986504043963, 0.5292120805193679], [-1.435801378002047, 0.5984379055706345], [-1.269064824935693, 0.4801147311570932], [-1.2329440136808374, 0.46520241919884137], [-1.447679785398125, 0.6105893549384375], [-1.231892918127118, 0.46453870658181917], [-1.2329440136808374, 0.46520241919884137], [-1.2470619088124326, 0.4672069907520127], [-1.2726801637398435, 0.4870960247704317], [-1.3970815221847066, 0.534197616954201], [-1.1553648816579642, 0.42995427113165163], [-1.4597395868318794, 0.7722765640862461], [-1.2546493459958317, 0.4740459252587026], [-1.4242527110943703, 0.5499403809251503], [-1.1358701449637882, 0.4228630155024416], [-1.4617320640100544, 0.767892439379572], [-1.4507508310917885, 0.6144927168875075], [-1.4507508310917885, 0.6144927168875075], [-1.3191670465238248, 0.5131371227986213], [-1.435801378002047, 0.5984379055706345], [-1.2329440136808374, 0.46520241919884137], [-1.2726801637398435, 0.4870960247704317], [-1.435801378002047, 0.5984379055706345], [-1.1553648816579642, 0.42995427113165163], [-1.2443925841829422, 0.4669392760282682], [-1.391979369681459, 0.5268140280861031], [-1.447679785398125, 0.6105893549384375], [-1.396986504043963, 0.5292120805193679], [-1.447679785398125, 0.6105893549384375], [-1.2329440136808374, 0.46520241919884137], [-1.447679785398125, 0.6105893549384375], [-1.4242527110943703, 0.5499403809251503], [-1.2286563657537781, 0.45611086069481216], [-1.3970815221847066, 0.534197616954201], [-1.1049779710667047, 0.41840821599395744], [-1.1358701449637882, 0.4228630155024416], [-1.4617320640100544, 0.767892439379572], [-1.4645586761321363, 0.7726044862061713], [-1.4530016117511162, 0.6244876006404031], [-1.4507508310917885, 0.6144927168875075], [-1.4507508310917885, 0.6144927168875075], [-1.391979369681459, 0.5268140280861031], [-1.1553648816579642, 0.42995427113165163], [-1.2726801637398435, 0.4870960247704317], [-1.4617320640100544, 0.767892439379572], [-1.2329440136808374, 0.46520241919884137], [-1.3007172499239872, 0.5022063721687686], [-1.2286563657537781, 0.45611086069481216], [-1.447679785398125, 0.6105893549384375], [-1.435801378002047, 0.5984379055706345], [-1.435801378002047, 0.5984379055706345], [-1.2443925841829422, 0.4669392760282682], [-1.3970815221847066, 0.534197616954201], [-1.440727690942105, 0.6048769477923133], [-1.1049779710667047, 0.41840821599395744], [-1.1049779710667047, 0.41840821599395744], [-1.1358701449637882, 0.4228630155024416], [-1.4617320640100544, 0.767892439379572], [-1.4645586761321363, 0.7726044862061713], [-1.4606659482955846, 0.7662643479567087], [-1.4591231432903362, 0.576640943950948], [-1.1553648816579642, 0.42995427113165163], [-1.2329440136808374, 0.46520241919884137], [-1.3007172499239872, 0.5022063721687686], [-1.2286563657537781, 0.45611086069481216], [-1.3970815221847066, 0.534197616954201], [-1.4617320640100544, 0.767892439379572], [-1.3458400690511887, 0.503544760282753], [-1.1551521350754776, 0.42922809060684786], [-1.391979369681459, 0.5268140280861031], [-1.4617320640100544, 0.767892439379572], [-1.107304982306546, 0.4225545937528842], [-1.2443925841829422, 0.4669392760282682], [-1.3007172499239872, 0.5022063721687686], [-1.0976096364905312, 0.4036707079861922], [-1.1049779710667047, 0.41840821599395744], [-1.1049779710667047, 0.41840821599395744], [-1.1358701449637882, 0.4228630155024416], [-1.4617320640100544, 0.767892439379572], [-1.4645586761321363, 0.7726044862061713], [-1.4606659482955846, 0.7662643479567087], [-1.4591231432903362, 0.576640943950948], [-1.3000742333769435, 0.5002213988675384], [-1.1553648816579642, 0.42995427113165163], [-1.4591231432903362, 0.576640943950948], [-1.2329440136808374, 0.46520241919884137], [-1.3458400690511887, 0.503544760282753], [-1.3972974215176028, 0.5350917596226195], [-1.4432475571803516, 0.5607698005743105], [-1.3007172499239872, 0.5022063721687686], [-1.4606659482955846, 0.7662643479567087], [-1.2282308245247329, 0.45498604612024257], [-1.1551521350754776, 0.42922809060684786], [-1.3970815221847066, 0.534197616954201], [-1.0976096364905312, 0.4036707079861922], [-1.1049779710667047, 0.41840821599395744], [-1.1049779710667047, 0.41840821599395744], [-1.1358701449637882, 0.4228630155024416], [-1.4649877882117017, 0.766262506576986], [-1.4591231432903362, 0.576640943950948], [-1.4615647646510972, 0.7474501294573122], [-1.2329440136808374, 0.46520241919884137], [-1.3970815221847066, 0.534197616954201], [-1.4432475571803516, 0.5607698005743105], [-1.1553648816579642, 0.42995427113165163], [-1.2282308245247329, 0.45498604612024257], [-1.4591231432903362, 0.576640943950948], [-1.3972974215176028, 0.5350917596226195], [-1.3972974215176028, 0.5350917596226195], [-1.3370334922183431, 0.499208211388305], [-1.3458400690511887, 0.503544760282753], [-1.1551521350754776, 0.42922809060684786], [-1.4610742393957705, 0.5797329201655924], [-1.3970852877281918, 0.5342126449214631], [-1.0976096364905312, 0.4036707079861922], [-1.0976096364905312, 0.4036707079861922], [-1.1049779710667047, 0.41840821599395744], [-1.1049779710667047, 0.41840821599395744], [-1.1358701449637882, 0.4228630155024416], [-1.4649877882117017, 0.766262506576986], [-1.465829818384037, 0.7693597557981479], [-1.4615647646510972, 0.7474501294573122], [-1.4591231432903362, 0.576640943950948], [-1.4615647646510972, 0.7474501294573122], [-1.2329440136808374, 0.46520241919884137], [-1.3970815221847066, 0.534197616954201], [-1.4610742393957705, 0.5797329201655924], [-1.4432475571803516, 0.5607698005743105], [-1.2311017940707962, 0.4518236168926855], [-1.400195201297545, 0.5374997660997722], [-1.4591231432903362, 0.576640943950948], [-1.3494687671892458, 0.5057641000824181], [-1.3458400690511887, 0.503544760282753], [-1.2039739025353948, 0.4413469685367676], [-1.097665908796641, 0.40361647192658684], [-1.1049779710667047, 0.41840821599395744], [-1.1049779710667047, 0.41840821599395744], [-1.1358701449637882, 0.4228630155024416], [-1.4649877882117017, 0.766262506576986], [-1.465829818384037, 0.7693597557981479], [-1.469670176934978, 0.7838898270286523], [-1.4615647646510972, 0.7474501294573122], [-1.4610742393957705, 0.5797329201655924], [-1.4615647646510972, 0.7474501294573122], [-1.3494687671892458, 0.5057641000824181], [-1.4591231432903362, 0.576640943950948], [-1.4591231432903362, 0.576640943950948], [-1.3970815221847066, 0.534197616954201], [-1.4610742393957705, 0.5797329201655924], [-1.4432475571803516, 0.5607698005743105], [-1.4615647646510972, 0.7474501294573122], [-1.4016887188319367, 0.5366921891416486], [-1.2329440136808374, 0.46520241919884137], [-1.272372953062656, 0.4779753566862755], [-1.097665908796641, 0.40361647192658684], [-1.1049779710667047, 0.41840821599395744], [-1.1049779710667047, 0.41840821599395744], [-1.1358701449637882, 0.4228630155024416], [-1.470003285518649, 0.7731404162015014], [-1.4612858698950095, 0.652587695812399], [-1.4591231432903362, 0.576640943950948], [-1.4610823287182242, 0.5793899423999257], [-1.4615647646510972, 0.7474501294573122], [-1.4432475571803516, 0.5607698005743105], [-1.2329440136808374, 0.46520241919884137], [-1.3970815221847066, 0.534197616954201], [-1.3494687671892458, 0.5057641000824181], [-1.4615647646510972, 0.7474501294573122], [-1.4016887188319367, 0.5366921891416486], [-1.4615647646510972, 0.7474501294573122], [-1.3437962972956552, 0.5042618871517688], [-1.4591231432903362, 0.576640943950948], [-1.4016887188319367, 0.5366921891416486], [-1.4615647646510972, 0.7474501294573122], [-1.097665908796641, 0.40361647192658684], [-1.1358701449637882, 0.4228630155024416], [-1.470003285518649, 0.7731404162015014], [-1.4612858698950095, 0.652587695812399], [-1.4612551719354625, 0.5828440782704758], [-1.2329440136808374, 0.46520241919884137], [-1.3970815221847066, 0.534197616954201], [-1.4432475571803516, 0.5607698005743105], [-1.4591231432903362, 0.576640943950948], [-1.4610823287182242, 0.5793899423999257], [-1.401907788944277, 0.5375643620123312], [-1.4622617096213848, 0.7404534787392707], [-1.4016887188319367, 0.5366921891416486], [-1.4591231432903362, 0.576640943950948], [-1.1428569141375782, 0.4259875112094893], [-1.2201808084398664, 0.46133388256448565], [-1.4016887188319367, 0.5366921891416486], [-1.3437962972956552, 0.5042618871517688], [-1.3437962972956552, 0.5042618871517688], [-1.4610823287182242, 0.5793899423999257], [-1.097665908796641, 0.40361647192658684], [-1.1358701449637882, 0.4228630155024416], [-1.470003285518649, 0.7731404162015014], [-1.470003285518649, 0.7731404162015014], [-1.404197980921613, 0.5500500082230926], [-1.4622617096213848, 0.7404534787392707], [-1.2329440136808374, 0.46520241919884137], [-1.4420777796426356, 0.5598198310369173], [-1.3970815221847066, 0.534197616954201], [-1.401907788944277, 0.5375643620123312], [-1.4432475571803516, 0.5607698005743105], [-1.3650440467661118, 0.5178780609979738], [-1.401907788944277, 0.5375643620123312], [-1.4622617096213848, 0.7404534787392707], [-1.1428569141375782, 0.4259875112094893], [-1.3521279002842708, 0.49802911613150536], [-1.4432475571803516, 0.5607698005743105], [-1.218080911126794, 0.4604823691221135], [-1.4593684032894867, 0.5685276245261229], [-1.4016887188319367, 0.5366921891416486], [-1.097665908796641, 0.40361647192658684], [-1.097665908796641, 0.40361647192658684], [-1.1358701449637882, 0.4228630155024416], [-1.470003285518649, 0.7731404162015014], [-1.470003285518649, 0.7731404162015014], [-1.470003285518649, 0.7731404162015014], [-1.4593684032894867, 0.5685276245261229], [-1.2329440136808374, 0.46520241919884137], [-1.404197980921613, 0.5500500082230926], [-1.3521279002842708, 0.49802911613150536], [-1.3970815221847066, 0.534197616954201], [-1.3502645436920238, 0.49091892014378885], [-1.1428569141375782, 0.4259875112094893], [-1.4419327200384568, 0.5597100621911506], [-1.4420777796426356, 0.5598198310369173], [-1.4041656341641655, 0.5388057182412473], [-1.401907788944277, 0.5375643620123312], [-1.4432475571803516, 0.5607698005743105], [-1.401907788944277, 0.5375643620123312], [-1.1858037638169967, 0.44065503204843787], [-1.097665908796641, 0.40361647192658684], [-1.097665908796641, 0.40361647192658684], [-1.097665908796641, 0.40361647192658684], [-1.1358701449637882, 0.4228630155024416], [-1.470003285518649, 0.7731404162015014], [-1.470003285518649, 0.7731404162015014], [-1.470003285518649, 0.7731404162015014], [-1.470003285518649, 0.7731404162015014], [-1.4682704020065411, 0.5893616237494839], [-1.4041656341641655, 0.5388057182412473], [-1.404197980921613, 0.5500500082230926], [-1.4593684032894867, 0.5685276245261229], [-1.3502645436920238, 0.49091892014378885], [-1.2329440136808374, 0.46520241919884137], [-1.4419327200384568, 0.5597100621911506], [-1.1543149304457763, 0.42884483297154175], [-1.4420777796426356, 0.5598198310369173], [-1.1428569141375782, 0.4259875112094893], [-1.404197980921613, 0.5500500082230926], [-1.1858037638169967, 0.44065503204843787], [-1.098022326588332, 0.40311329833922843], [-10, 100], [-1.2329440136808374, 0.46520241919884137], [-1.4700197515825624, 0.7689578201196334], [-1.3638255755565132, 0.500023445201446], [-1.1858037638169967, 0.44065503204843787], [-1.4041656341641655, 0.5388057182412473], [-1.3502645436920238, 0.49091892014378885], [-1.4419327200384568, 0.5597100621911506], [-1.4702310060525667, 0.7699219004433043], [-1.3502645436920238, 0.49091892014378885], [-1.1860803466773946, 0.4406661345107496], [-1.1428569141375782, 0.4259875112094893], [-1.1358701449637882, 0.4228630155024416], [-1.1578970758708604, 0.4287928060223426], [-1.439195925214304, 0.5440912450475734], [-1.4419327200384568, 0.5597100621911506], [-1.4420777796426356, 0.5598198310369173], [-1.4420777796426356, 0.5598198310369173], [-1.4593705777433919, 0.5683737436610892], [-1.098022326588332, 0.40311329833922843], [-10, 100], [-1.4652200917799203, 0.753370794445556], [-1.4600568556218463, 0.5668250852187532], [-1.4041656341641655, 0.5388057182412473], [-1.2329440136808374, 0.46520241919884137], [-1.1858037638169967, 0.44065503204843787], [-1.4420777796426356, 0.5598198310369173], [-1.1718836355144717, 0.43000104141519624], [-1.3502645436920238, 0.49091892014378885], [-1.4700197515825624, 0.7689578201196334], [-1.4700197515825624, 0.7689578201196334], [-1.3502645436920238, 0.49091892014378885], [-1.1578970758708604, 0.4287928060223426], [-1.4420777796426356, 0.5598198310369173], [-1.4420777796426356, 0.5598198310369173], [-1.4419822470254764, 0.5467587483097218], [-1.439195925214304, 0.5440912450475734], [-1.1860803466773946, 0.4406661345107496], [-1.1578970758708604, 0.4287928060223426], [-1.098022326588332, 0.40311329833922843], [-10, 100], [-1.4612739855074506, 0.5925914212308453], [-1.4615146944534996, 0.7433493358988065], [-1.4652200917799203, 0.753370794445556], [-1.4600568556218463, 0.5668250852187532], [-1.413237695850086, 0.5286036092589589], [-1.3502645436920238, 0.49091892014378885], [-1.1858037638169967, 0.44065503204843787], [-1.4420777796426356, 0.5598198310369173], [-1.1578970758708604, 0.4287928060223426], [-1.2329440136808374, 0.46520241919884137], [-1.1860803466773946, 0.4406661345107496], [-1.3502645436920238, 0.49091892014378885], [-1.2337369836766368, 0.46648806835090467], [-1.190318265109093, 0.4416152892570175], [-1.3502645436920238, 0.49091892014378885], [-1.4700197515825624, 0.7689578201196334], [-1.4652200917799203, 0.753370794445556], [-1.1578970758708604, 0.4287928060223426], [-1.098022326588332, 0.40311329833922843], [-10, 100], [-1.4612739855074506, 0.5925914212308453], [-1.4615146944534996, 0.7433493358988065], [-1.3711257463900972, 0.5057081837974337], [-1.4600568556218463, 0.5668250852187532], [-1.2344636506569786, 0.4667779077016581], [-1.1805009155094464, 0.4285520883680098], [-1.413237695850086, 0.5286036092589589], [-1.1858037638169967, 0.44065503204843787], [-1.3502645436920238, 0.49091892014378885], [-1.2430810827755097, 0.4848378450428089], [-1.2329440136808374, 0.46520241919884137], [-1.4420777796426356, 0.5598198310369173], [-1.1860803466773946, 0.4406661345107496], [-1.3502645436920238, 0.49091892014378885], [-1.413071653052006, 0.5255313323868802], [-1.1858037638169967, 0.44065503204843787], [-1.2337369836766368, 0.46648806835090467], [-1.4656834302128978, 0.7460345483957368], [-1.098022326588332, 0.40311329833922843], [-10, 100], [-1.4656834302128978, 0.7460345483957368], [-1.2416391359048284, 0.4646249342180273], [-1.3711257463900972, 0.5057081837974337], [-1.413237695850086, 0.5286036092589589], [-1.1860803466773946, 0.4406661345107496], [-1.3502645436920238, 0.49091892014378885], [-1.1858037638169967, 0.44065503204843787], [-1.3502645436920238, 0.49091892014378885], [-1.460097096602103, 0.5653194436923115], [-1.1858037638169967, 0.44065503204843787], [-1.413071653052006, 0.5255313323868802], [-1.1805009155094464, 0.4285520883680098], [-1.461830185478351, 0.5720998057467068], [-1.4642246158147192, 0.573817240235129], [-1.186770361057228, 0.4445584054631476], [-1.3502645436920238, 0.49091892014378885], [-1.413237695850086, 0.5286036092589589], [-1.258902993521303, 0.4732833747544728], [-1.098022326588332, 0.40311329833922843], [-10, 100], [-1.4652312870717983, 0.7389884952860336], [-1.4642246158147192, 0.573817240235129], [-1.4656834302128978, 0.7460345483957368], [-1.2416391359048284, 0.4646249342180273], [-1.413071653052006, 0.5255313323868802], [-1.3888370999326876, 0.5078397124805725], [-1.3711257463900972, 0.5057081837974337], [-1.186770361057228, 0.4445584054631476], [-1.1860803466773946, 0.4406661345107496], [-1.413237695850086, 0.5286036092589589], [-1.1858037638169967, 0.44065503204843787], [-1.460097096602103, 0.5653194436923115], [-1.1858037638169967, 0.44065503204843787], [-1.258902993521303, 0.4732833747544728], [-1.461830185478351, 0.5720998057467068], [-1.413237695850086, 0.5286036092589589], [-1.1022617625705629, 0.4111724951617836], [-1.413237695850086, 0.5286036092589589], [-1.098022326588332, 0.40311329833922843], [-10, 100], [-1.4642246158147192, 0.573817240235129], [-1.413071653052006, 0.5255313323868802], [-1.3888370999326876, 0.5078397124805725], [-1.2723922953306193, 0.47900784030407284], [-1.2416391359048284, 0.4646249342180273], [-1.186770361057228, 0.4445584054631476], [-1.1860803466773946, 0.4406661345107496], [-1.2416391359048284, 0.4646249342180273], [-1.413237695850086, 0.5286036092589589], [-1.413237695850086, 0.5286036092589589], [-1.1858037638169967, 0.44065503204843787], [-1.460097096602103, 0.5653194436923115], [-1.461830185478351, 0.5720998057467068], [-1.413237695850086, 0.5286036092589589], [-1.375779346225734, 0.5003814916115482], [-1.0985198210616993, 0.4040813643548153], [-1.098546645106189, 0.40413387836755243], [-1.3886639484721575, 0.5073817086221525], [-1.098022326588332, 0.40311329833922843], [-10, 100], [-1.4642246158147192, 0.573817240235129], [-1.2416391359048284, 0.4646249342180273], [-1.3888370999326876, 0.5078397124805725], [-1.3905910815090996, 0.5190349838193987], [-1.4090624188827479, 0.5235293422554911], [-1.413071653052006, 0.5255313323868802], [-1.1860803466773946, 0.4406661345107496], [-1.186770361057228, 0.4445584054631476], [-1.460097096602103, 0.5653194436923115], [-1.413237695850086, 0.5286036092589589], [-1.2416391359048284, 0.4646249342180273], [-1.413071653052006, 0.5255313323868802], [-1.4603074027471756, 0.5684254176392424], [-1.413237695850086, 0.5286036092589589], [-1.3886639484721575, 0.5073817086221525], [-1.1882397082554168, 0.44561739023377533], [-1.0985198210616993, 0.4040813643548153], [-1.2034888507471329, 0.44750825787955556], [-1.098022326588332, 0.40311329833922843], [-10, 100], [-1.2416391359048284, 0.4646249342180273], [-1.4642246158147192, 0.573817240235129], [-1.2420676209254171, 0.48279770328024], [-1.460097096602103, 0.5653194436923115], [-1.3905910815090996, 0.5190349838193987], [-1.3919509651631081, 0.5196028483434431], [-1.3888370999326876, 0.5078397124805725], [-1.186770361057228, 0.4445584054631476], [-1.413071653052006, 0.5255313323868802], [-1.4090624188827479, 0.5235293422554911], [-1.4064797446166217, 0.5208498272015343], [-1.4603074027471756, 0.5684254176392424], [-1.3897491664181592, 0.5165103237433776], [-1.186196446567517, 0.44426723145526986], [-1.4674542051490322, 0.583700852122823], [-1.2416391359048284, 0.4646249342180273], [-1.413237695850086, 0.5286036092589589], [-1.0985198210616993, 0.4040813643548153], [-1.0455841917580444, 0.382937058406747], [-1.098022326588332, 0.40311329833922843], [-10, 100], [-1.2420676209254171, 0.48279770328024], [-1.2416391359048284, 0.4646249342180273], [-1.4642246158147192, 0.573817240235129], [-1.3905910815090996, 0.5190349838193987], [-1.460097096602103, 0.5653194436923115], [-1.3919509651631081, 0.5196028483434431], [-1.3888370999326876, 0.5078397124805725], [-1.4064797446166217, 0.5208498272015343], [-1.186770361057228, 0.4445584054631476], [-1.413071653052006, 0.5255313323868802], [-1.3897491664181592, 0.5165103237433776], [-1.186196446567517, 0.44426723145526986], [-1.4603074027471756, 0.5684254176392424], [-1.0985198210616993, 0.4040813643548153], [-1.2416391359048284, 0.4646249342180273], [-1.0849100663136064, 0.39909762194084836], [-1.4674542051490322, 0.583700852122823], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-1.098022326588332, 0.40311329833922843], [-10, 100], [-1.4642246158147192, 0.573817240235129], [-1.2416391359048284, 0.4646249342180273], [-1.3905910815090996, 0.5190349838193987], [-1.3919509651631081, 0.5196028483434431], [-1.0849100663136064, 0.39909762194084836], [-1.3888370999326876, 0.5078397124805725], [-1.2420676209254171, 0.48279770328024], [-1.3897491664181592, 0.5165103237433776], [-1.4674542051490322, 0.583700852122823], [-1.413071653052006, 0.5255313323868802], [-1.46803226841374, 0.5886216461253598], [-1.186196446567517, 0.44426723145526986], [-1.0985198210616993, 0.4040813643548153], [-1.4674542051490322, 0.583700852122823], [-1.1861760923107632, 0.444143376640262], [-1.0977282374711526, 0.4030562354343325], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-1.098022326588332, 0.40311329833922843], [-10, 100], [-1.4642246158147192, 0.573817240235129], [-1.3905910815090996, 0.5190349838193987], [-1.0849100663136064, 0.39909762194084836], [-1.2416391359048284, 0.4646249342180273], [-1.3919509651631081, 0.5196028483434431], [-1.2420676209254171, 0.48279770328024], [-1.3888370999326876, 0.5078397124805725], [-1.3897491664181592, 0.5165103237433776], [-1.413071653052006, 0.5255313323868802], [-1.4642246158147192, 0.573817240235129], [-1.1028323463441119, 0.4122471480787993], [-1.3919509651631081, 0.5196028483434431], [-1.0985198210616993, 0.4040813643548153], [-1.3900259496569995, 0.5166656367260642], [-1.413071653052006, 0.5255313323868802], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-1.098022326588332, 0.40311329833922843], [-10, 100], [-1.3905910815090996, 0.5190349838193987], [-1.4642246158147192, 0.573817240235129], [-1.3919509651631081, 0.5196028483434431], [-1.3897491664181592, 0.5165103237433776], [-1.413071653052006, 0.5255313323868802], [-1.0473706461173196, 0.38684563700629154], [-1.3919509651631081, 0.5196028483434431], [-1.413071653052006, 0.5255313323868802], [-1.256329259450963, 0.44446603942140717], [-1.4642246158147192, 0.573817240235129], [-1.095044027383473, 0.393636576365465], [-1.390098904770752, 0.5175015997814054], [-1.4642246158147192, 0.573817240235129], [-1.3900259496569995, 0.5166656367260642], [-1.3888370999326876, 0.5078397124805725], [-0.9540288225056899, 0.3684942105919057], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-1.098022326588332, 0.40311329833922843], [-10, 100], [-1.256329259450963, 0.44446603942140717], [-1.4642246158147192, 0.573817240235129], [-1.3897491664181592, 0.5165103237433776], [-1.095044027383473, 0.393636576365465], [-1.3919509651631081, 0.5196028483434431], [-1.413071653052006, 0.5255313323868802], [-1.0473706461173196, 0.38684563700629154], [-1.390696982027019, 0.5189210896329557], [-1.2709560472217643, 0.4505208309351606], [-1.3919509651631081, 0.5196028483434431], [-1.390098904770752, 0.5175015997814054], [-1.4224421673638956, 0.5526570770033263], [-1.4130695989436457, 0.5255080959855084], [-1.0455841917580444, 0.382937058406747], [-0.9539344596568613, 0.3680822513306155], [-0.9540288225056899, 0.3684942105919057], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-1.098022326588332, 0.40311329833922843], [-10, 100], [-1.2929389096436823, 0.46711348969254857], [-1.095044027383473, 0.393636576365465], [-1.390696982027019, 0.5189210896329557], [-1.2783595552855107, 0.4489028824719865], [-1.3901018396174984, 0.517510149680716], [-1.4642246158147192, 0.573817240235129], [-1.3919509651631081, 0.5196028483434431], [-1.0473706461173196, 0.38684563700629154], [-1.256329259450963, 0.44446603942140717], [-1.4130695989436457, 0.5255080959855084], [-1.413071653052006, 0.5255313323868802], [-1.4130695989436457, 0.5255080959855084], [-1.3919509651631081, 0.5196028483434431], [-0.9539344596568613, 0.3680822513306155], [-0.9540288225056899, 0.3684942105919057], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-1.098022326588332, 0.40311329833922843], [-10, 100], [-1.2929389096436823, 0.46711348969254857], [-1.4130695989436457, 0.5255080959855084], [-1.0473706461173196, 0.38684563700629154], [-1.2789591912869407, 0.4497352935391749], [-1.3909396837521038, 0.5015541215991208], [-1.095044027383473, 0.393636576365465], [-1.2783595552855107, 0.4489028824719865], [-1.4642246158147192, 0.573817240235129], [-1.4176322005330935, 0.5354869087283456], [-1.4130695989436457, 0.5255080959855084], [-1.256329259450963, 0.44446603942140717], [-1.4130695989436457, 0.5255080959855084], [-0.9540288225056899, 0.3684942105919057], [-0.9539344596568613, 0.3680822513306155], [-0.9540288225056899, 0.3684942105919057], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-10, 100], [-1.2929389096436823, 0.46711348969254857], [-1.4121378416404786, 0.5151443645527445], [-1.4130695989436457, 0.5255080959855084], [-1.0473706461173196, 0.38684563700629154], [-1.095806577448146, 0.3959677830050857], [-1.2820027411357504, 0.4517032845847724], [-1.4130695989436457, 0.5255080959855084], [-1.2783595552855107, 0.4489028824719865], [-1.2789591912869407, 0.4497352935391749], [-1.3946058601047842, 0.4987134209628028], [-1.4176322005330935, 0.5354869087283456], [-1.095044027383473, 0.393636576365465], [-1.464431045190313, 0.5742662272406743], [-1.4130695989436457, 0.5255080959855084], [-0.9539344596568613, 0.3680822513306155], [-0.9539344596568613, 0.3680822513306155], [-0.9540288225056899, 0.3684942105919057], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-10, 100], [-1.4121378416404786, 0.5151443645527445], [-1.0473706461173196, 0.38684563700629154], [-1.2929389096436823, 0.46711348969254857], [-1.3946058601047842, 0.4987134209628028], [-1.464431045190313, 0.5742662272406743], [-1.4176322005330935, 0.5354869087283456], [-1.4174471275180691, 0.5243645979701984], [-1.2828164816503962, 0.4529457129292251], [-1.095044027383473, 0.393636576365465], [-1.2820027411357504, 0.4517032845847724], [-1.0446069861277396, 0.3801804213559368], [-1.2789591912869407, 0.4497352935391749], [-1.2783595552855107, 0.4489028824719865], [-0.95124885240383, 0.3675829094979643], [-0.9539344596568613, 0.3680822513306155], [-0.9539344596568613, 0.3680822513306155], [-0.9540288225056899, 0.3684942105919057], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-10, 100], [-1.4121378416404786, 0.5151443645527445], [-1.3946058601047842, 0.4987134209628028], [-1.4174471275180691, 0.5243645979701984], [-1.0473706461173196, 0.38684563700629154], [-1.2828164816503962, 0.4529457129292251], [-1.4174760002713334, 0.5337078649934772], [-1.0473706461173196, 0.38684563700629154], [-1.3946058601047842, 0.4987134209628028], [-1.095044027383473, 0.393636576365465], [-1.464431045190313, 0.5742662272406743], [-1.2820027411357504, 0.4517032845847724], [-1.2783595552855107, 0.4489028824719865], [-0.9481225102144295, 0.36582531549104746], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-10, 100], [-1.464431045190313, 0.5742662272406743], [-1.4121378416404786, 0.5151443645527445], [-1.3946058601047842, 0.4987134209628028], [-1.095495296766469, 0.39490246221352976], [-1.3946058601047842, 0.4987134209628028], [-1.095044027383473, 0.393636576365465], [-1.2842549432583186, 0.4865764262988022], [-1.3946058601047842, 0.4987134209628028], [-1.2820027411357504, 0.4517032845847724], [-1.2828164816503962, 0.4529457129292251], [-1.26473838315431, 0.44165109457779395], [-1.0455841917580444, 0.382937058406747], [-1.4121378416404786, 0.5151443645527445], [-1.094514432400109, 0.3915161242860949], [-1.095044027383473, 0.393636576365465], [-0.9475101384099525, 0.36460125446780434], [-0.9481225102144295, 0.36582531549104746], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-1.0455841917580444, 0.382937058406747], [-10, 100], [-1.4121378416404786, 0.5151443645527445], [-1.464431045190313, 0.5742662272406743], [-1.4132549846664133, 0.5288226924427221], [-1.3898534065426116, 0.4961957439605789], [-1.410437918704374, 0.5115042106479962], [-1.095495296766469, 0.39490246221352976], [-1.394748278856181, 0.502963804204601], [-1.4121378416404786, 0.5151443645527445], [-1.095044027383473, 0.393636576365465], [-1.3946058601047842, 0.4987134209628028], [-1.2820027411357504, 0.4517032845847724], [-1.095044027383473, 0.393636576365465], [-1.3946058601047842, 0.4987134209628028], [-1.26473838315431, 0.44165109457779395], [-0.9475101384099525, 0.36460125446780434], [-0.9481225102144295, 0.36582531549104746], [-10, 100], [-1.4107857261965118, 0.5112875630521161], [-1.394748278856181, 0.502963804204601], [-1.2820027411357504, 0.4517032845847724], [-1.42832648957261, 0.5295517857923744], [-1.3898534065426116, 0.4961957439605789], [-1.3898534065426116, 0.4961957439605789], [-1.0717218301574805, 0.3852760492075957], [-1.3898861787766408, 0.49631216160890274], [-1.095044027383473, 0.393636576365465], [-1.095495296766469, 0.39490246221352976], [-1.095044027383473, 0.393636576365465], [-1.4654755733135758, 0.5683175601360823], [-1.3946058601047842, 0.4987134209628028], [-1.2662954651288798, 0.44299447500735634], [-1.3946058601047842, 0.4987134209628028], [-1.41394086066078, 0.513387056812002], [-1.0447181571916138, 0.3817208420227628], [-0.9475101384099525, 0.36460125446780434], [-0.9481225102144295, 0.36582531549104746], [-10, 100], [-1.3898534065426116, 0.4961957439605789], [-1.4654755733135758, 0.5683175601360823], [-1.2820027411357504, 0.4517032845847724], [-1.4107857261965118, 0.5112875630521161], [-1.4107791007938577, 0.5112058718425246], [-1.395601098858185, 0.502989913311754], [-1.095495296766469, 0.39490246221352976], [-1.42832648957261, 0.5295517857923744], [-1.0717218301574805, 0.3852760492075957], [-1.394748278856181, 0.502963804204601], [-1.3898534065426116, 0.4961957439605789], [-1.3888308268133163, 0.4935385595069597], [-1.095044027383473, 0.393636576365465], [-1.095044027383473, 0.393636576365465], [-0.9639687274485625, 0.36674548298648885], [-1.095044027383473, 0.393636576365465], [-1.3898861787766408, 0.49631216160890274], [-0.9471301652012611, 0.3642337388746554], [-0.9475101384099525, 0.36460125446780434], [-0.9481225102144295, 0.36582531549104746], [-10, 100], [-1.4654755733135758, 0.5683175601360823], [-1.2820027411357504, 0.4517032845847724], [-1.4107791007938577, 0.5112058718425246], [-1.3888308268133163, 0.4935385595069597], [-1.4200585819033904, 0.5125698188604988], [-1.3898534065426116, 0.4961957439605789], [-1.095495296766469, 0.39490246221352976], [-1.3873269038225167, 0.4857484169747021], [-1.4281849778792164, 0.5270738505119006], [-1.395605506957734, 0.5030020932257016], [-1.395601098858185, 0.502989913311754], [-1.0717218301574805, 0.3852760492075957], [-1.095044027383473, 0.393636576365465], [-1.394748278856181, 0.502963804204601], [-1.3898534065426116, 0.4961957439605789], [-1.0766686455774899, 0.38941695242026175], [-0.9471616745248362, 0.36275753291613033], [-0.9475101384099525, 0.36460125446780434], [-0.9481225102144295, 0.36582531549104746], [-10, 100], [-1.4654755733135758, 0.5683175601360823], [-1.3888308268133163, 0.4935385595069597], [-1.3873269038225167, 0.4857484169747021], [-1.428854513524691, 0.5274924787804054], [-1.4107791007938577, 0.5112058718425246], [-1.2820027411357504, 0.4517032845847724], [-1.0766686455774899, 0.38941695242026175], [-1.4202128111695589, 0.511686090845711], [-1.0717218301574805, 0.3852760492075957], [-1.3898534065426116, 0.4961957439605789], [-1.4281849778792164, 0.5270738505119006], [-1.3898534065426116, 0.4961957439605789], [-1.095495296766469, 0.39490246221352976], [-1.395601098858185, 0.502989913311754], [-1.0793871644755832, 0.39357425196549456], [-1.0685544262617181, 0.3843818650469295], [-0.937082460145327, 0.3622933937315996], [-0.9471616745248362, 0.36275753291613033], [-0.9475101384099525, 0.36460125446780434], [-0.9481225102144295, 0.36582531549104746], [-10, 100], [-1.4103870275678385, 0.5089730384487641], [-1.4654755733135758, 0.5683175601360823], [-1.4281849778792164, 0.5270738505119006], [-1.3873269038225167, 0.4857484169747021], [-1.402764005349501, 0.4978940051465041], [-1.4400902471706698, 0.5357877580036139], [-1.0717218301574805, 0.3852760492075957], [-1.2820027411357504, 0.4517032845847724], [-1.4107791007938577, 0.5112058718425246], [-1.428854513524691, 0.5274924787804054], [-1.095495296766469, 0.39490246221352976], [-1.4575856317622842, 0.5667263016499872], [-1.4202128111695589, 0.511686090845711], [-1.0883417652068945, 0.3908605680837851], [-1.0766686455774899, 0.38941695242026175], [-0.937082460145327, 0.3622933937315996], [-0.9471616745248362, 0.36275753291613033], [-0.9475101384099525, 0.36460125446780434], [-0.9481225102144295, 0.36582531549104746], [-10, 100], [-1.402764005349501, 0.4978940051465041], [-1.4202128111695589, 0.511686090845711], [-1.428854513524691, 0.5274924787804054], [-1.095495296766469, 0.39490246221352976], [-1.4575856317622842, 0.5667263016499872], [-1.42622833603564, 0.5237189956746435], [-1.4654755733135758, 0.5683175601360823], [-1.4281849778792164, 0.5270738505119006], [-1.412324167535324, 0.5011356474313767], [-1.0717218301574805, 0.3852760492075957], [-1.0717218301574805, 0.3852760492075957], [-1.4402072951530656, 0.544610325047035], [-1.3820288284157454, 0.48317654319231285], [-1.0952885057419783, 0.38844202685707585], [-1.2820027411357504, 0.4517032845847724], [-0.9325063610138403, 0.36162053788921994], [-0.937082460145327, 0.3622933937315996], [-0.9471616745248362, 0.36275753291613033], [-0.9475101384099525, 0.36460125446780434], [-10, 100], [-1.4202128111695589, 0.511686090845711], [-1.095495296766469, 0.39490246221352976], [-1.42622833603564, 0.5237189956746435], [-1.4575856317622842, 0.5667263016499872], [-1.412324167535324, 0.5011356474313767], [-1.0952885057419783, 0.38844202685707585], [-1.3820288284157454, 0.48317654319231285], [-1.448977613501996, 0.5428047262166655], [-1.428854513524691, 0.5274924787804054], [-1.0717218301574805, 0.3852760492075957], [-1.4281849778792164, 0.5270738505119006], [-1.4654755733135758, 0.5683175601360823], [-1.0717218301574805, 0.3852760492075957], [-1.4588516197581252, 0.5681974048575728], [-1.402764005349501, 0.4978940051465041], [-0.9163780621187891, 0.35661439683943996], [-0.9325063610138403, 0.36162053788921994], [-0.937082460145327, 0.3622933937315996], [-0.9471616745248362, 0.36275753291613033], [-0.9475101384099525, 0.36460125446780434], [-10, 100], [-1.4202128111695589, 0.511686090845711], [-1.4202128111695589, 0.511686090845711], [-1.4575856317622842, 0.5667263016499872], [-1.42622833603564, 0.5237189956746435], [-1.4654755733135758, 0.5683175601360823], [-1.448977613501996, 0.5428047262166655], [-1.0952885057419783, 0.38844202685707585], [-1.0717218301574805, 0.3852760492075957], [-1.4281849778792164, 0.5270738505119006], [-1.428854513524691, 0.5274924787804054], [-1.0716679666541087, 0.38517271488155375], [-1.448954093209837, 0.5395195105028984], [-1.3820288284157454, 0.48317654319231285], [-1.412324167535324, 0.5011356474313767], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-0.9325063610138403, 0.36162053788921994], [-0.937082460145327, 0.3622933937315996], [-0.9471616745248362, 0.36275753291613033], [-0.9475101384099525, 0.36460125446780434], [-10, 100], [-1.42622833603564, 0.5237189956746435], [-1.3829588123605956, 0.48359024141722307], [-1.4654755733135758, 0.5683175601360823], [-1.4202128111695589, 0.511686090845711], [-1.2611386568403367, 0.4723029014236037], [-1.4202128111695589, 0.511686090845711], [-1.3820288284157454, 0.48317654319231285], [-1.4202128111695589, 0.511686090845711], [-1.448977613501996, 0.5428047262166655], [-1.428854513524691, 0.5274924787804054], [-1.4281849778792164, 0.5270738505119006], [-1.105381402950308, 0.4075655811097542], [-1.0716679666541087, 0.38517271488155375], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-0.9325063610138403, 0.36162053788921994], [-0.937082460145327, 0.3622933937315996], [-0.9471616745248362, 0.36275753291613033], [-10, 100], [-1.42622833603564, 0.5237189956746435], [-1.448977613501996, 0.5428047262166655], [-1.3829588123605956, 0.48359024141722307], [-1.4202128111695589, 0.511686090845711], [-1.454612577704107, 0.5504081786171011], [-1.2611386568403367, 0.4723029014236037], [-1.408153595490304, 0.5073689310763663], [-1.4202128111695589, 0.511686090845711], [-1.4654755733135758, 0.5683175601360823], [-1.3820288284157454, 0.48317654319231285], [-1.4202128111695589, 0.511686090845711], [-1.4202128111695589, 0.511686090845711], [-1.11245431169609, 0.4100218978783978], [-1.428093811055137, 0.5257349714759298], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-0.9325063610138403, 0.36162053788921994], [-0.937082460145327, 0.3622933937315996], [-0.9471616745248362, 0.36275753291613033], [-10, 100], [-1.42622833603564, 0.5237189956746435], [-1.449416305974799, 0.5456909594099467], [-1.448977613501996, 0.5428047262166655], [-1.4202128111695589, 0.511686090845711], [-1.3820288284157454, 0.48317654319231285], [-1.460327780986089, 0.559726604136297], [-1.408153595490304, 0.5073689310763663], [-1.4665940089303866, 0.5693812934519735], [-1.4202128111695589, 0.511686090845711], [-1.3829588123605956, 0.48359024141722307], [-1.4202128111695589, 0.511686090845711], [-1.11245431169609, 0.4100218978783978], [-1.4202128111695589, 0.511686090845711], [-1.3833949957451959, 0.48379206676827574], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-0.9325063610138403, 0.36162053788921994], [-0.937082460145327, 0.3622933937315996], [-0.9471616745248362, 0.36275753291613033], [-10, 100], [-1.42622833603564, 0.5237189956746435], [-1.448977613501996, 0.5428047262166655], [-1.4666008691564976, 0.5657126801535856], [-1.408153595490304, 0.5073689310763663], [-1.3820288284157454, 0.48317654319231285], [-1.460327780986089, 0.559726604136297], [-1.4202128111695589, 0.511686090845711], [-1.4095100003646372, 0.5082757376786292], [-0.9596520316551872, 0.3655932135113937], [-1.4202128111695589, 0.511686090845711], [-1.3834141455535711, 0.4844372732227525], [-1.11245431169609, 0.4100218978783978], [-1.4202128111695589, 0.511686090845711], [-1.3829588123605956, 0.48359024141722307], [-0.8696057691579352, 0.34464588233014276], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-0.9325063610138403, 0.36162053788921994], [-0.937082460145327, 0.3622933937315996], [-0.9471616745248362, 0.36275753291613033], [-10, 100], [-1.448977613501996, 0.5428047262166655], [-1.4666008691564976, 0.5657126801535856], [-1.42622833603564, 0.5237189956746435], [-1.4206519551536683, 0.5175696244051936], [-1.4095100003646372, 0.5082757376786292], [-1.3834141455535711, 0.4844372732227525], [-1.4202128111695589, 0.511686090845711], [-1.3820288284157454, 0.48317654319231285], [-1.408348970000566, 0.4980240950683841], [-1.3820288284157454, 0.48317654319231285], [-1.4202128111695589, 0.511686090845711], [-0.9596520316551872, 0.3655932135113937], [-1.4202094229990645, 0.5116732869390036], [-0.8652031377048817, 0.34367376052212883], [-0.8696057691579352, 0.34464588233014276], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-0.9471616745248362, 0.36275753291613033], [-10, 100], [-1.4666008691564976, 0.5657126801535856], [-1.42622833603564, 0.5237189956746435], [-1.4243461189224553, 0.5203129020626762], [-1.408348970000566, 0.4980240950683841], [-1.4231695212065647, 0.5148592328396601], [-1.4544519724606004, 0.5507071798955901], [-1.4095100003646372, 0.5082757376786292], [-1.3834141455535711, 0.4844372732227525], [-1.4505131188873661, 0.5352347375527318], [-1.439811855031002, 0.5239147194242194], [-0.9471616745248362, 0.36275753291613033], [-1.3820288284157454, 0.48317654319231285], [-0.9596520316551872, 0.3655932135113937], [-1.3820288284157454, 0.48317654319231285], [-0.8652031377048817, 0.34367376052212883], [-0.8696057691579352, 0.34464588233014276], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-10, 100], [-1.4243461189224553, 0.5203129020626762], [-1.4544519724606004, 0.5507071798955901], [-1.468490350907183, 0.5800517520961818], [-1.4505131188873661, 0.5352347375527318], [-0.9596520316551872, 0.3655932135113937], [-1.4666008691564976, 0.5657126801535856], [-1.4596463334870424, 0.5558086785830106], [-1.4095100003646372, 0.5082757376786292], [-1.408348970000566, 0.4980240950683841], [-1.4231695212065647, 0.5148592328396601], [-1.0103810642927267, 0.3982674401622578], [-0.9550808624575816, 0.35907318371447494], [-1.4231695212065647, 0.5148592328396601], [-1.4085977990776475, 0.5025150290238427], [-1.3834141455535711, 0.4844372732227525], [-0.8652031377048817, 0.34367376052212883], [-0.8696057691579352, 0.34464588233014276], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-10, 100], [-1.468490350907183, 0.5800517520961818], [-1.4544519724606004, 0.5507071798955901], [-1.4243461189224553, 0.5203129020626762], [-1.4514988595036011, 0.5402898399626129], [-0.9596520316551872, 0.3655932135113937], [-1.3834141455535711, 0.4844372732227525], [-1.4505131188873661, 0.5352347375527318], [-1.4243461189224553, 0.5203129020626762], [-1.0051124729087775, 0.3875731422278766], [-1.4231695212065647, 0.5148592328396601], [-1.4666008691564976, 0.5657126801535856], [-1.408348970000566, 0.4980240950683841], [-1.45051421455045, 0.5352392632748519], [-1.4596463334870424, 0.5558086785830106], [-1.40869111770323, 0.5021114262975112], [-0.8520425699835998, 0.3388470392286221], [-0.8696057691579352, 0.34464588233014276], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-10, 100], [-1.4514988595036011, 0.5402898399626129], [-1.0051124729087775, 0.3875731422278766], [-1.468490350907183, 0.5800517520961818], [-1.4544519724606004, 0.5507071798955901], [-0.9596520316551872, 0.3655932135113937], [-1.468490350907183, 0.5800517520961818], [-1.4596463334870424, 0.5558086785830106], [-1.45051421455045, 0.5352392632748519], [-1.4243461189224553, 0.5203129020626762], [-1.4505131188873661, 0.5352347375527318], [-1.4243461189224553, 0.5203129020626762], [-0.8691328500366201, 0.3432547598415758], [-1.4231695212065647, 0.5148592328396601], [-1.4240223550578976, 0.5187830801789081], [-1.40869111770323, 0.5021114262975112], [-0.8518927971654053, 0.33878475958587995], [-0.8520425699835998, 0.3388470392286221], [-0.8696057691579352, 0.34464588233014276], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-10, 100], [-1.4596463334870424, 0.5558086785830106], [-1.4514988595036011, 0.5402898399626129], [-1.468490350907183, 0.5800517520961818], [-1.0051124729087775, 0.3875731422278766], [-1.45051421455045, 0.5352392632748519], [-1.4602945951540034, 0.5570124626901498], [-1.468490350907183, 0.5800517520961818], [-1.4231695212065647, 0.5148592328396601], [-1.4240223550578976, 0.5187830801789081], [-1.4571590196336945, 0.5434185964655055], [-0.9163780621187891, 0.35661439683943996], [-1.4514988595036011, 0.5402898399626129], [-1.0089636716304216, 0.3896984009693242], [-1.40869111770323, 0.5021114262975112], [-0.8518927971654053, 0.33878475958587995], [-0.8520425699835998, 0.3388470392286221], [-0.8696057691579352, 0.34464588233014276], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-10, 100], [-1.468490350907183, 0.5800517520961818], [-1.4596463334870424, 0.5558086785830106], [-1.40869111770323, 0.5021114262975112], [-0.9199153284337566, 0.3576675878960487], [-1.4571590196336945, 0.5434185964655055], [-1.4514988595036011, 0.5402898399626129], [-1.4514988595036011, 0.5402898399626129], [-1.4231695212065647, 0.5148592328396601], [-1.45051421455045, 0.5352392632748519], [-0.9360468242015144, 0.3764485801101868], [-1.1569013129743826, 0.4525362130652837], [-1.4602945951540034, 0.5570124626901498], [-0.9163780621187891, 0.35661439683943996], [-0.8636527771095619, 0.3432820585964958], [-0.8518927971654053, 0.33878475958587995], [-0.8518927971654053, 0.33878475958587995], [-0.8520425699835998, 0.3388470392286221], [-0.8696057691579352, 0.34464588233014276], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-10, 100], [-1.468490350907183, 0.5800517520961818], [-0.9200792091811787, 0.3577169376803148], [-1.4596463334870424, 0.5558086785830106], [-1.3883280195495393, 0.49028777982619404], [-1.40869111770323, 0.5021114262975112], [-1.4577493004312787, 0.5458269574383501], [-0.9199153284337566, 0.3576675878960487], [-1.4514988595036011, 0.5402898399626129], [-1.45051421455045, 0.5352392632748519], [-1.451427450916217, 0.5398803897523893], [-1.4514988595036011, 0.5402898399626129], [-0.8636527771095619, 0.3432820585964958], [-0.9163780621187891, 0.35661439683943996], [-0.8512420194579635, 0.3375408323482645], [-0.8518927971654053, 0.33878475958587995], [-0.8518927971654053, 0.33878475958587995], [-0.8520425699835998, 0.3388470392286221], [-0.8696057691579352, 0.34464588233014276], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-10, 100], [-1.4596463334870424, 0.5558086785830106], [-1.3883280195495393, 0.49028777982619404], [-1.4577493004312787, 0.5458269574383501], [-1.468490350907183, 0.5800517520961818], [-0.9200792091811787, 0.3577169376803148], [-0.8636527771095619, 0.3432820585964958], [-0.9749550816484938, 0.3770582471586645], [-1.451427450916217, 0.5398803897523893], [-1.45051421455045, 0.5352392632748519], [-1.40869111770323, 0.5021114262975112], [-0.9239029671137645, 0.35903347249249257], [-0.9199153284337566, 0.3576675878960487], [-0.8512420194579635, 0.3375408323482645], [-0.8518927971654053, 0.33878475958587995], [-0.8518927971654053, 0.33878475958587995], [-0.8520425699835998, 0.3388470392286221], [-0.8696057691579352, 0.34464588233014276], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-10, 100], [-1.468490350907183, 0.5800517520961818], [-1.3883280195495393, 0.49028777982619404], [-1.40869111770323, 0.5021114262975112], [-1.451427450916217, 0.5398803897523893], [-1.4596463334870424, 0.5558086785830106], [-1.2385502792858696, 0.44871836142377547], [-1.4596463334870424, 0.5558086785830106], [-1.4583445937837736, 0.5538946260968097], [-0.8636527771095619, 0.3432820585964958], [-1.4577493004312787, 0.5458269574383501], [-0.9200792091811787, 0.3577169376803148], [-0.9269497809730017, 0.36250711886906734], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-0.8518927971654053, 0.33878475958587995], [-0.8518927971654053, 0.33878475958587995], [-0.8520425699835998, 0.3388470392286221], [-0.8696057691579352, 0.34464588233014276], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-10, 100], [-1.3883280195495393, 0.49028777982619404], [-0.9269497809730017, 0.36250711886906734], [-1.4583445937837736, 0.5538946260968097], [-1.468490350907183, 0.5800517520961818], [-1.2385502792858696, 0.44871836142377547], [-1.40869111770323, 0.5021114262975112], [-1.451427450916217, 0.5398803897523893], [-1.4577493004312787, 0.5458269574383501], [-1.4596463334870424, 0.5558086785830106], [-1.4596463334870424, 0.5558086785830106], [-0.9111747896976676, 0.35517115478213984], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-0.8518927971654053, 0.33878475958587995], [-0.8518927971654053, 0.33878475958587995], [-0.8520425699835998, 0.3388470392286221], [-0.8696057691579352, 0.34464588233014276], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-10, 100], [-1.2385502792858696, 0.44871836142377547], [-1.468490350907183, 0.5800517520961818], [-1.4583445937837736, 0.5538946260968097], [-1.451427450916217, 0.5398803897523893], [-1.40869111770323, 0.5021114262975112], [-1.2585012982919621, 0.47316383378394455], [-0.8675740214893913, 0.3398796054879388], [-1.3883280195495393, 0.49028777982619404], [-1.2385502792858696, 0.44871836142377547], [-0.8514409200405614, 0.33788724136877946], [-0.9111747896976676, 0.35517115478213984], [-0.8370224472069098, 0.3345934420747154], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-0.8520425699835998, 0.3388470392286221], [-0.8696057691579352, 0.34464588233014276], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-10, 100], [-1.468490350907183, 0.5800517520961818], [-1.2585012982919621, 0.47316383378394455], [-1.4583445937837736, 0.5538946260968097], [-1.451427450916217, 0.5398803897523893], [-1.2430428669389935, 0.4520081287954331], [-1.40869111770323, 0.5021114262975112], [-1.3883280195495393, 0.49028777982619404], [-1.4568852716395295, 0.5517947213944236], [-0.9491025596523968, 0.3598707635722677], [-0.9111747896976676, 0.35517115478213984], [-0.8675740214893913, 0.3398796054879388], [-0.8514409200405614, 0.33788724136877946], [-0.8370224472069098, 0.3345934420747154], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-0.8520425699835998, 0.3388470392286221], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-10, 100], [-0.9581063844745832, 0.3681448043854911], [-1.40869111770323, 0.5021114262975112], [-1.243978579278641, 0.45318768322168046], [-1.2585012982919621, 0.47316383378394455], [-1.468490350907183, 0.5800517520961818], [-0.8520425699835998, 0.3388470392286221], [-1.2504584739335691, 0.4672679153863782], [-0.9111747896976676, 0.35517115478213984], [-1.4568852716395295, 0.5517947213944236], [-1.4583445937837736, 0.5538946260968097], [-1.461610422966705, 0.5628962410534393], [-0.8675740214893913, 0.3398796054879388], [-1.451427450916217, 0.5398803897523893], [-0.8370224472069098, 0.3345934420747154], [-0.8370224472069098, 0.3345934420747154], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-0.8520425699835998, 0.3388470392286221], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-10, 100], [-1.2504584739335691, 0.4672679153863782], [-1.249566652255188, 0.46585235593242286], [-1.468490350907183, 0.5800517520961818], [-1.243978579278641, 0.45318768322168046], [-1.4568852716395295, 0.5517947213944236], [-0.8520425699835998, 0.3388470392286221], [-1.40869111770323, 0.5021114262975112], [-1.2585012982919621, 0.47316383378394455], [-1.451427450916217, 0.5398803897523893], [-0.9084020981223969, 0.35441725324216883], [-0.8675740214893913, 0.3398796054879388], [-1.243978579278641, 0.45318768322168046], [-0.8359508296062127, 0.33288636862017057], [-0.8370224472069098, 0.3345934420747154], [-0.8370224472069098, 0.3345934420747154], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-0.8520425699835998, 0.3388470392286221], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-10, 100], [-1.2504584739335691, 0.4672679153863782], [-1.4568852716395295, 0.5517947213944236], [-1.4086992242377239, 0.5022895082746355], [-1.40869111770323, 0.5021114262975112], [-1.249566652255188, 0.46585235593242286], [-0.8761206543570286, 0.34226824824983365], [-1.4685005072968105, 0.5796856494654627], [-1.249566652255188, 0.46585235593242286], [-1.4618463590364401, 0.5712515465337507], [-1.243978579278641, 0.45318768322168046], [-0.8520425699835998, 0.3388470392286221], [-0.8359508296062127, 0.33288636862017057], [-0.8370224472069098, 0.3345934420747154], [-0.8370224472069098, 0.3345934420747154], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-0.8520425699835998, 0.3388470392286221], [-0.9163780621187891, 0.35661439683943996], [-0.9163780621187891, 0.35661439683943996], [-10, 100], [-0.8789745121558297, 0.35152854477191847], [-1.40869111770323, 0.5021114262975112], [-1.2497148046368016, 0.4628323561019188], [-0.8761206543570286, 0.34226824824983365], [-1.4275082213277788, 0.5357297687172062], [-1.4685005072968105, 0.5796856494654627], [-1.4568852716395295, 0.5517947213944236], [-1.4087044301530915, 0.5024051405742], [-1.2504584739335691, 0.4672679153863782], [-1.4086992242377239, 0.5022895082746355], [-1.243978579278641, 0.45318768322168046], [-0.8305550830384089, 0.3325714653162293], [-0.8359508296062127, 0.33288636862017057], [-0.8370224472069098, 0.3345934420747154], [-0.8370224472069098, 0.3345934420747154], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-0.8520425699835998, 0.3388470392286221], [-10, 100], [-1.2497148046368016, 0.4628323561019188], [-1.4275082213277788, 0.5357297687172062], [-1.4685005072968105, 0.5796856494654627], [-0.8789745121558297, 0.35152854477191847], [-1.4087044301530915, 0.5024051405742], [-0.8761206543570286, 0.34226824824983365], [-1.40869111770323, 0.5021114262975112], [-1.4274497606015184, 0.5353844607021304], [-1.243978579278641, 0.45318768322168046], [-1.467772041542539, 0.5760107481558602], [-1.4086992242377239, 0.5022895082746355], [-0.9438455098485448, 0.35527475185479446], [-0.8305550830384089, 0.3325714653162293], [-0.8359508296062127, 0.33288636862017057], [-0.8370224472069098, 0.3345934420747154], [-0.8370224472069098, 0.3345934420747154], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-0.8520425699835998, 0.3388470392286221], [-10, 100], [-1.4685005072968105, 0.5796856494654627], [-1.2497148046368016, 0.4628323561019188], [-1.4087044301530915, 0.5024051405742], [-1.4274497606015184, 0.5353844607021304], [-0.8761206543570286, 0.34226824824983365], [-1.4275082213277788, 0.5357297687172062], [-1.4711078120200307, 0.606436251944316], [-1.406759492572669, 0.5008911762561385], [-1.467772041542539, 0.5760107481558602], [-1.40869111770323, 0.5021114262975112], [-1.044944389977716, 0.42916762553128285], [-1.243978579278641, 0.45318768322168046], [-0.8305550830384089, 0.3325714653162293], [-0.8359508296062127, 0.33288636862017057], [-0.8370224472069098, 0.3345934420747154], [-0.8370224472069098, 0.3345934420747154], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-0.8520425699835998, 0.3388470392286221], [-10, 100], [-1.4685005072968105, 0.5796856494654627], [-1.4274497606015184, 0.5353844607021304], [-1.4087044301530915, 0.5024051405742], [-1.144617373070981, 0.42881732739605405], [-1.406759492572669, 0.5008911762561385], [-1.2497148046368016, 0.4628323561019188], [-1.4275082213277788, 0.5357297687172062], [-1.467772041542539, 0.5760107481558602], [-0.8761206543570286, 0.34226824824983365], [-1.40869111770323, 0.5021114262975112], [-1.4682060385097597, 0.5779778371822906], [-1.243978579278641, 0.45318768322168046], [-0.8305550830384089, 0.3325714653162293], [-0.8359508296062127, 0.33288636862017057], [-0.8370224472069098, 0.3345934420747154], [-0.8370224472069098, 0.3345934420747154], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-10, 100], [-1.144617373070981, 0.42881732739605405], [-1.4087044301530915, 0.5024051405742], [-1.4274497606015184, 0.5353844607021304], [-1.406759492572669, 0.5008911762561385], [-1.4691168850402714, 0.5714225939493394], [-1.243978579278641, 0.45318768322168046], [-1.2497148046368016, 0.4628323561019188], [-1.144617373070981, 0.42881732739605405], [-1.40869111770323, 0.5021114262975112], [-0.8761206543570286, 0.34226824824983365], [-0.8512420194579635, 0.3375408323482645], [-0.8370224472069098, 0.3345934420747154], [-1.4227905498798583, 0.5123188223525489], [-0.8305550830384089, 0.3325714653162293], [-0.8359508296062127, 0.33288636862017057], [-0.8370224472069098, 0.3345934420747154], [-0.8370224472069098, 0.3345934420747154], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-10, 100], [-1.4274497606015184, 0.5353844607021304], [-1.144617373070981, 0.42881732739605405], [-1.4691168850402714, 0.5714225939493394], [-1.4227905498798583, 0.5123188223525489], [-1.4698734674829539, 0.5993462622050555], [-1.4087044301530915, 0.5024051405742], [-1.406759492572669, 0.5008911762561385], [-1.406759492572669, 0.5008911762561385], [-1.144617373070981, 0.42881732739605405], [-1.243978579278641, 0.45318768322168046], [-0.8761206543570286, 0.34226824824983365], [-1.243978579278641, 0.45318768322168046], [-0.8370224472069098, 0.3345934420747154], [-0.8356726531981759, 0.33245534528112614], [-0.8359508296062127, 0.33288636862017057], [-0.8370224472069098, 0.3345934420747154], [-0.8370224472069098, 0.3345934420747154], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-10, 100], [-1.4691168850402714, 0.5714225939493394], [-1.4274497606015184, 0.5353844607021304], [-1.4235870460190811, 0.5148902068026332], [-1.2613909651710036, 0.45834461648082914], [-1.4227905498798583, 0.5123188223525489], [-1.144617373070981, 0.42881732739605405], [-1.406759492572669, 0.5008911762561385], [-1.4022926008872345, 0.49783985476830106], [-1.4087044301530915, 0.5024051405742], [-1.2431769297975523, 0.45216667280755707], [-1.406759492572669, 0.5008911762561385], [-1.144617373070981, 0.42881732739605405], [-0.8370224472069098, 0.3345934420747154], [-0.8356726531981759, 0.33245534528112614], [-0.8359508296062127, 0.33288636862017057], [-0.8370224472069098, 0.3345934420747154], [-0.8370224472069098, 0.3345934420747154], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-10, 100], [-1.4691168850402714, 0.5714225939493394], [-1.2613909651710036, 0.45834461648082914], [-1.406759492572669, 0.5008911762561385], [-1.4048207689773367, 0.49721510225627696], [-1.4227905498798583, 0.5123188223525489], [-1.144617373070981, 0.42881732739605405], [-1.4259376971153004, 0.524379055717493], [-1.406759492572669, 0.5008911762561385], [-1.144617373070981, 0.42881732739605405], [-0.8370224472069098, 0.3345934420747154], [-0.9247068489428798, 0.3505267147370436], [-1.2431769297975523, 0.45216667280755707], [-1.4601206800909854, 0.5585464656621542], [-0.8356726531981759, 0.33245534528112614], [-0.8356726531981759, 0.33245534528112614], [-0.8359508296062127, 0.33288636862017057], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-10, 100], [-1.4601206800909854, 0.5585464656621542], [-1.2613909651710036, 0.45834461648082914], [-1.144617373070981, 0.42881732739605405], [-1.4018607973365784, 0.49528929937354715], [-0.8512420194579635, 0.3375408323482645], [-1.4691168850402714, 0.5714225939493394], [-1.4048207689773367, 0.49721510225627696], [-1.2780787531418372, 0.47573978635941117], [-1.4259376971153004, 0.524379055717493], [-1.144617373070981, 0.42881732739605405], [-1.406759492572669, 0.5008911762561385], [-0.9247068489428798, 0.3505267147370436], [-1.406759492572669, 0.5008911762561385], [-1.2431769297975523, 0.45216667280755707], [-0.8356726531981759, 0.33245534528112614], [-0.8356726531981759, 0.33245534528112614], [-0.8359508296062127, 0.33288636862017057], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-10, 100], [-1.2780787531418372, 0.47573978635941117], [-1.4702877693860232, 0.5734599941024449], [-1.4601206800909854, 0.5585464656621542], [-1.2613909651710036, 0.45834461648082914], [-1.2431769297975523, 0.45216667280755707], [-1.4259376971153004, 0.524379055717493], [-1.4601206800909854, 0.5585464656621542], [-1.144617373070981, 0.42881732739605405], [-1.286558243916366, 0.4876654498722231], [-1.144617373070981, 0.42881732739605405], [-1.4048207689773367, 0.49721510225627696], [-0.8512420194579635, 0.3375408323482645], [-1.458491631655698, 0.5565161406801061], [-1.4058751813560562, 0.4981990919909382], [-0.8356726531981759, 0.33245534528112614], [-0.8356726531981759, 0.33245534528112614], [-0.8359508296062127, 0.33288636862017057], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-10, 100], [-1.4702877693860232, 0.5734599941024449], [-1.286558243916366, 0.4876654498722231], [-1.281689724176304, 0.4804756645144134], [-1.4611492002225646, 0.5601819903393058], [-1.144617373070981, 0.42881732739605405], [-1.2780787531418372, 0.47573978635941117], [-1.4259376971153004, 0.524379055717493], [-1.2613909651710036, 0.45834461648082914], [-1.4601206800909854, 0.5585464656621542], [-1.2431769297975523, 0.45216667280755707], [-1.2431769297975523, 0.45216667280755707], [-1.2780787531418372, 0.47573978635941117], [-1.4601206800909854, 0.5585464656621542], [-1.4058751813560562, 0.4981990919909382], [-0.8356726531981759, 0.33245534528112614], [-0.8356726531981759, 0.33245534528112614], [-0.8356726531981759, 0.33245534528112614], [-0.8359508296062127, 0.33288636862017057], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-10, 100], [-1.286558243916366, 0.4876654498722231], [-1.4611492002225646, 0.5601819903393058], [-1.4259376971153004, 0.524379055717493], [-1.281689724176304, 0.4804756645144134], [-1.4702877693860232, 0.5734599941024449], [-1.144617373070981, 0.42881732739605405], [-1.286558243916366, 0.4876654498722231], [-1.2780787531418372, 0.47573978635941117], [-1.4377496368372287, 0.5422084378383311], [-1.2426284265260619, 0.45152371420911924], [-1.2431769297975523, 0.45216667280755707], [-1.2780787531418372, 0.47573978635941117], [-1.4601206800909854, 0.5585464656621542], [-0.835625474955489, 0.332241809182064], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-10, 100], [-1.286558243916366, 0.4876654498722231], [-1.4259376971153004, 0.524379055717493], [-1.461455217895138, 0.5609818332623171], [-1.4611492002225646, 0.5601819903393058], [-1.4192559286916024, 0.506276633062343], [-1.281689724176304, 0.4804756645144134], [-1.2780787531418372, 0.47573978635941117], [-1.144617373070981, 0.42881732739605405], [-1.4377496368372287, 0.5422084378383311], [-1.4702877693860232, 0.5734599941024449], [-1.2426284265260619, 0.45152371420911924], [-1.2431769297975523, 0.45216667280755707], [-1.4259376971153004, 0.524379055717493], [-1.4601206800909854, 0.5585464656621542], [-0.837380179112038, 0.33245179503276046], [-1.2780787531418372, 0.47573978635941117], [-0.835625474955489, 0.332241809182064], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-10, 100], [-1.4377496368372287, 0.5422084378383311], [-1.4192559286916024, 0.506276633062343], [-1.4261028874463633, 0.5193139162458872], [-1.281689724176304, 0.4804756645144134], [-1.4611492002225646, 0.5601819903393058], [-1.286558243916366, 0.4876654498722231], [-1.2780787531418372, 0.47573978635941117], [-1.4611492002225646, 0.5601819903393058], [-1.144617373070981, 0.42881732739605405], [-0.837380179112038, 0.33245179503276046], [-1.2801055580639362, 0.4782898873153594], [-1.4702877693860232, 0.5734599941024449], [-1.43620078966569, 0.5317377300422322], [-1.147400106024086, 0.4312765016814903], [-1.2431769297975523, 0.45216667280755707], [-1.2426284265260619, 0.45152371420911924], [-0.835625474955489, 0.332241809182064], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-10, 100], [-1.4192559286916024, 0.506276633062343], [-1.4261028874463633, 0.5193139162458872], [-1.461830185478351, 0.5720998057467068], [-1.4611492002225646, 0.5601819903393058], [-1.4702877693860232, 0.5734599941024449], [-1.144617373070981, 0.42881732739605405], [-1.281689724176304, 0.4804756645144134], [-1.4611492002225646, 0.5601819903393058], [-1.281689724176304, 0.4804756645144134], [-1.4611492002225646, 0.5601819903393058], [-1.2426284265260619, 0.45152371420911924], [-1.2801055580639362, 0.4782898873153594], [-1.2780787531418372, 0.47573978635941117], [-1.2431769297975523, 0.45216667280755707], [-1.4213106424027508, 0.5080513536212983], [-1.2780787531418372, 0.47573978635941117], [-0.835625474955489, 0.332241809182064], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-10, 100], [-1.4337755758489437, 0.5279991894549766], [-1.281689724176304, 0.4804756645144134], [-1.4702877693860232, 0.5734599941024449], [-1.4192559286916024, 0.506276633062343], [-1.4612288794405206, 0.5633278732575574], [-1.2735136515762913, 0.46994420475457627], [-1.4261028874463633, 0.5193139162458872], [-1.144617373070981, 0.42881732739605405], [-1.4611492002225646, 0.5601819903393058], [-1.2426284265260619, 0.45152371420911924], [-1.4213106424027508, 0.5080513536212983], [-1.4244337427422766, 0.5182708203833086], [-1.2801055580639362, 0.4782898873153594], [-1.281689724176304, 0.4804756645144134], [-1.2801055580639362, 0.4782898873153594], [-1.4611492002225646, 0.5601819903393058], [-0.835625474955489, 0.332241809182064], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-10, 100], [-1.144617373070981, 0.42881732739605405], [-1.4337755758489437, 0.5279991894549766], [-1.281689724176304, 0.4804756645144134], [-1.4612996886170784, 0.5527801306094635], [-1.452168063549555, 0.5473761681270669], [-1.4261028874463633, 0.5193139162458872], [-1.4213106424027508, 0.5080513536212983], [-1.2801055580639362, 0.4782898873153594], [-1.4702877693860232, 0.5734599941024449], [-1.2735136515762913, 0.46994420475457627], [-1.4255431872703057, 0.5120579738981517], [-1.2801055580639362, 0.4782898873153594], [-1.2426284265260619, 0.45152371420911924], [-1.281689724176304, 0.4804756645144134], [-1.2422384550325423, 0.4512747436632315], [-1.1572510912672853, 0.4406784974981777], [-0.835625474955489, 0.332241809182064], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-10, 100], [-1.4612996886170784, 0.5527801306094635], [-1.281689724176304, 0.4804756645144134], [-1.4213106424027508, 0.5080513536212983], [-1.4702877693860232, 0.5734599941024449], [-1.4337755758489437, 0.5279991894549766], [-1.1462901370136553, 0.4334602157441589], [-1.007052104089752, 0.3699907220995523], [-1.4261028874463633, 0.5193139162458872], [-1.2426284265260619, 0.45152371420911924], [-1.4255431872703057, 0.5120579738981517], [-1.2801055580639362, 0.4782898873153594], [-1.2801055580639362, 0.4782898873153594], [-1.452168063549555, 0.5473761681270669], [-1.1572510912672853, 0.4406784974981777], [-1.2764480304229857, 0.4737554890835612], [-0.8909245586793777, 0.3547643676038195], [-0.835625474955489, 0.332241809182064], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-10, 100], [-1.4702877693860232, 0.5734599941024449], [-1.281689724176304, 0.4804756645144134], [-1.4522474391618105, 0.5425606592980426], [-1.4213106424027508, 0.5080513536212983], [-1.439467916931539, 0.5410800671041534], [-1.1572510912672853, 0.4406784974981777], [-1.2801055580639362, 0.4782898873153594], [-1.1462901370136553, 0.4334602157441589], [-1.007052104089752, 0.3699907220995523], [-0.8909245586793777, 0.3547643676038195], [-1.2801055580639362, 0.4782898873153594], [-1.4306408974469698, 0.5183082404734164], [-1.2764480304229857, 0.4737554890835612], [-1.4255431872703057, 0.5120579738981517], [-1.4612996886170784, 0.5527801306094635], [-1.4255431872703057, 0.5120579738981517], [-0.835625474955489, 0.332241809182064], [-0.8500008045122308, 0.33727588281028986], [-0.8512420194579635, 0.3375408323482645], [-10, 100], [-1.4702877693860232, 0.5734599941024449], [-1.4612996886170784, 0.5527801306094635], [-1.439467916931539, 0.5410800671041534], [-1.1462901370136553, 0.4334602157441589], [-1.1572510912672853, 0.4406784974981777], [-1.4306408974469698, 0.5183082404734164], [-1.2801055580639362, 0.4782898873153594], [-1.4112704392018698, 0.5033776167242376], [-1.2764480304229857, 0.4737554890835612], [-1.4522474391618105, 0.5425606592980426], [-1.281689724176304, 0.4804756645144134], [-0.8920644501649885, 0.35584725220038904], [-1.2801055580639362, 0.4782898873153594], [-1.2485093478798317, 0.45163734686188056], [-0.8909245586793777, 0.3547643676038195], [-1.007052104089752, 0.3699907220995523], [-0.835625474955489, 0.332241809182064], [-0.835625474955489, 0.332241809182064], [-0.8512420194579635, 0.3375408323482645], [-10, 100], [-1.4702877693860232, 0.5734599941024449], [-1.4522474391618105, 0.5425606592980426], [-1.1462901370136553, 0.4334602157441589], [-1.4306408974469698, 0.5183082404734164], [-1.4612996886170784, 0.5527801306094635], [-1.1572510912672853, 0.4406784974981777], [-1.4112704392018698, 0.5033776167242376], [-1.2801055580639362, 0.4782898873153594], [-1.2801055580639362, 0.4782898873153594], [-1.281689724176304, 0.4804756645144134], [-1.007052104089752, 0.3699907220995523], [-0.9059333742369033, 0.3584606431562269], [-1.4699390847703802, 0.5715552382463817], [-1.416082076065158, 0.5094728044536663], [-0.8909245586793777, 0.3547643676038195], [-1.2764480304229857, 0.4737554890835612]]]
    # Bayes with surrogate - get_kpls_regressor
    # LoudspeakerMultiProblem-874752 - run (174) - NSGA_II: elapsed time: 371.1219701766968 s
    # LoudspeakerMultiProblem-874752 - nsga2 (247) - surrogate.predict_counter: 1950
    # LoudspeakerMultiProblem-874752 - nsga2 (248) - surrogate.eval_counter: 130
    multi_nsga_kpls_x = [[0.020572876363344716, 0.015804268160010897, 0.030442474269937736, 0.015786894714037325], [0.032800801168127995, 0.032021231252483226, 0.011389872037859185, 0.024149117681807015], [0.010848305402638037, 0.022746939271037804, 0.014781274489082022, 0.0074069882689080215], [0.014570043532524508, 0.018824024597929454, 0.02361773533715171, 0.011280392162032105], [0.021163246065228198, 0.02879311819915076, 0.022011768679278065, 0.01041769167657446], [0.01370630835104417, 0.026741637303658874, 0.025835381556257295, 0.027446036926090975], [0.025179268905090174, 0.02354270796516472, 0.019755617913340347, 0.0328608360822937], [0.005445229597939702, 0.027164502237216746, 0.02888143656472248, 0.021752798459313792], [0.028183224466085482, 0.03001115418826731, 0.023439570337940772, 0.017625657896934897], [0.02580567144745752, 0.032998404124448405, 0.027484957779848135, 0.03199372748972021], [0.018002440632991067, 0.02456143669324245, 0.015730783891156598, 0.009254096156908306], [0.019590560261929024, 0.034351380733116124, 0.018379721917964158, 0.014104263412142892], [0.007920390329039639, 0.015153302349261016, 0.03148392821595923, 0.033654545664022734], [0.023792764880475516, 0.02941237993806807, 0.022719870830264177, 0.00983817694947498], [0.02649770143757359, 0.01735278600996485, 0.016770596824400196, 0.018706531759137644], [0.0088493797914781, 0.023913166218898482, 0.029592539918684962, 0.014992294726141771], [0.012492440890491625, 0.029808219439655656, 0.0161701065605897, 0.026063817892353038], [0.03224295708289705, 0.022451971264035177, 0.01095513215379644, 0.03109462912265175], [0.024565117716622707, 0.018216134644786823, 0.01587258129457384, 0.02450744287525203], [0.009361607898031678, 0.01606551899715996, 0.011452698420197395, 0.012351598455311247], [0.027232154682009055, 0.025628328280059058, 0.02241578183019648, 0.00806376763281132], [0.008292315254401113, 0.03123484980439654, 0.016949567526501494, 0.007762036781640969], [0.019212536543751532, 0.0230873468954809, 0.02555149535312467, 0.01994619519272012], [0.0218639639240751, 0.01775312487223641, 0.017567548223978543, 0.034367865098720425], [0.005708657038674117, 0.0198711558682092, 0.019082809860624994, 0.012520175413957498], [0.017622579129159516, 0.02478348528072543, 0.02663688105341635, 0.02114907770190146], [0.03440444049815186, 0.023277740127934744, 0.010307683776769952, 0.025086518159129656], [0.021704673483237572, 0.015608630693664924, 0.02156878615645242, 0.009956214503945493], [0.028092931254756683, 0.01694239844215894, 0.014051696426434359, 0.019677587234702767], [0.02274837845976316, 0.029145141812543733, 0.012264274675168529, 0.005483768920987865], [0.016560262451862953, 0.021240997391722936, 0.025326166991541464, 0.026910246497009557], [0.007030846956691889, 0.025144832204486586, 0.01213147801862902, 0.01670781527976522], [0.02675856953465235, 0.016530959301295355, 0.010606933290400303, 0.01715986732011603], [0.0065601825386606884, 0.025811580239620137, 0.026425743562158432, 0.02854998490296667], [0.01595410136100527, 0.033391996127672505, 0.02136394735324494, 0.006358153548851258], [0.03105336044518517, 0.022133007058288755, 0.030180995759631582, 0.017909014037962693], [0.00951158263700087, 0.0315500973246224, 0.013883045244475877, 0.013386918740986006], [0.031202115501661704, 0.0303055086769865, 0.017258513704623106, 0.014359623191232958], [0.019849959710492533, 0.01997083580110076, 0.030814175699217253, 0.01921577376151798], [0.023490402894440692, 0.02175099528409319, 0.019617974406115193, 0.016227542405131492], [0.02944390614129484, 0.02062840356023706, 0.026908170369320077, 0.03033038720510188], [0.014046968484273642, 0.02798024692778403, 0.0243785374101321, 0.021235106739790618], [0.0061226162994553876, 0.03177338241456148, 0.02005068245995221, 0.005555176203732043], [0.03366160598805651, 0.02418805054007346, 0.020799558904702545, 0.011397151477348776], [0.03186001438990546, 0.019461060399619208, 0.029095282078297376, 0.030608105995743852], [0.0332249366925635, 0.03237569711718255, 0.012553285536023424, 0.03207443740885472], [0.018717860702192477, 0.020443230430745474, 0.013385561909744492, 0.02899441550257702], [0.02880937956149033, 0.028543422449294146, 0.027944534706495508, 0.023373413832266935], [0.01213784924457778, 0.018399752556462073, 0.013162432978588216, 0.026222837557940873], [0.030523692386671873, 0.028090488304118637, 0.014611248393792051, 0.020678576338331776], [0.010940364477900014, 0.03392410354530428, 0.029897225901664062, 0.02949619352215998], [0.011644467134561489, 0.033545048320533316, 0.02418428692237954, 0.027682324198405963], [0.013345525298212532, 0.026679206696556386, 0.022934915786037208, 0.022743099238970586], [0.010328662708599357, 0.02101485561853094, 0.02833024505332831, 0.02237488733670237], [0.029821033159059058, 0.026286951707501026, 0.017979712277016315, 0.012888801771615863], [0.0226995461278737, 0.027515295491969766, 0.018952396310453735, 0.02518435196494325], [0.025036432527800852, 0.019109367366954837, 0.020705607360256315, 0.006661617769921322], [0.016949109557574484, 0.017081793455743215, 0.02479943817660186, 0.033492988114533176], [0.015051627436207936, 0.03279365964714899, 0.0278064137951878, 0.008511841060260532], [0.015417082013756803, 0.030891619327625065, 0.015290197401870811, 0.029919684590412814], [0.021867023675, 0.021533870223, 0.017176355033, 0.0066129715609999995], [0.01812288762, 0.018451768216999998, 0.011565791362, 0.021670812361], [0.009889158434, 0.021795884863999998, 0.012845948429, 0.025254350704], [0.007743104675, 0.019737520246, 0.021088787786, 0.025316852108], [0.027814138387, 0.025453119466, 0.016544233658, 0.008311825526], [0.021279998974, 0.026590674264, 0.015515795677, 0.015642108213], [0.028994066577, 0.023220436795, 0.021539244396999998, 0.01068733654], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.01358958651], [0.025706056454, 0.021240480803, 0.017000518337, 0.022617229164], [0.020418193705, 0.0316980651, 0.012637863193, 0.023065719315], [0.03083589066, 0.022055475204, 0.018757755679999998, 0.018730043825], [0.029838145919, 0.030721640867, 0.019141398447, 0.010886602908999999], [0.010469467896, 0.028659922802, 0.024014892535, 0.026672279481], [0.018009602156, 0.031026603861, 0.017172769391, 0.030962406547], [0.029125745993, 0.030910235653, 0.011694976447, 0.028159599139999998], [0.028887660629, 0.032668147265, 0.020484789072999998, 0.01933503303], [0.024266143999, 0.030810988738, 0.024996183398999998, 0.022476862499], [0.031704530753, 0.025392384142, 0.027057382769, 0.019744948595], [0.03087019314, 0.020268536681, 0.027929733624, 0.027701796655999998], [0.034043387965, 0.027339655791, 0.029423517604, 0.027668825071], [0.021867023675, 0.021533870223, 0.017176355033, 0.006541411270419419], [0.024266143999, 0.030810988738, 0.024996183398999998, 0.022476862499], [0.03087019314, 0.03357729860804629, 0.027929733624, 0.027701796655999998], [0.027814138387, 0.025453119466, 0.016544233658, 0.008311825526], [0.010469467896, 0.028659922802, 0.024014892535, 0.026672279481], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.01358958651], [0.017985443136112805, 0.030828597065160383, 0.011565791362, 0.021670812361], [0.028994066577, 0.023220436795, 0.020947081957801896, 0.01068733654], [0.007743104675, 0.019737520246, 0.021088787786, 0.025316852108], [0.024095379236940016, 0.029142800879824814, 0.02541241722853356, 0.022476862499], [0.021279998974, 0.026590674264, 0.015515795677, 0.015642108213], [0.028994066577, 0.023220436795, 0.021539244396999998, 0.01068733654], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.01358958651], [0.021867023675, 0.021533870223, 0.017176355033, 0.0066129715609999995], [0.034043387965, 0.027339655791, 0.029423517604, 0.027668825071], [0.025706056454, 0.021240480803, 0.017000518337, 0.022617229164], [0.031704530753, 0.025392384142, 0.027057382769, 0.019744948595], [0.029838145919, 0.030721640867, 0.019141398447, 0.010886602908999999], [0.029838145919, 0.030390283333012157, 0.019141398447, 0.01078439037448456], [0.034043387965, 0.027339655791, 0.029423517604, 0.02764831076109532], [0.021867023675, 0.021533870223, 0.017176355033, 0.006541411270419419], [0.029950257339252002, 0.030721640867, 0.029034042012380832, 0.010886602908999999], [0.010469467896, 0.028659922802, 0.024014892535, 0.026672279481], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.01358958651], [0.027814138387, 0.025453119466, 0.016544233658, 0.00829807293990171], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.01358958651], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.01358958651], [0.021279998974, 0.026590674264, 0.015515795677, 0.015642108213], [0.021279998974, 0.026590674264, 0.015515795677, 0.015642108213], [0.01742456847121146, 0.028259127275, 0.022028722088571602, 0.013080294650344453], [0.027814138387, 0.025453119466, 0.01664036840184696, 0.008311825526], [0.007743104675, 0.019737520246, 0.021088787786, 0.025316852108], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.014187892671371992], [0.021867023675, 0.021533870223, 0.017176355033, 0.0066129715609999995], [0.024266143999, 0.030810988738, 0.024996183398999998, 0.022476862499], [0.03087019314, 0.03357729860804629, 0.027929733624, 0.027701796655999998], [0.027814138387, 0.025453119466, 0.016544233658, 0.008311825526], [0.024095379236940016, 0.029180238864423662, 0.02541241722853356, 0.022476862499], [0.028994066577, 0.023220436795, 0.021539244396999998, 0.01068733654], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.015099722736222912], [0.021867023675, 0.021533870223, 0.017176355033, 0.006541411270419419], [0.029950257339252002, 0.030721640867, 0.029034042012380832, 0.010886602908999999], [0.009252185870025856, 0.02910260278984047, 0.024014892535, 0.026672279481], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.01358958651], [0.021643167888342647, 0.025968853327843173, 0.017176355033, 0.0066129715609999995], [0.022029954780999998, 0.028259127275, 0.023404807998536635, 0.014187892671371992], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.01358958651], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.01358958651], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.01358958651], [0.0051205865877699255, 0.03365302128652102, 0.024025630034811903, 0.027726744265641652], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.014114758989068091], [0.02780121809722261, 0.030658898247331094, 0.029237991497772925, 0.008311825526], [0.01742456847121146, 0.028259127275, 0.022028722088571602, 0.013080294650344453], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.014187892671371992], [0.022029954780999998, 0.033552039136684614, 0.022003037263, 0.01358958651], [0.021867023675, 0.021533870223, 0.017176355033, 0.0066129715609999995], [0.027814138387, 0.025453119466, 0.016544233658, 0.00829807293990171], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.015099722736222912], [0.021279998974, 0.026590674264, 0.015515795677, 0.015642108213], [0.021279998974, 0.026590674264, 0.015515795677, 0.015642108213], [0.021867023675, 0.021533870223, 0.017176355033, 0.006541411270419419], [0.029950257339252002, 0.030721640867, 0.029034042012380832, 0.010886602908999999], [0.021643167888342647, 0.025968853327843173, 0.017176355033, 0.0066129715609999995], [0.022029954780999998, 0.028259127275, 0.022397120583223427, 0.01358958651], [0.02780121809722261, 0.030658898247331094, 0.029237991497772925, 0.008311825526], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.013548255131774538], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.01358958651], [0.01742456847121146, 0.028259127275, 0.022028722088571602, 0.013080294650344453], [0.0051205865877699255, 0.03365302128652102, 0.024025630034811903, 0.027726744265641652], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.01358958651], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.01358958651], [0.021672144803779943, 0.027690991687504073, 0.01720085532621488, 0.006687604145110257], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.01358958651], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.013581734486300519], [0.022029954780999998, 0.028259127275, 0.022003037263, 0.01358958651], [0.022029954780999998, 0.028259127275, 0.023404807998536635, 0.014187892671371992], [0.02007758390644883, 0.028259127275, 0.022003037263, 0.01358958651], [0.01742456847121146, 0.028259127275, 0.021432187484533453, 0.013080294650344453], [0.022029954780999998, 0.033552039136684614, 0.022003037263, 0.01358958651], [0.022029954780999998, 0.026025484943070245, 0.02879582301528562, 0.014187892671371992], [0.021867023675, 0.021533870223, 0.017176355033, 0.006541411270419419], [0.029950257339252002, 0.030721640867, 0.029034042012380832, 0.010886602908999999], [0.02780121809722261, 0.030658898247331094, 0.029237991497772925, 0.008311825526], [0.0051205865877699255, 0.03365302128652102, 0.024025630034811903, 0.027726744265641652], [0.022029954780999998, 0.026025484943070245, 0.02879582301528562, 0.014187892671371992], [0.022029954780999998, 0.028259127275, 0.023404807998536635, 0.014187892671371992], [0.021643167888342647, 0.025968853327843173, 0.017176355033, 0.0066129715609999995], [0.021643167888342647, 0.025096372123888783, 0.017176355033, 0.0066129715609999995], [0.01708951031209053, 0.028211886665966957, 0.022028722088571602, 0.013080294650344453], [0.021672144803779943, 0.027690991687504073, 0.01720085532621488, 0.006687604145110257], [0.022029954780999998, 0.028259127275, 0.023404807998536635, 0.014187892671371992], [0.021643167888342647, 0.025464782028263938, 0.017176635990523256, 0.0066893162882859705], [0.01375456978471739, 0.028259127275, 0.022375880411555603, 0.0157871738635478], [0.021672144803779943, 0.027701516395046057, 0.01720085532621488, 0.006687604145110257], [0.02780121809722261, 0.030658898247331094, 0.029237991497772925, 0.008311825526], [0.01742456847121146, 0.028259127275, 0.021432187484533453, 0.013080294650344453], [0.021589628225069046, 0.02768858799200491, 0.017392524738423346, 0.006687604145110257], [0.01846096495350094, 0.030908614921440494, 0.02172195865015278, 0.01358958651], [0.022029954780999998, 0.026483206788776786, 0.029140645493157733, 0.013551514314912918], [0.022029954780999998, 0.033552039136684614, 0.022003037263, 0.01358958651], [0.021867023675, 0.021533870223, 0.017176355033, 0.006541411270419419], [0.02460203629992428, 0.030658898247331094, 0.029237991497772925, 0.008311825526], [0.022029954780999998, 0.026483206788776786, 0.029140645493157733, 0.013551514314912918], [0.01375456978471739, 0.028259127275, 0.022375880411555603, 0.0157871738635478], [0.022029954780999998, 0.028259127275, 0.023404807998536635, 0.014187892671371992], [0.01708951031209053, 0.028486088771952807, 0.0233194507153011, 0.013080294650344453], [0.022029954780999998, 0.028259127275, 0.023404807998536635, 0.014187892671371992], [0.02050016694142159, 0.026425239875560346, 0.029140645493157733, 0.013551514314912918], [0.021643167888342647, 0.025968853327843173, 0.017176355033, 0.0066129715609999995], [0.022029954780999998, 0.028259127275, 0.023404807998536635, 0.014187892671371992], [0.021643167888342647, 0.025096372123888783, 0.017176355033, 0.0066129715609999995], [0.021672144803779943, 0.027690991687504073, 0.01720085532621488, 0.006687604145110257], [0.021643167888342647, 0.025464782028263938, 0.017176635990523256, 0.0066893162882859705], [0.021672144803779943, 0.027690991687504073, 0.01720085532621488, 0.006687604145110257], [0.021643167888342647, 0.025096372123888783, 0.017176355033, 0.0066129715609999995], [0.021672144803779943, 0.027701516395046057, 0.01720085532621488, 0.006687604145110257], [0.021672144803779943, 0.027701516395046057, 0.01720085532621488, 0.006687604145110257], [0.02421095856345164, 0.028259127275, 0.023404807998536635, 0.014187892671371992], [0.013257086332216146, 0.028259127275, 0.022375880411555603, 0.0157871738635478], [0.01742456847121146, 0.028259127275, 0.021432187484533453, 0.013080294650344453], [0.021867023675, 0.021533870223, 0.017176355033, 0.006541411270419419], [0.021867023675, 0.021533870223, 0.017176355033, 0.006541411270419419], [0.02460203629992428, 0.030658898247331094, 0.029237991497772925, 0.008311825526], [0.022029954780999998, 0.026483206788776786, 0.029140645493157733, 0.013551514314912918], [0.01742456847121146, 0.028259127275, 0.021432187484533453, 0.013080294650344453], [0.01375456978471739, 0.028259127275, 0.022375880411555603, 0.0157871738635478], [0.01375456978471739, 0.028259127275, 0.022375880411555603, 0.0157871738635478], [0.021643167888342647, 0.025464782028263938, 0.017176635990523256, 0.0066893162882859705], [0.021643167888342647, 0.025096372123888783, 0.01674362833278764, 0.0066129715609999995], [0.021643167888342647, 0.025464782028263938, 0.017872149508276487, 0.007750115109076364], [0.021643167888342647, 0.025096372123888783, 0.017176355033, 0.0066129715609999995], [0.02050016694142159, 0.026425239875560346, 0.029140645493157733, 0.013551514314912918], [0.021643167888342647, 0.025968853327843173, 0.017176355033, 0.0066129715609999995], [0.021643167888342647, 0.025096372123888783, 0.017176355033, 0.0066129715609999995], [0.021643167888342647, 0.025096372123888783, 0.017176355033, 0.0066129715609999995], [0.021672144803779943, 0.027690991687504073, 0.017356927726646182, 0.006687604145110257], [0.021643167888342647, 0.02586818150812905, 0.017176355033, 0.0066129715609999995], [0.02167745797809526, 0.028787303710089078, 0.024435876010506445, 0.0066893162882859705], [0.01375456978471739, 0.026289459183658683, 0.022375880411555603, 0.016108221264691608], [0.021643167888342647, 0.02823653862998602, 0.017176355033, 0.01069491110318847], [0.021867023675, 0.021533870223, 0.017176355033, 0.006541411270419419], [0.021867023675, 0.021533870223, 0.017176355033, 0.006541411270419419], [0.02460203629992428, 0.030658898247331094, 0.029237991497772925, 0.008311825526], [0.02167745797809526, 0.028787303710089078, 0.02288151697682903, 0.0066893162882859705], [0.01742456847121146, 0.028259127275, 0.021432187484533453, 0.013080294650344453], [0.022029954780999998, 0.026483206788776786, 0.029140645493157733, 0.013551514314912918], [0.02460203629992428, 0.0295495340372893, 0.029237991497772925, 0.008311825526], [0.01375456978471739, 0.026289459183658683, 0.022375880411555603, 0.016108221264691608], [0.021643167888342647, 0.025096372123888783, 0.01674362833278764, 0.0066129715609999995], [0.021643167888342647, 0.025464782028263938, 0.017176635990523256, 0.0066893162882859705], [0.02057124825943907, 0.026425239875560346, 0.029140645493157733, 0.013551514314912918], [0.02460203629992428, 0.030658898247331094, 0.029237991497772925, 0.008311825526], [0.021643167888342647, 0.02346813329728693, 0.017176635990523256, 0.006683312684964757], [0.01375456978471739, 0.026289459183658683, 0.022375880411555603, 0.01476913282743879], [0.021643167888342647, 0.025096372123888783, 0.01674362833278764, 0.0066129715609999995], [0.02050016694142159, 0.026425239875560346, 0.029140645493157733, 0.013551514314912918], [0.021643167888342647, 0.025921540029090584, 0.01819841020927088, 0.0066129715609999995], [0.021643167888342647, 0.025096372123888783, 0.017176355033, 0.0066129715609999995], [0.021643167888342647, 0.025096372123888783, 0.017176355033, 0.0066129715609999995], [0.021643167888342647, 0.025096372123888783, 0.017176355033, 0.0066129715609999995], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.021867023675, 0.021533870223, 0.017176355033, 0.006541411270419419], [0.021867023675, 0.021533870223, 0.017176355033, 0.006541411270419419], [0.02460203629992428, 0.030658898247331094, 0.029237991497772925, 0.008311825526], [0.02167745797809526, 0.028787303710089078, 0.02288151697682903, 0.0066893162882859705], [0.02460203629992428, 0.030658898247331094, 0.029237991497772925, 0.008311825526], [0.01375456978471739, 0.027414597846718857, 0.022375880411555603, 0.01476913282743879], [0.021643167888342647, 0.025464782028263938, 0.017176635990523256, 0.0066893162882859705], [0.021643167888342647, 0.025096372123888783, 0.017176355033, 0.0066129715609999995], [0.021643167888342647, 0.02346813329728693, 0.017176635990523256, 0.005129629164760743], [0.021643167888342647, 0.02297960740460408, 0.01799897044979077, 0.006678662247912942], [0.021867023675, 0.021049751030611928, 0.017176355033, 0.006541411270419419], [0.021432290794776356, 0.0295495340372893, 0.029237991497772925, 0.008311825526], [0.01375456978471739, 0.02648623953417335, 0.022323162157953238, 0.01476913282743879], [0.021643167888342647, 0.025096372123888783, 0.017176632563682983, 0.006678324425798351], [0.021643167888342647, 0.025096372123888783, 0.017176355033, 0.0066129715609999995], [0.021643167888342647, 0.025096372123888783, 0.017176355033, 0.0066129715609999995], [0.021643167888342647, 0.025096372123888783, 0.017176355033, 0.0066129715609999995], [0.021643167888342647, 0.025921540029090584, 0.01819841020927088, 0.0066129715609999995], [0.01375456978471739, 0.026289459183658683, 0.022375880411555603, 0.01476913282743879], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.021867023675, 0.021533870223, 0.017176355033, 0.006541411270419419], [0.021867023675, 0.021533870223, 0.017176355033, 0.006541411270419419], [0.02460203629992428, 0.030658898247331094, 0.029237991497772925, 0.008311825526], [0.02167745797809526, 0.028787303710089078, 0.022270246497306214, 0.0066893162882859705], [0.01375456978471739, 0.02648623953417335, 0.022323162157953238, 0.01476913282743879], [0.02167745797809526, 0.028787303710089078, 0.02288151697682903, 0.0066893162882859705], [0.021432290794776356, 0.0295495340372893, 0.029237991497772925, 0.008311825526], [0.01375456978471739, 0.027414597846718857, 0.022375880411555603, 0.01476913282743879], [0.021643167888342647, 0.026430202318954647, 0.017176355033, 0.005656734301503121], [0.021867023675, 0.021049751030611928, 0.017176355033, 0.006541411270419419], [0.021643167888342647, 0.02346813329728693, 0.017176635990523256, 0.005129629164760743], [0.021438619623671842, 0.028427140572279515, 0.029237991497772925, 0.008311825526], [0.021867023675, 0.021049751030611928, 0.017176355033, 0.006541411270419419], [0.021643167888342647, 0.025921540029090584, 0.01819841020927088, 0.0066129715609999995], [0.021407223563355032, 0.0306213930304018, 0.029237991497772925, 0.007820606202126379], [0.02143206869500859, 0.02921920218726456, 0.029237991497772925, 0.00840199372759022], [0.021658000083466914, 0.018380370794375406, 0.017159938575632746, 0.006612921458805356], [0.021867023675, 0.021533870223, 0.0171525853450659, 0.006541411270419419], [0.02055511863970574, 0.023909503101865178, 0.01799897044979077, 0.006678662247912942], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.02460203629992428, 0.030658898247331094, 0.029237991497772925, 0.008311825526], [0.02167745797809526, 0.028787303710089078, 0.022270246497306214, 0.0066893162882859705], [0.013780533182873795, 0.027414597846718857, 0.022375880411555603, 0.01476913282743879], [0.021407223563355032, 0.0306213930304018, 0.029237991497772925, 0.007820606202126379], [0.021432290794776356, 0.0295495340372893, 0.029237991497772925, 0.008311825526], [0.02167745797809526, 0.028787303710089078, 0.02288151697682903, 0.0066893162882859705], [0.02055511863970574, 0.023909503101865178, 0.01799897044979077, 0.006678662247912942], [0.01375456978471739, 0.027414597846718857, 0.022375880411555603, 0.01476913282743879], [0.021643167888342647, 0.02346813329728693, 0.017176635990523256, 0.005129629164760743], [0.021658000083466914, 0.018380370794375406, 0.017159938575632746, 0.006612921458805356], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.005347245136564347], [0.02167745797809526, 0.028787303710089078, 0.02936649033668482, 0.005867300219346419], [0.01375456978471739, 0.027414597846718857, 0.022375880411555603, 0.01476913282743879], [0.021438619623671842, 0.029678399455009732, 0.029237991497772925, 0.008311825526], [0.021432062841775693, 0.02921920218726456, 0.029237991497772925, 0.00840199372759022], [0.021438619623671842, 0.02584491510899918, 0.029237991497772925, 0.008311825526], [0.021438619623671842, 0.028427140572279515, 0.029237991497772925, 0.008390845838656399], [0.021438619623671842, 0.028427140572279515, 0.029237991497772925, 0.008311825526], [0.021643167888342647, 0.02346813329728693, 0.017176635990523256, 0.005129629164760743], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.021407223563355032, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02167745797809526, 0.028787303710089078, 0.021280691460574663, 0.0066893162882859705], [0.013614056785665324, 0.027414597846718857, 0.022375880411555603, 0.01476913282743879], [0.021643167888342647, 0.02346813329728693, 0.017176635990523256, 0.005129629164760743], [0.021643167888342647, 0.02346813329728693, 0.017176635990523256, 0.005129629164760743], [0.02167745797809526, 0.028787303710089078, 0.02936649033668482, 0.005867300219346419], [0.021438619623671842, 0.028427140572279515, 0.029237991497772925, 0.008311825526], [0.02167745797809526, 0.02982822394372242, 0.02288151697682903, 0.00680115976419949], [0.021432062841775693, 0.02921920218726456, 0.029237991497772925, 0.00840199372759022], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.005347245136564347], [0.021438619623671842, 0.029678399455009732, 0.029237991497772925, 0.008311825526], [0.02167745797809526, 0.028787303710089078, 0.02288151697682903, 0.0066893162882859705], [0.01375456978471739, 0.027414597846718857, 0.022375880411555603, 0.01476913282743879], [0.021432290794776356, 0.0295495340372893, 0.029237991497772925, 0.008311825526], [0.02167745797809526, 0.02880892063013699, 0.02288151697682903, 0.0066893162882859705], [0.021438619623671842, 0.028427140572279515, 0.029237991497772925, 0.008390845838656399], [0.022986237995690217, 0.028690543914321423, 0.017039858397390084, 0.005129629164760743], [0.01375456978471739, 0.027414597846718857, 0.022375880411555603, 0.01476913282743879], [0.01375456978471739, 0.027414597846718857, 0.022375880411555603, 0.01476913282743879], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.021407223563355032, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.022986237995690217, 0.028690543914321423, 0.017039858397390084, 0.005129629164760743], [0.013614056785665324, 0.027414597846718857, 0.022375880411555603, 0.01476913282743879], [0.02167745797809526, 0.028787303710089078, 0.021375716960835883, 0.006100800633274789], [0.010577996143958608, 0.026577072064306897, 0.02235046414421528, 0.01476913282743879], [0.022986237995690217, 0.028690543914321423, 0.017039858397390084, 0.005129629164760743], [0.02167745797809526, 0.02982822394372242, 0.02288151697682903, 0.00680115976419949], [0.02167745797809526, 0.028787303710089078, 0.02936649033668482, 0.005867300219346419], [0.022986237995690217, 0.028690543914321423, 0.02221274055968192, 0.005129629164760743], [0.021438619623671842, 0.029678399455009732, 0.029237991497772925, 0.008311825526], [0.021432062841775693, 0.02921920218726456, 0.029237991497772925, 0.00840199372759022], [0.021432290794776356, 0.0295495340372893, 0.028866208797735108, 0.006916416456524843], [0.01375456978471739, 0.027414597846718857, 0.022375880411555603, 0.01476913282743879], [0.02167745797809526, 0.02986299379382287, 0.02288151697682903, 0.00680115976419949], [0.02167745797809526, 0.02982822394372242, 0.02365346612569608, 0.005978295149888301], [0.021432290794776356, 0.0295495340372893, 0.029237991497772925, 0.008311825526], [0.01375456978471739, 0.027414597846718857, 0.022375880411555603, 0.01476913282743879], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.005347245136564347], [0.01375456978471739, 0.027414597846718857, 0.022375880411555603, 0.01476913282743879], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.021407223563355032, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02144009979263758, 0.021285826643964157, 0.022795171435353194, 0.006612921458805356], [0.022986237995690217, 0.028743955902401867, 0.02221274055968192, 0.005081540397965795], [0.01375456978471739, 0.028635259441620813, 0.022375880411555603, 0.01476913282743879], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.005347245136564347], [0.021432290794776356, 0.0295495340372893, 0.028866208797735108, 0.006916416456524843], [0.02167745797809526, 0.02903226162565705, 0.028396175845474596, 0.005867300219346419], [0.021438619623671842, 0.029678399455009732, 0.029996600486635074, 0.008311825526], [0.021438619623671842, 0.029678399455009732, 0.029237991497772925, 0.008311825526], [0.021432062841775693, 0.02921920218726456, 0.029237991497772925, 0.00840199372759022], [0.021432290794776356, 0.0295495340372893, 0.028866208797735108, 0.006916416456524843], [0.022986237995690217, 0.028690543914321423, 0.02221274055968192, 0.005129629164760743], [0.02167745797809526, 0.02986299379382287, 0.02288151697682903, 0.00680115976419949], [0.02167745797809526, 0.02982822394372242, 0.02365346612569608, 0.005978295149888301], [0.021432290794776356, 0.0295495340372893, 0.029237991497772925, 0.008311825526], [0.021432062841775693, 0.02921920218726456, 0.029237991497772925, 0.00840199372759022], [0.02167745797809526, 0.030403686382156767, 0.02365346612569608, 0.0056676846370033035], [0.021432290794776356, 0.0295495340372893, 0.029237991497772925, 0.008311825526], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.005347245136564347], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.021203443447649066, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02144009979263758, 0.021285826643964157, 0.022795171435353194, 0.006612921458805356], [0.022986237995690217, 0.028743955902401867, 0.02221274055968192, 0.005081540397965795], [0.01375456978471739, 0.028635259441620813, 0.022375880411555603, 0.0146875005074934], [0.01375456978471739, 0.028635259441620813, 0.022375880411555603, 0.01476913282743879], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.005347245136564347], [0.02167745797809526, 0.02903226162565705, 0.028396175845474596, 0.005867300219346419], [0.021432290794776356, 0.0295495340372893, 0.028866208797735108, 0.006916416456524843], [0.02209763476050199, 0.02163336097576765, 0.0171525853450659, 0.005347245136564347], [0.02167745797809526, 0.030403686382156767, 0.02365346612569608, 0.0056676846370033035], [0.021432290794776356, 0.029304415460557377, 0.028866208797735108, 0.006916416456524843], [0.021432290794776356, 0.031083657504932645, 0.028861857462817563, 0.008311825526], [0.02167745797809526, 0.02889092626831404, 0.02365346612569608, 0.005978295149888301], [0.02143854741771369, 0.0295495340372893, 0.029262315829237658, 0.006916416456524843], [0.021432290794776356, 0.0295495340372893, 0.028866208797735108, 0.006916416456524843], [0.021432062841775693, 0.030123165124437155, 0.029237991497772925, 0.00840199372759022], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.005347245136564347], [0.022986237995690217, 0.028690543914321423, 0.02221274055968192, 0.005129629164760743], [0.021432290794776356, 0.030121860060905007, 0.028866208797735108, 0.006916416456524843], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.021203443447649066, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02144009979263758, 0.021285826643964157, 0.022795171435353194, 0.006612921458805356], [0.021432290794776356, 0.0295495340372893, 0.028866208797735108, 0.006916416456524843], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.005347245136564347], [0.02167745797809526, 0.02903226162565705, 0.028396175845474596, 0.005867300219346419], [0.02490141507208184, 0.02163336097576765, 0.017208220683114582, 0.00815515003884113], [0.021432290794776356, 0.0295495340372893, 0.028866208797735108, 0.006916416456524843], [0.02209763476050199, 0.02163336097576765, 0.0171525853450659, 0.005347245136564347], [0.02167745797809526, 0.02903226162565705, 0.028396175845474596, 0.005867300219346419], [0.02167745797809526, 0.030403686382156767, 0.02365346612569608, 0.0056676846370033035], [0.021432290794776356, 0.031083657504932645, 0.028861857462817563, 0.008311825526], [0.02167745797809526, 0.030312033718357723, 0.02365346612569608, 0.0056676846370033035], [0.021432290794776356, 0.031083657504932645, 0.028861857462817563, 0.008311825526], [0.021432290794776356, 0.030121860060905007, 0.028866208797735108, 0.006916416456524843], [0.02167745797809526, 0.030403686382156767, 0.02365346612569608, 0.0056676846370033035], [0.021432290794776356, 0.0295495340372893, 0.028866208797735108, 0.007055088775759886], [0.021432062841775693, 0.030123165124437155, 0.029237991497772925, 0.00840199372759022], [0.02143854741771369, 0.0295495340372893, 0.029262315829237658, 0.006916416456524843], [0.021432290794776356, 0.029304415460557377, 0.028866208797735108, 0.006916416456524843], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.021203443447649066, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.021203443447649066, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02144009979263758, 0.021285826643964157, 0.022795171435353194, 0.006612921458805356], [0.02490141507208184, 0.02163336097576765, 0.017208220683114582, 0.00815515003884113], [0.02167745797809526, 0.02903226162565705, 0.028396175845474596, 0.005867300219346419], [0.02167745797809526, 0.028597860183417763, 0.023933432348128076, 0.006918763151709622], [0.021432290794776356, 0.031083657504932645, 0.028861857462817563, 0.008311825526], [0.021432290794776356, 0.030121860060905007, 0.028866208797735108, 0.006893635624962583], [0.021432290794776356, 0.029304415460557377, 0.028866208797735108, 0.006916416456524843], [0.021432290794776356, 0.031083657504932645, 0.028736477546058358, 0.008623289568701106], [0.02167745797809526, 0.02903226162565705, 0.028396175845474596, 0.005867300219346419], [0.02167745797809526, 0.030403686382156767, 0.02365346612569608, 0.0056676846370033035], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.005347245136564347], [0.021432290794776356, 0.031083657504932645, 0.028861857462817563, 0.008311825526], [0.02167745797809526, 0.030403686382156767, 0.02335650701744153, 0.0056676846370033035], [0.02167745797809526, 0.030312033718357723, 0.02365346612569608, 0.0056676846370033035], [0.021432290794776356, 0.031083657504932645, 0.028861857462817563, 0.008311825526], [0.021432290794776356, 0.030121860060905007, 0.028866208797735108, 0.006916416456524843], [0.01740640713494191, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.021203443447649066, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.021203443447649066, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02144009979263758, 0.021285826643964157, 0.022795171435353194, 0.006612921458805356], [0.02490141507208184, 0.02163336097576765, 0.017208220683114582, 0.00815515003884113], [0.02167745797809526, 0.02903226162565705, 0.028396175845474596, 0.005867300219346419], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.005347245136564347], [0.021432290794776356, 0.031083657504932645, 0.028736477546058358, 0.008623289568701106], [0.021432290794776356, 0.031083657504932645, 0.028861857462817563, 0.008311825526], [0.021432290794776356, 0.029304415460557377, 0.028866208797735108, 0.006916416456524843], [0.021432290794776356, 0.030121860060905007, 0.028866208797735108, 0.006893635624962583], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.005347245136564347], [0.02167745797809526, 0.02903226162565705, 0.028396175845474596, 0.005867300219346419], [0.021432290794776356, 0.030419808800502542, 0.028861857462817563, 0.008311825526], [0.02167745797809526, 0.030403686382156767, 0.02365346612569608, 0.0056676846370033035], [0.021432290794776356, 0.031083657504932645, 0.028861857462817563, 0.008311825526], [0.021432290794776356, 0.030121860060905007, 0.028866208797735108, 0.006893635624962583], [0.021432290794776356, 0.031083657504932645, 0.028861857462817563, 0.008311825526], [0.01740640713494191, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.01765907071266686, 0.032027108254405585, 0.030038459400667822, 0.007820606202126379], [0.02167745797809526, 0.02903226162565705, 0.028396175845474596, 0.005867300219346419], [0.02144009979263758, 0.021285826643964157, 0.012876672362777286, 0.006612921458805356], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.005347245136564347], [0.02144009979263758, 0.021285826643964157, 0.022795171435353194, 0.006612921458805356], [0.02167745797809526, 0.030403686382156767, 0.02365346612569608, 0.0056676846370033035], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.005347245136564347], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.020233000853400705, 0.029065258862912276, 0.023642647775059065, 0.006612921458805356], [0.021440034602470782, 0.02970053301499826, 0.028866208797735108, 0.005708529890205415], [0.02167745797809526, 0.03152575931600083, 0.028364078148444907, 0.005889182826391952], [0.021432290794776356, 0.029380699808997022, 0.028866208797735108, 0.006916416456524843], [0.021432290794776356, 0.030419808800502542, 0.028861881864098844, 0.006270255292413323], [0.02167745797809526, 0.02903226162565705, 0.028396175845474596, 0.005867300219346419], [0.021643379454030876, 0.030403686382156767, 0.02365346612569608, 0.00818436182812909], [0.02166160477967363, 0.02156353793744081, 0.018539611662375506, 0.008917107780344777], [0.021203443447649066, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.01740640713494191, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02167745797809526, 0.02903226162565705, 0.028396175845474596, 0.005867300219346419], [0.02144009979263758, 0.021285826643964157, 0.012876672362777286, 0.006612921458805356], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.005347245136564347], [0.02144009979263758, 0.021285826643964157, 0.022795171435353194, 0.006612921458805356], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.008796948061830987], [0.02167745797809526, 0.030403686382156767, 0.02365346612569608, 0.0056676846370033035], [0.020233000853400705, 0.029065258862912276, 0.023642647775059065, 0.006612921458805356], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.005347245136564347], [0.02144009979263758, 0.021285826643964157, 0.022795171435353194, 0.006612921458805356], [0.02167745797809526, 0.03152575931600083, 0.028364078148444907, 0.005889182826391952], [0.022161550338997058, 0.021533870223, 0.0171525853450659, 0.005347245136564347], [0.021432290794776356, 0.029380699808997022, 0.028866208797735108, 0.006916416456524843], [0.021440034602470782, 0.02970053301499826, 0.028866208797735108, 0.005708529890205415], [0.01967091668374835, 0.0217348585510034, 0.019164342065715663, 0.006488288720258096], [0.02167745797809526, 0.02903226162565705, 0.028396175845474596, 0.005867300219346419], [0.01740640713494191, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02167745797809526, 0.02903226162565705, 0.028396175845474596, 0.005867300219346419], [0.02144009979263758, 0.021285826643964157, 0.012876672362777286, 0.006612921458805356], [0.02144009979263758, 0.021285826643964157, 0.022795171435353194, 0.006612921458805356], [0.020233000853400705, 0.029065258862912276, 0.023642647775059065, 0.006612921458805356], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.005183410233652236], [0.020233000853400705, 0.029065258862912276, 0.023518749926551764, 0.006612921458805356], [0.02167745797809526, 0.030403686382156767, 0.02365346612569608, 0.0056676846370033035], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.021440034602470782, 0.02970053301499826, 0.028866208797735108, 0.005708529890205415], [0.02167745797809526, 0.02903226162565705, 0.028396175845474596, 0.0058687387573879745], [0.02167745797809526, 0.02903226162565705, 0.028396175845474596, 0.005867300219346419], [0.021440034602470782, 0.02970053301499826, 0.028866208797735108, 0.005708529890205415], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.00514030798011163], [0.02144009979263758, 0.021285826643964157, 0.022795171435353194, 0.006612921458805356], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.021560969078980037, 0.030038870567490394, 0.023806871481240156, 0.006488288720258096], [0.015392677627532412, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02144009979263758, 0.021285826643964157, 0.013649875991904512, 0.006342463461915685], [0.02167745797809526, 0.02903226162565705, 0.028396175845474596, 0.005867300219346419], [0.020818927817468674, 0.0317121841612087, 0.025202851841684235, 0.006488288720258096], [0.02144009979263758, 0.021285826643964157, 0.022795171435353194, 0.006612921458805356], [0.021525034004921235, 0.029065258862912276, 0.027996826037485847, 0.006612921458805356], [0.02167745797809526, 0.02903226162565705, 0.028396175845474596, 0.005867300219346419], [0.02144009979263758, 0.021285826643964157, 0.022795171435353194, 0.006612921458805356], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.005183410233652236], [0.02167745797809526, 0.02903226162565705, 0.028396175845474596, 0.0058687387573879745], [0.02167745797809526, 0.02903226162565705, 0.028396175845474596, 0.005867300219346419], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.00514030798011163], [0.019591617944796483, 0.03286116397781197, 0.029237991497772925, 0.005183410233652236], [0.018240930303691262, 0.030403686382156767, 0.02365346612569608, 0.0056676846370033035], [0.021130548443287572, 0.02970053301499826, 0.028866208797735108, 0.005708529890205415], [0.02144009979263758, 0.021285826643964157, 0.01372386398816422, 0.006612921458805356], [0.01740640713494191, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.02144009979263758, 0.021285826643964157, 0.013134821812848972, 0.006612921458805356], [0.015392677627532412, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.015392677627532412, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02144009979263758, 0.021285826643964157, 0.013649875991904512, 0.006342463461915685], [0.020818927817468674, 0.0317121841612087, 0.025202851841684235, 0.006488288720258096], [0.019473970638231742, 0.031272370948185774, 0.025202851841684235, 0.006488288720258096], [0.020818927817468674, 0.03161554426329904, 0.02521726173254459, 0.006488288720258096], [0.021525034004921235, 0.029065258862912276, 0.027996826037485847, 0.006612921458805356], [0.022097870888875357, 0.021451477813241776, 0.0171525853450659, 0.00514030798011163], [0.02144009979263758, 0.021285826643964157, 0.015005596631107176, 0.006342463461915685], [0.02144009979263758, 0.021285826643964157, 0.022795171435353194, 0.006612921458805356], [0.019591617944796483, 0.03212239752044923, 0.029237991497772925, 0.007820606202126379], [0.019591617944796483, 0.03286116397781197, 0.029237991497772925, 0.005183410233652236], [0.02144009979263758, 0.02130206882142003, 0.01372386398816422, 0.006612921458805356], [0.019591617944796483, 0.032896322262226926, 0.029237991497772925, 0.005183410233652236], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.019591617944796483, 0.03286116397781197, 0.029237991497772925, 0.005183410233652236], [0.021023211796927996, 0.02903226162565705, 0.028396175845474596, 0.0058687387573879745], [0.02144009979263758, 0.021285826643964157, 0.021758879391737714, 0.006612921458805356], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.00514030798011163], [0.015175344621345566, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.015392677627532412, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.015392677627532412, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.019591617944796483, 0.03212239752044923, 0.029237991497772925, 0.007820606202126379], [0.019591617944796483, 0.03286116397781197, 0.029237991497772925, 0.005183410233652236], [0.02144009979263758, 0.021285826643964157, 0.02516654989210908, 0.006612921458805356], [0.02144009979263758, 0.032414411965390855, 0.021783204179036883, 0.006612921458805356], [0.019473970638231742, 0.031272370948185774, 0.025202851841684235, 0.006488288720258096], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.00514030798011163], [0.021525034004921235, 0.029065258862912276, 0.027996826037485847, 0.006612921458805356], [0.019591617944796483, 0.03286116397781197, 0.029237991497772925, 0.005183410233652236], [0.02144009979263758, 0.021285826643964157, 0.015005596631107176, 0.006342463461915685], [0.019591617944796483, 0.03212239752044923, 0.029237991497772925, 0.007820606202126379], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.019591617944796483, 0.03286116397781197, 0.029237991497772925, 0.005183410233652236], [0.02144009979263758, 0.021378075848905605, 0.013640729753672868, 0.0066099367221040305], [0.02144009979263758, 0.031621252039312694, 0.025836194780769324, 0.00634634921247952], [0.02144009979263758, 0.021285826643964157, 0.015005596631107176, 0.006342463461915685], [0.02144009979263758, 0.02130206882142003, 0.01372386398816422, 0.006612921458805356], [0.015175344621345566, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.015392677627532412, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.015392677627532412, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02144009979263758, 0.032414411965390855, 0.021783204179036883, 0.006612921458805356], [0.02144009979263758, 0.031621252039312694, 0.025836194780769324, 0.00634634921247952], [0.019473970638231742, 0.031272370948185774, 0.025202851841684235, 0.006488288720258096], [0.02144009979263758, 0.021285826643964157, 0.02516654989210908, 0.006612921458805356], [0.019591617944796483, 0.03212239752044923, 0.029237991497772925, 0.007820606202126379], [0.02144009979263758, 0.022467941835964577, 0.028790238347330833, 0.006612921458805356], [0.016214315898705506, 0.02152762042756843, 0.01434379189647943, 0.006612921458805356], [0.022097870888875357, 0.021533870223, 0.01725482944386016, 0.00514030798011163], [0.019591617944796483, 0.03212239752044923, 0.029237991497772925, 0.007820606202126379], [0.019591617944796483, 0.03286116397781197, 0.029237991497772925, 0.005183410233652236], [0.021525034004921235, 0.029065258862912276, 0.027996826037485847, 0.006612921458805356], [0.020867087003893933, 0.021378075848905605, 0.024125234166694424, 0.0066099367221040305], [0.019591617944796483, 0.03286116397781197, 0.029237991497772925, 0.005183410233652236], [0.02144009979263758, 0.021285826643964157, 0.015005596631107176, 0.006342463461915685], [0.019591617944796483, 0.03279320952222805, 0.029123421552307953, 0.007820606202126379], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.00514030798011163], [0.015392677627532412, 0.021285826643964157, 0.01295171629905202, 0.006612921458805356], [0.015175344621345566, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.015392677627532412, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.015392677627532412, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02144009979263758, 0.021285826643964157, 0.015005596631107176, 0.006342463461915685], [0.015392677627532412, 0.021285826643964157, 0.01540467005593002, 0.006612921458805356], [0.02144009979263758, 0.031621252039312694, 0.025836194780769324, 0.00634634921247952], [0.02144009979263758, 0.022467941835964577, 0.029380763698483733, 0.006612921458805356], [0.02144009979263758, 0.021285826643964157, 0.02516654989210908, 0.006612921458805356], [0.021525034004921235, 0.029065258862912276, 0.027996826037485847, 0.006612921458805356], [0.02144009979263758, 0.022467941835964577, 0.028790238347330833, 0.006612921458805356], [0.019473970638231742, 0.031272370948185774, 0.025202851841684235, 0.006488288720258096], [0.019591617944796483, 0.03212239752044923, 0.028355769936362515, 0.007820606202126379], [0.016214315898705506, 0.02152762042756843, 0.01434379189647943, 0.006612921458805356], [0.019591617944796483, 0.03286116397781197, 0.029237991497772925, 0.005183410233652236], [0.021190878873365487, 0.032414411965390855, 0.021891134235134675, 0.006614778326201737], [0.022097870888875357, 0.021533870223, 0.01725482944386016, 0.00514030798011163], [0.020867087003893933, 0.021378075848905605, 0.024125234166694424, 0.0066099367221040305], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.00514030798011163], [0.013491908759678543, 0.021285826643964157, 0.011330411892366238, 0.006612921458805356], [0.015392677627532412, 0.021285826643964157, 0.01295171629905202, 0.006612921458805356], [0.015175344621345566, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.015392677627532412, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.015392677627532412, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02144009979263758, 0.021285826643964157, 0.015005596631107176, 0.006342463461915685], [0.02144009979263758, 0.022467941835964577, 0.029380763698483733, 0.006612921458805356], [0.01578747321398217, 0.021285826643964157, 0.01540467005593002, 0.006612921458805356], [0.022097870888875357, 0.0216301565176629, 0.028395705219370116, 0.00514030798011163], [0.015392677627532412, 0.021285826643964157, 0.01540467005593002, 0.006612921458805356], [0.02144009979263758, 0.031621252039312694, 0.025836194780769324, 0.00634634921247952], [0.021190878873365487, 0.032414411965390855, 0.021891134235134675, 0.006614778326201737], [0.02144009979263758, 0.022467941835964577, 0.028790238347330833, 0.006612921458805356], [0.021525034004921235, 0.029065258862912276, 0.027996826037485847, 0.006612921458805356], [0.019591617944796483, 0.03212239752044923, 0.028355769936362515, 0.007820606202126379], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.00514030798011163], [0.020867087003893933, 0.021378075848905605, 0.024125234166694424, 0.005245247371766813], [0.015392677627532412, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.019591617944796483, 0.03286116397781197, 0.029237991497772925, 0.005183410233652236], [0.013491908759678543, 0.021285826643964157, 0.011330411892366238, 0.006612921458805356], [0.015392677627532412, 0.021285826643964157, 0.01295171629905202, 0.006612921458805356], [0.015175344621345566, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02144009979263758, 0.021285826643964157, 0.015005596631107176, 0.006342463461915685], [0.02144009979263758, 0.022467941835964577, 0.029380763698483733, 0.006612921458805356], [0.01578747321398217, 0.021285826643964157, 0.01540467005593002, 0.006612921458805356], [0.021190878873365487, 0.032414411965390855, 0.021891134235134675, 0.006614778326201737], [0.022097870888875357, 0.0216301565176629, 0.028395705219370116, 0.00514030798011163], [0.021525034004921235, 0.029065258862912276, 0.027996826037485847, 0.006612921458805356], [0.02144009979263758, 0.021285826643964157, 0.015005596631107176, 0.006342463461915685], [0.015392677627532412, 0.021285826643964157, 0.01540467005593002, 0.006612921458805356], [0.02144009979263758, 0.022467941835964577, 0.028790238347330833, 0.006612921458805356], [0.020867087003893933, 0.021378075848905605, 0.024125234166694424, 0.005245247371766813], [0.02144009979263758, 0.031621252039312694, 0.025836194780769324, 0.00634634921247952], [0.019591617944796483, 0.03212239752044923, 0.028355769936362515, 0.007820606202126379], [0.02144009979263758, 0.031621252039312694, 0.025836194780769324, 0.00634634921247952], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.00514030798011163], [0.02144009979263758, 0.03189982063475966, 0.025836194780769324, 0.007361773791618299], [0.015175344621345566, 0.02135228721964267, 0.01434379189647943, 0.006612921458805356], [0.013491908759678543, 0.021285826643964157, 0.011330411892366238, 0.006612921458805356], [0.015392677627532412, 0.021285826643964157, 0.01295171629905202, 0.006612921458805356], [0.015175344621345566, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.01578747321398217, 0.021285826643964157, 0.01540467005593002, 0.006612921458805356], [0.02144009979263758, 0.022467941835964577, 0.029380763698483733, 0.006612921458805356], [0.021190878873365487, 0.032414411965390855, 0.021891134235134675, 0.006614778326201737], [0.02144009979263758, 0.02241604231415367, 0.01471040495971938, 0.006648656469275155], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.00514030798011163], [0.022097870888875357, 0.0216301565176629, 0.028395705219370116, 0.00514030798011163], [0.02144009979263758, 0.031621252039312694, 0.025836194780769324, 0.00634634921247952], [0.015392677627532412, 0.021285826643964157, 0.01540467005593002, 0.006612921458805356], [0.021720081226608998, 0.021378075848905605, 0.024462168186497612, 0.005245247371766813], [0.02144009979263758, 0.022467941835964577, 0.028790238347330833, 0.006612921458805356], [0.015175344621345566, 0.021285826643964157, 0.0152526001280243, 0.006612921458805356], [0.019591617944796483, 0.03212239752044923, 0.028355769936362515, 0.007820606202126379], [0.019591617944796483, 0.03212239752044923, 0.029397399300619742, 0.007820606202126379], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02144009979263758, 0.031621252039312694, 0.025836194780769324, 0.00634634921247952], [0.015175344621345566, 0.02135228721964267, 0.01434379189647943, 0.006612921458805356], [0.013491908759678543, 0.021285826643964157, 0.010793770723947134, 0.006612921458805356], [0.013491908759678543, 0.021285826643964157, 0.011330411892366238, 0.006612921458805356], [0.015392677627532412, 0.021285826643964157, 0.01295171629905202, 0.006612921458805356], [0.015175344621345566, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02144009979263758, 0.022467941835964577, 0.029380763698483733, 0.006612921458805356], [0.01578747321398217, 0.021285826643964157, 0.01540467005593002, 0.006612921458805356], [0.021190878873365487, 0.032414411965390855, 0.021891134235134675, 0.006614778326201737], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.00514030798011163], [0.015175344621345566, 0.02135228721964267, 0.01434379189647943, 0.006612921458805356], [0.02144009979263758, 0.031621252039312694, 0.024484345667425654, 0.00634634921247952], [0.015181724132424416, 0.02164519662410026, 0.011330411892366238, 0.006612921458805356], [0.013491908759678543, 0.021286050294003635, 0.011371902156421199, 0.006612921458805356], [0.015392677627532412, 0.021285826643964157, 0.01540467005593002, 0.006612921458805356], [0.022097870888875357, 0.0216301565176629, 0.028395705219370116, 0.00514030798011163], [0.020188819537533584, 0.020797064506348828, 0.018130275138215492, 0.005067257762034565], [0.02144009979263758, 0.031621252039312694, 0.025836194780769324, 0.00634634921247952], [0.021720081226608998, 0.021378075848905605, 0.026567581866728664, 0.005245247371766813], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007808753124501213], [0.021720081226608998, 0.021378075848905605, 0.024462168186497612, 0.005245247371766813], [0.009629943495458518, 0.02135228721964267, 0.01434379189647943, 0.006496983034381279], [0.015175344621345566, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02144009979263758, 0.022467941835964577, 0.029380763698483733, 0.006612921458805356], [0.021190878873365487, 0.032414411965390855, 0.021891134235134675, 0.006614778326201737], [0.013410178383488774, 0.021350592380848153, 0.017081979023962128, 0.006612921458805356], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.00514030798011163], [0.015175344621345566, 0.02135228721964267, 0.01434379189647943, 0.006612921458805356], [0.022097870888875357, 0.0216301565176629, 0.028395705219370116, 0.00514030798011163], [0.02144009979263758, 0.031621252039312694, 0.025836194780769324, 0.00634634921247952], [0.02144009979263758, 0.031621252039312694, 0.024484345667425654, 0.00634634921247952], [0.020188819537533584, 0.020797064506348828, 0.018130275138215492, 0.005067257762034565], [0.021720081226608998, 0.021378075848905605, 0.028642377871141447, 0.005245247371766813], [0.021720081226608998, 0.021378075848905605, 0.026567581866728664, 0.005245247371766813], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007808753124501213], [0.021720081226608998, 0.02160645681634121, 0.024464991780569365, 0.005245247371766813], [0.021720081226608998, 0.021378075848905605, 0.024462168186497612, 0.0051557288644816695], [0.022097870888875357, 0.0216301565176629, 0.028395705219370116, 0.00514030798011163], [0.019591617944796483, 0.03310080242396479, 0.027680851080035866, 0.007820606202126379], [0.02144009979263758, 0.031621252039312694, 0.025836194780769324, 0.00634634921247952], [0.009629943495458518, 0.02135228721964267, 0.01434379189647943, 0.006496983034381279], [0.015175344621345566, 0.021285826643964157, 0.01434379189647943, 0.006612921458805356], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02144009979263758, 0.022467941835964577, 0.029380763698483733, 0.006612921458805356], [0.021190878873365487, 0.032414411965390855, 0.021891134235134675, 0.006614778326201737], [0.013410178383488774, 0.021350592380848153, 0.017081979023962128, 0.006612921458805356], [0.009629943495458518, 0.02135228721964267, 0.01434379189647943, 0.009093109553305171], [0.022097870888875357, 0.021533870223, 0.0171525853450659, 0.00514030798011163], [0.021714037370718604, 0.021378075848905605, 0.026567581866728664, 0.005245247371766813], [0.022097870888875357, 0.0216301565176629, 0.028395705219370116, 0.00514030798011163], [0.019591617944796483, 0.03310080242396479, 0.027680851080035866, 0.007820606202126379], [0.021720081226608998, 0.021158055150919456, 0.028642377871141447, 0.005245247371766813], [0.02144009979263758, 0.031621252039312694, 0.025836194780769324, 0.00634634921247952], [0.020188819537533584, 0.020797064506348828, 0.018130275138215492, 0.005067257762034565], [0.02144009979263758, 0.031621252039312694, 0.024484345667425654, 0.00634634921247952], [0.021720081226608998, 0.021378075848905605, 0.028642377871141447, 0.005245247371766813], [0.021720081226608998, 0.021378075848905605, 0.028642377871141447, 0.005245247371766813], [0.022097870888875357, 0.021614118388081124, 0.029384735779221544, 0.005083971689360524], [0.021720081226608998, 0.021378075848905605, 0.026567581866728664, 0.005245247371766813], [0.021720081226608998, 0.021378075848905605, 0.028642377871141447, 0.00519578945774779], [0.008097631249135585, 0.02135228721964267, 0.01434379189647943, 0.005849131402005698], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.020188819537533584, 0.020797064506348828, 0.018130275138215492, 0.005067257762034565], [0.022097870888875357, 0.021614118388081124, 0.029384735779221544, 0.005083971689360524], [0.02144009979263758, 0.031621252039312694, 0.025836194780769324, 0.00634634921247952], [0.010336141512871844, 0.021350592380848153, 0.017704353625209064, 0.006612921458805356], [0.009629943495458518, 0.021354114986531846, 0.014413875691078899, 0.006367276536668543], [0.009629943495458518, 0.02135228721964267, 0.01434379189647943, 0.009093109553305171], [0.019591617944796483, 0.03310080242396479, 0.027680851080035866, 0.007820606202126379], [0.021713873788201544, 0.021378075848905605, 0.02633506735288809, 0.005245247371766813], [0.0162966415275547, 0.032414411965390855, 0.021891134235134675, 0.00919923180940799], [0.022097870888875357, 0.0216301565176629, 0.028395705219370116, 0.00514030798011163], [0.021714037370718604, 0.021378075848905605, 0.026567581866728664, 0.005245247371766813], [0.021720081226608998, 0.021378075848905605, 0.028642377871141447, 0.005245247371766813], [0.021720081226608998, 0.021158055150919456, 0.028642377871141447, 0.005245247371766813], [0.021720081226608998, 0.021378075848905605, 0.028642377871141447, 0.005245247371766813], [0.020188819537533584, 0.020797064506348828, 0.018130275138215492, 0.005067257762034565], [0.021720081226608998, 0.021378075848905605, 0.028642377871141447, 0.00519578945774779], [0.021720081226608998, 0.021378075848905605, 0.028642377871141447, 0.00519578945774779], [0.021714037370718604, 0.021378075848905605, 0.026567581866728664, 0.005245247371766813], [0.007852664085093393, 0.02135327485223508, 0.01434379189647943, 0.005849131402005698], [0.008097631249135585, 0.02135228721964267, 0.01434379189647943, 0.005849131402005698], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.01886347627987659, 0.021378075848905605, 0.02633506735288809, 0.005245247371766813], [0.010336141512871844, 0.021350592380848153, 0.017704353625209064, 0.006612921458805356], [0.02144009979263758, 0.031621252039312694, 0.025836194780769324, 0.00634634921247952], [0.0162966415275547, 0.032414411965390855, 0.021891134235134675, 0.00919923180940799], [0.022097870888875357, 0.021614118388081124, 0.029384735779221544, 0.005083971689360524], [0.009629943495458518, 0.021354114986531846, 0.014413875691078899, 0.006367276536668543], [0.010336141512871844, 0.021350592380848153, 0.017704353625209064, 0.006612921458805356], [0.02144009979263758, 0.03242231563200544, 0.025836194780769324, 0.006329365890549131], [0.009629943495458518, 0.02135228721964267, 0.01434379189647943, 0.009093109553305171], [0.021720081226608998, 0.021158055150919456, 0.028642377871141447, 0.005245247371766813], [0.022097870888875357, 0.0216301565176629, 0.028395705219370116, 0.00514030798011163], [0.009629943495458518, 0.02135228721964267, 0.01434379189647943, 0.009093109553305171], [0.021713873788201544, 0.021378075848905605, 0.02633506735288809, 0.005245247371766813], [0.021714037370718604, 0.021378075848905605, 0.026567581866728664, 0.005245247371766813], [0.021720081226608998, 0.021378075848905605, 0.028642377871141447, 0.005245247371766813], [0.021720081226608998, 0.021378075848905605, 0.028642377871141447, 0.005245247371766813], [0.021720081226608998, 0.021378075848905605, 0.028642377871141447, 0.005245247371766813], [0.007852664085093393, 0.02135327485223508, 0.01434379189647943, 0.005849131402005698], [0.008097631249135585, 0.02135228721964267, 0.01434379189647943, 0.005849131402005698], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02144009979263758, 0.031621252039312694, 0.025836194780769324, 0.00634634921247952], [0.0162966415275547, 0.032414411965390855, 0.021891134235134675, 0.00919923180940799], [0.010336141512871844, 0.021350592380848153, 0.017914935586448334, 0.006265381471287876], [0.009629943495458518, 0.021354114986531846, 0.014413875691078899, 0.006367276536668543], [0.008097631249135585, 0.0213522903142584, 0.01434379189647943, 0.005849131402005698], [0.02144009979263758, 0.03242231563200544, 0.025836194780769324, 0.006329365890549131], [0.009629943495458518, 0.02135228721964267, 0.01434379189647943, 0.009093109553305171], [0.009629943495458518, 0.02135228721964267, 0.01434379189647943, 0.009093109553305171], [0.0162966415275547, 0.032414411965390855, 0.021891134235134675, 0.00919923180940799], [0.009629943495458518, 0.02135228721964267, 0.01434379189647943, 0.009093109553305171], [0.02144009979263758, 0.03242231563200544, 0.025836194780769324, 0.006329365890549131], [0.010336141512871844, 0.03187693615658554, 0.022072671094815324, 0.006612921458805356], [0.010336141512871844, 0.021350592380848153, 0.017704353625209064, 0.009266604057072027], [0.01886347627987659, 0.021378075848905605, 0.02633506735288809, 0.005245247371766813], [0.010336141512871844, 0.021350592380848153, 0.017704353625209064, 0.006612921458805356], [0.021713873788201544, 0.021378075848905605, 0.02633506735288809, 0.005245247371766813], [0.010336141512871844, 0.021350592380848153, 0.017704353625209064, 0.006612921458805356], [0.008097631249135585, 0.0213522903142584, 0.013734415682953827, 0.005885408159991917], [0.007852664085093393, 0.02135327485223508, 0.01434379189647943, 0.005849131402005698], [0.008097631249135585, 0.02135228721964267, 0.01434379189647943, 0.005849131402005698], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02144009979263758, 0.031621252039312694, 0.025836194780769324, 0.00634634921247952], [0.010337909062256474, 0.019665811000742366, 0.017704353625209064, 0.006612921458805356], [0.010336141512871844, 0.03187693615658554, 0.025111481940323116, 0.006612921458805356], [0.010336141512871844, 0.024733321561927632, 0.017914935586448334, 0.006265381471287876], [0.010336141512871844, 0.021350592380848153, 0.017914935586448334, 0.006265381471287876], [0.008097631249135585, 0.0213522903142584, 0.01434379189647943, 0.005849131402005698], [0.02144009979263758, 0.03242231563200544, 0.025836194780769324, 0.006329365890549131], [0.007852664085093393, 0.021001017797192854, 0.015978291118772137, 0.005849131402005698], [0.010336141512871844, 0.021350592380848153, 0.017704353625209064, 0.005813041370656633], [0.02144009979263758, 0.03242231563200544, 0.025836194780769324, 0.006329365890549131], [0.019633832035910466, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.02144009979263758, 0.03244271586949886, 0.025836194780769324, 0.008157791699648682], [0.010336141512871844, 0.03187693615658554, 0.022072671094815324, 0.006612921458805356], [0.010336141512871844, 0.03187693615658554, 0.022072671094815324, 0.006612921458805356], [0.0162966415275547, 0.032414411965390855, 0.021891134235134675, 0.00919923180940799], [0.010336141512871844, 0.021350592380848153, 0.017704353625209064, 0.009266604057072027], [0.008097631249135585, 0.02135228721964267, 0.01434379189647943, 0.005080769132792791], [0.007852664085093393, 0.02135327485223508, 0.01434379189647943, 0.005849131402005698], [0.008097631249135585, 0.02135228721964267, 0.01434379189647943, 0.005849131402005698], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.010336141512871844, 0.024733321561927632, 0.017914935586448334, 0.006265381471287876], [0.010336141512871844, 0.03187693615658554, 0.025111481940323116, 0.006183420760308459], [0.010337909062256474, 0.019665811000742366, 0.017704353625209064, 0.006612921458805356], [0.01907340804498152, 0.03103746615782982, 0.029237991497772925, 0.007820606202126379], [0.010337909062256474, 0.03226992154331163, 0.018634137800987504, 0.006612921458805356], [0.010336141512871844, 0.03187693615658554, 0.022072671094815324, 0.006612921458805356], [0.010336141512871844, 0.021350592380848153, 0.017704353625209064, 0.005240483859925435], [0.010336141512871844, 0.021350592380848153, 0.017914935586448334, 0.006265381471287876], [0.007852664085093393, 0.021155063282379438, 0.016187823435208186, 0.005695529839549805], [0.007852664085093393, 0.021001017797192854, 0.015978291118772137, 0.005849131402005698], [0.008097631249135585, 0.0213522903142584, 0.01434379189647943, 0.005849131402005698], [0.019633832035910466, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.008097631249135585, 0.02135228721964267, 0.01434379189647943, 0.005849131402005698], [0.019633832035910466, 0.03279320952222805, 0.029386509286429052, 0.007789339956543575], [0.019633832035910466, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.007852664085093393, 0.021001017797192854, 0.015978291118772137, 0.005849131402005698], [0.00513361804469043, 0.02134927080634289, 0.015978291118772137, 0.005849131402005698], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.01907340804498152, 0.03103746615782982, 0.029237991497772925, 0.007820606202126379], [0.010336141512871844, 0.03187693615658554, 0.025111481940323116, 0.006183420760308459], [0.010336141512871844, 0.024733321561927632, 0.017914935586448334, 0.006265381471287876], [0.010336141512871844, 0.02626504370636168, 0.017914935586448334, 0.006265381471287876], [0.010337909062256474, 0.030961471265780496, 0.018634137800987504, 0.006612921458805356], [0.007852664085093393, 0.021155063282379438, 0.016187823435208186, 0.005695529839549805], [0.010337909062256474, 0.03226992154331163, 0.018634137800987504, 0.006612921458805356], [0.010336141512871844, 0.021350592380848153, 0.017704353625209064, 0.005240483859925435], [0.010336141512871844, 0.021350592380848153, 0.019231794126096758, 0.006265381471287876], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.0076551542108829005], [0.010336141512871844, 0.021350592380848153, 0.017704353625209064, 0.005240483859925435], [0.010336141512871844, 0.021350592380848153, 0.017914935586448334, 0.006265381471287876], [0.010336141512871844, 0.03187693615658554, 0.022072671094815324, 0.006612921458805356], [0.019633832035910466, 0.03279320952222805, 0.029386509286429052, 0.007789339956543575], [0.010336141512871844, 0.022058820125778873, 0.018629338320900794, 0.005224959917817353], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.007852664085093393, 0.02008785174913574, 0.01830158570402275, 0.005849131402005698], [0.007852664085093393, 0.02135327485223508, 0.01434379189647943, 0.005849131402005698], [0.00513361804469043, 0.02134927080634289, 0.015978291118772137, 0.005849131402005698], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.010336141512871844, 0.03187693615658554, 0.025111481940323116, 0.006183420760308459], [0.010336141512871844, 0.024733321561927632, 0.017914935586448334, 0.006265381471287876], [0.01907340804498152, 0.03103746615782982, 0.029237991497772925, 0.007463311918166004], [0.01907340804498152, 0.03103746615782982, 0.029237991497772925, 0.007820606202126379], [0.010336141512871844, 0.02626504370636168, 0.017914935586448334, 0.006265381471287876], [0.010850425043178532, 0.028487235470963, 0.017925888076614806, 0.006265381471287876], [0.010336141512871844, 0.022058820125778873, 0.018629338320900794, 0.005224959917817353], [0.010337909062256474, 0.030961471265780496, 0.018634137800987504, 0.006612921458805356], [0.010337909062256474, 0.03226992154331163, 0.018634137800987504, 0.006612921458805356], [0.01907340804498152, 0.03143808380299728, 0.029237991497772925, 0.007820606202126379], [0.007852664085093393, 0.021155063282379438, 0.016187823435208186, 0.005695529839549805], [0.010337909062256474, 0.030961471265780496, 0.018634137800987504, 0.006612921458805356], [0.007852664085093393, 0.02137536604504272, 0.01830158570402275, 0.005849131402005698], [0.010336141512871844, 0.02626504370636168, 0.017914935586448334, 0.006265381471287876], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.0076551542108829005], [0.010336141512871844, 0.021350592380848153, 0.019231794126096758, 0.006265381471287876], [0.010336141512871844, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.007852664085093393, 0.02008785174913574, 0.01830158570402275, 0.005849131402005698], [0.00513361804469043, 0.02134927080634289, 0.015978291118772137, 0.005849131402005698], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.010336141512871844, 0.024733321561927632, 0.017914935586448334, 0.006265381471287876], [0.010336141512871844, 0.03187693615658554, 0.025111481940323116, 0.006183420760308459], [0.010337909062256474, 0.030934713551751293, 0.02927801230679543, 0.006665887155338937], [0.010850425043178532, 0.028487235470963, 0.017925888076614806, 0.006265381471287876], [0.010337909062256474, 0.030961471265780496, 0.018634137800987504, 0.006612921458805356], [0.01907340804498152, 0.03103746615782982, 0.029237991497772925, 0.007463311918166004], [0.010336141512871844, 0.021696112368517945, 0.018629338320900794, 0.005224959917817353], [0.010336141512871844, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.01907340804498152, 0.03103746615782982, 0.029237991497772925, 0.007820606202126379], [0.007852664085093393, 0.02137536604504272, 0.01830158570402275, 0.005849131402005698], [0.01907340804498152, 0.03143808380299728, 0.029237991497772925, 0.007820606202126379], [0.010336141512871844, 0.02626504370636168, 0.017914935586448334, 0.006265381471287876], [0.010336141512871844, 0.02626504370636168, 0.018258731416993364, 0.006265381471287876], [0.010337909062256474, 0.030961471265780496, 0.018634137800987504, 0.006612921458805356], [0.010850425043178532, 0.028487235470963, 0.017925888076614806, 0.006265381471287876], [0.010336141512871844, 0.021350592380848153, 0.019231794126096758, 0.006265381471287876], [0.0050347244811542865, 0.02134927080634289, 0.016633673951296987, 0.005849131402005698], [0.007852664085093393, 0.02008785174913574, 0.01830158570402275, 0.005849131402005698], [0.00513361804469043, 0.02134927080634289, 0.015978291118772137, 0.005849131402005698], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.010336141512871844, 0.03187693615658554, 0.025111481940323116, 0.006183420760308459], [0.010336141512871844, 0.024733321561927632, 0.017914935586448334, 0.006265381471287876], [0.010337909062256474, 0.030201077651193713, 0.01888139478969799, 0.007744660426153558], [0.010336141512871844, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.010337909062256474, 0.030961471265780496, 0.018634137800987504, 0.006612921458805356], [0.010336141512871844, 0.02626504370636168, 0.018258731416993364, 0.006265381471287876], [0.010336141512871844, 0.021696112368517945, 0.018629338320900794, 0.005224959917817353], [0.007852664085093393, 0.02137536604504272, 0.01830158570402275, 0.005849131402005698], [0.010850425043178532, 0.028487235470963, 0.017891250548956173, 0.006265381471287876], [0.010336141512871844, 0.021350592380848153, 0.019231794126096758, 0.006265381471287876], [0.010850425043178532, 0.028487235470963, 0.017925888076614806, 0.006265381471287876], [0.010850425043178532, 0.02849930620949607, 0.017925888076614806, 0.006265381471287876], [0.007852664085093393, 0.02188859882171748, 0.01830158570402275, 0.005849131402005698], [0.010336141512871844, 0.02508196213282648, 0.01789287170690406, 0.006450751165633481], [0.0050347244811542865, 0.02134927080634289, 0.016633673951296987, 0.005849131402005698], [0.007852664085093393, 0.02008785174913574, 0.01830158570402275, 0.005849131402005698], [0.010020184809922247, 0.02626504370636168, 0.017914935586448334, 0.006195645643767024], [0.010850425043178532, 0.028487235470963, 0.017925888076614806, 0.006265381471287876], [0.00513361804469043, 0.02134927080634289, 0.015978291118772137, 0.005849131402005698], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.010336141512871844, 0.03187693615658554, 0.025111481940323116, 0.006183420760308459], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.010336141512871844, 0.024733321561927632, 0.017914935586448334, 0.006265381471287876], [0.007852664085093393, 0.02008785174913574, 0.01830158570402275, 0.005849131402005698], [0.010850425043178532, 0.028487235470963, 0.018629112974608765, 0.006265381471287876], [0.010336141512871844, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.010020184809922247, 0.02626504370636168, 0.017914935586448334, 0.006195645643767024], [0.010337909062256474, 0.030201077651193713, 0.01888139478969799, 0.007744660426153558], [0.010336141512871844, 0.021350592380848153, 0.019231794126096758, 0.006265381471287876], [0.010337909062256474, 0.030961471265780496, 0.018634137800987504, 0.006612921458805356], [0.009188467144015057, 0.021696112368517945, 0.018629338320900794, 0.005204035425035198], [0.010336141512871844, 0.02508196213282648, 0.01789287170690406, 0.006450751165633481], [0.010336141512871844, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.010373637209749294, 0.030201077651193713, 0.01888139478969799, 0.007744660426153558], [0.0050347244811542865, 0.02134927080634289, 0.016633673951296987, 0.005849131402005698], [0.010336141512871844, 0.03187693615658554, 0.024967783850931707, 0.006182781554520372], [0.007852664085093393, 0.02137536604504272, 0.01830158570402275, 0.005849131402005698], [0.007852664085093393, 0.02188859882171748, 0.01830158570402275, 0.005849131402005698], [0.00513361804469043, 0.02134927080634289, 0.015978291118772137, 0.005464940900052216], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.010336141512871844, 0.03187693615658554, 0.025111481940323116, 0.006183420760308459], [0.010336141512871844, 0.03187693615658554, 0.025357180181193137, 0.006183420760308459], [0.010850425043178532, 0.028487235470963, 0.018629112974608765, 0.006265381471287876], [0.010020184809922247, 0.02626504370636168, 0.017914935586448334, 0.006195645643767024], [0.007852664085093393, 0.02008785174913574, 0.01830158570402275, 0.005849131402005698], [0.010336141512871844, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.010337909062256474, 0.030961471265780496, 0.018634137800987504, 0.006612921458805356], [0.010336141512871844, 0.021350592380848153, 0.019231794126096758, 0.006265381471287876], [0.010373637209749294, 0.030201077651193713, 0.01888139478969799, 0.007744660426153558], [0.010337909062256474, 0.031263164357783715, 0.018634137800987504, 0.006524254323797354], [0.010336141512871844, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.0050347244811542865, 0.02134927080634289, 0.016633673951296987, 0.005849131402005698], [0.010336141512871844, 0.031000969341455804, 0.024967783850931707, 0.005254865565256831], [0.007852664085093393, 0.02188859882171748, 0.01830158570402275, 0.005849131402005698], [0.007852664085093393, 0.02137536604504272, 0.018056105073170365, 0.005849131402005698], [0.007852664085093393, 0.02137536604504272, 0.01830158570402275, 0.005849131402005698], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.00513361804469043, 0.02134927080634289, 0.015978291118772137, 0.005464940900052216], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.010020184809922247, 0.02626504370636168, 0.017914935586448334, 0.006195645643767024], [0.010336141512871844, 0.03187693615658554, 0.025357180181193137, 0.006183420760308459], [0.010336141512871844, 0.02337307429707984, 0.018629338320900794, 0.005217135689917205], [0.010337909062256474, 0.030961471265780496, 0.018634137800987504, 0.006612921458805356], [0.019277006377137636, 0.03315588095053574, 0.029237991497772925, 0.0059710846084598795], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.010336141512871844, 0.03187693615658554, 0.025111481940323116, 0.006183420760308459], [0.007852664085093393, 0.02188859882171748, 0.01830158570402275, 0.005849131402005698], [0.0050347244811542865, 0.02134927080634289, 0.016633673951296987, 0.005849131402005698], [0.010337909062256474, 0.030961471265780496, 0.018634137800987504, 0.006612921458805356], [0.010336141512871844, 0.031000969341455804, 0.024967783850931707, 0.005254865565256831], [0.007852664085093393, 0.03079222972053654, 0.018391664521737915, 0.005728249940374621], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.007852664085093393, 0.02137536604504272, 0.01830158570402275, 0.0051451220775079], [0.010373637209749294, 0.030201077651193713, 0.01888139478969799, 0.007744660426153558], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.010336141512871844, 0.030892210573510347, 0.024967783850931707, 0.005016216725831473], [0.007852664085093393, 0.02008785174913574, 0.01830158570402275, 0.005849131402005698], [0.00513361804469043, 0.02134927080634289, 0.015978291118772137, 0.005464940900052216], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.010336141512871844, 0.03187693615658554, 0.025357180181193137, 0.006183420760308459], [0.010337909062256474, 0.030961471265780496, 0.018634137800987504, 0.006612921458805356], [0.010336141512871844, 0.02337307429707984, 0.018629338320900794, 0.005217135689917205], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.010336141512871844, 0.02337307429707984, 0.01903085726852411, 0.005217135689917205], [0.010336141512871844, 0.03187693615658554, 0.025111481940323116, 0.006183420760308459], [0.019277006377137636, 0.03315588095053574, 0.029237991497772925, 0.0059710846084598795], [0.007852664085093393, 0.02008785174913574, 0.01830158570402275, 0.005849131402005698], [0.009407599128708734, 0.02045745558239832, 0.019777128694857676, 0.0051451220775079], [0.010336141512871844, 0.031000969341455804, 0.024967783850931707, 0.005254865565256831], [0.008676603615695064, 0.02337307429707984, 0.01661063414601361, 0.005224959917817353], [0.010373637209749294, 0.030201077651193713, 0.01888139478969799, 0.007744660426153558], [0.007852664085093393, 0.03079222972053654, 0.018391664521737915, 0.005728249940374621], [0.019277006377137636, 0.03315588095053574, 0.029237991497772925, 0.0059710846084598795], [0.007852664085093393, 0.020042683821005644, 0.018300935593983086, 0.005849131402005698], [0.008813959103677256, 0.026221922329841843, 0.01815104509238748, 0.006195645643767024], [0.010336141512871844, 0.03187693615658554, 0.025111481940323116, 0.006183420760308459], [0.00513361804469043, 0.02134927080634289, 0.015978291118772137, 0.005464940900052216], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.011713640821069456, 0.03187693615658554, 0.025746437721910164, 0.006183420760308459], [0.007852664085093393, 0.03079222972053654, 0.018391664521737915, 0.005728249940374621], [0.010336141512871844, 0.03187693615658554, 0.025357180181193137, 0.006183420760308459], [0.010337909062256474, 0.030961471265780496, 0.018634137800987504, 0.006612921458805356], [0.009115451874743474, 0.020042683821005644, 0.018300935593983086, 0.005849131402005698], [0.010336141512871844, 0.02337307429707984, 0.018629338320900794, 0.005217135689917205], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.008676603615695064, 0.02337307429707984, 0.01661063414601361, 0.005224959917817353], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.008813959103677256, 0.026221922329841843, 0.01815104509238748, 0.006195645643767024], [0.019277006377137636, 0.03315588095053574, 0.029237991497772925, 0.0059710846084598795], [0.010336141512871844, 0.031000969341455804, 0.024967783850931707, 0.005254865565256831], [0.010336141512871844, 0.03187693615658554, 0.025111481940323116, 0.006183420760308459], [0.019277006377137636, 0.03315588095053574, 0.029237991497772925, 0.0059710846084598795], [0.010373637209749294, 0.030201077651193713, 0.019743608935767213, 0.007744660426153558], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.010336141512871844, 0.03187693615658554, 0.025111481940323116, 0.006183420760308459], [0.008149264514959524, 0.03309111021668326, 0.023331269960564453, 0.006183420760308459], [0.00513361804469043, 0.02134927080634289, 0.015978291118772137, 0.005464940900052216], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.011713640821069456, 0.03187693615658554, 0.025746437721910164, 0.006183420760308459], [0.008813959103677256, 0.026221922329841843, 0.01815104509238748, 0.006195645643767024], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.019277006377137636, 0.03315588095053574, 0.029237991497772925, 0.0059710846084598795], [0.010336141512871844, 0.02337307429707984, 0.018629338320900794, 0.005217135689917205], [0.010336141512871844, 0.03187693615658554, 0.025357180181193137, 0.006183420760308459], [0.007852664085093393, 0.03079222972053654, 0.018391664521737915, 0.005728249940374621], [0.008676603615695064, 0.0328173459623586, 0.023346346335358027, 0.005960622620003911], [0.007852664085093393, 0.030761880886893286, 0.018391664521737915, 0.005743975010609918], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.010336141512871844, 0.03187693615658554, 0.025357180181193137, 0.006183420760308459], [0.019277006377137636, 0.03315588095053574, 0.029237991497772925, 0.0059710846084598795], [0.010336141512871844, 0.03187693615658554, 0.025111481940323116, 0.006183420760308459], [0.010336141512871844, 0.031000969341455804, 0.024967783850931707, 0.005254865565256831], [0.019277006377137636, 0.03315588095053574, 0.029237991497772925, 0.0059710846084598795], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.010336141512871844, 0.03187693615658554, 0.025111481940323116, 0.006183420760308459], [0.010336141512871844, 0.03187693615658554, 0.025111481940323116, 0.006183420760308459], [0.00513361804469043, 0.02134927080634289, 0.01401935227497709, 0.0053969058960827735], [0.00513361804469043, 0.02134927080634289, 0.015978291118772137, 0.005464940900052216], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.019591617944796483, 0.03405533490008292, 0.029237991497772925, 0.010152947909047371], [0.011713640821069456, 0.03240627241289958, 0.025746437721910164, 0.006183420760308459], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.007852664085093393, 0.030761880886893286, 0.018391664521737915, 0.005743975010609918], [0.011713640821069456, 0.03187693615658554, 0.025746437721910164, 0.006183420760308459], [0.007852664085093393, 0.03079222972053654, 0.018391664521737915, 0.005728249940374621], [0.008676603615695064, 0.0328173459623586, 0.023346346335358027, 0.005960622620003911], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.008813959103677256, 0.026221922329841843, 0.01815104509238748, 0.006195645643767024], [0.008813959103677256, 0.026221922329841843, 0.018449372622833523, 0.006994330996255793], [0.019277006377137636, 0.03315588095053574, 0.029237991497772925, 0.0059710846084598795], [0.008813959103677256, 0.026221922329841843, 0.01815104509238748, 0.006195645643767024], [0.010336141512871844, 0.03187693615658554, 0.025357180181193137, 0.006183420760308459], [0.010336141512871844, 0.03187693615658554, 0.025357180181193137, 0.006183420760308459], [0.010336141512871844, 0.03187693615658554, 0.025111481940323116, 0.005832879196271278], [0.012061957377374043, 0.03187693615658554, 0.025111481940323116, 0.00504531008357661], [0.019277006377137636, 0.03315588095053574, 0.029237991497772925, 0.0059710846084598795], [0.00513361804469043, 0.02127740513304877, 0.01401935227497709, 0.0053969058960827735], [0.00513361804469043, 0.02134927080634289, 0.01401935227497709, 0.0053969058960827735], [0.00513361804469043, 0.02134927080634289, 0.015978291118772137, 0.005464940900052216], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.019591617944796483, 0.03405533490008292, 0.029237991497772925, 0.010152947909047371], [0.019591617944796483, 0.03405533490008292, 0.029237991497772925, 0.010152947909047371], [0.011713640821069456, 0.03240627241289958, 0.025746437721910164, 0.006183420760308459], [0.019277006377137636, 0.03315588095053574, 0.029237991497772925, 0.0059710846084598795], [0.008676603615695064, 0.02337307429707984, 0.015893039072910085, 0.005224959917817353], [0.007852664085093393, 0.030761880886893286, 0.018391664521737915, 0.005743975010609918], [0.007852664085093393, 0.03079222972053654, 0.018391664521737915, 0.005728249940374621], [0.008813959103677256, 0.026221922329841843, 0.01815104509238748, 0.006195645643767024], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.008676603615695064, 0.0328173459623586, 0.023346346335358027, 0.005960622620003911], [0.010336141512871844, 0.03187693615658554, 0.025357180181193137, 0.006183420760308459], [0.012061957377374043, 0.03187693615658554, 0.025111481940323116, 0.00504531008357661], [0.019277006377137636, 0.03315588095053574, 0.029237991497772925, 0.0059710846084598795], [0.008813959103677256, 0.026221922329841843, 0.01815104509238748, 0.006195645643767024], [0.010336141512871844, 0.03187693615658554, 0.025357180181193137, 0.006183420760308459], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.00513361804469043, 0.02127740513304877, 0.01401935227497709, 0.0053969058960827735], [0.00513361804469043, 0.02134927080634289, 0.01401935227497709, 0.0053969058960827735], [0.00513361804469043, 0.02134927080634289, 0.015978291118772137, 0.005464940900052216], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.019591617944796483, 0.03405533490008292, 0.029237991497772925, 0.010152947909047371], [0.019591617944796483, 0.03405533490008292, 0.029237991497772925, 0.010152947909047371], [0.019591617944796483, 0.03405533490008292, 0.029855418050037247, 0.010152947909047371], [0.008813959103677256, 0.026221922329841843, 0.01815104509238748, 0.006195645643767024], [0.011713640821069456, 0.03240627241289958, 0.0293658715253199, 0.007783782881697367], [0.011713640821069456, 0.03240627241289958, 0.025746437721910164, 0.006183420760308459], [0.008813959103677256, 0.026221922329841843, 0.01815104509238748, 0.006195645643767024], [0.008676603615695064, 0.0328173459623586, 0.023346346335358027, 0.005960622620003911], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.010336141512871844, 0.03187693615658554, 0.025357180181193137, 0.006183420760308459], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.012061957377374043, 0.03187693615658554, 0.025111481940323116, 0.00504531008357661], [0.010336141512871844, 0.03187693615658554, 0.025357180181193137, 0.006183420760308459], [0.005072598721302958, 0.021257191955174753, 0.01858255399752829, 0.0053969058960827735], [0.005505189486018053, 0.0328173459623586, 0.02325042341076879, 0.005876374594149182], [0.019591617944796483, 0.0343798910613248, 0.029237991497772925, 0.009699397588280763], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.00513361804469043, 0.02134927080634289, 0.01401935227497709, 0.0053969058960827735], [0.00513361804469043, 0.02134927080634289, 0.015978291118772137, 0.005464940900052216], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.019591617944796483, 0.03405533490008292, 0.029237991497772925, 0.010152947909047371], [0.019591617944796483, 0.03405533490008292, 0.029237991497772925, 0.010152947909047371], [0.019591617944796483, 0.03405533490008292, 0.029855418050037247, 0.010152947909047371], [0.011713640821069456, 0.03240627241289958, 0.0293658715253199, 0.007783782881697367], [0.005177244797677447, 0.03153024194530667, 0.02556473471108739, 0.005464940900052216], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.0062257556513353966], [0.008813959103677256, 0.026221922329841843, 0.01815104509238748, 0.006195645643767024], [0.011713640821069456, 0.03240627241289958, 0.025746437721910164, 0.006183420760308459], [0.005072598721302958, 0.021257191955174753, 0.01858255399752829, 0.0053969058960827735], [0.005505189486018053, 0.0328173459623586, 0.02325042341076879, 0.005876374594149182], [0.010336141512871844, 0.03187693615658554, 0.025357180181193137, 0.006183420760308459], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.008813959103677256, 0.026221922329841843, 0.01815104509238748, 0.006195645643767024], [0.019591617944796483, 0.0343798910613248, 0.029237991497772925, 0.009699397588280763], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.00513361804469043, 0.02134927080634289, 0.01401935227497709, 0.0053969058960827735], [0.00513361804469043, 0.02134927080634289, 0.015978291118772137, 0.005464940900052216], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.019591617944796483, 0.03405533490008292, 0.029855418050037247, 0.010152947909047371], [0.019591617944796483, 0.03405533490008292, 0.030769455153419865, 0.010152947909047371], [0.011713640821069456, 0.03240627241289958, 0.0293658715253199, 0.007783782881697367], [0.010336141512871844, 0.03187693615658554, 0.025357180181193137, 0.006183420760308459], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.008813959103677256, 0.026221922329841843, 0.01858517592339679, 0.00590933930128206], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.011713640821069456, 0.03240627241289958, 0.025746437721910164, 0.006183420760308459], [0.005177244797677447, 0.03153024194530667, 0.02556473471108739, 0.005464940900052216], [0.005072598721302958, 0.021257191955174753, 0.01858255399752829, 0.0053969058960827735], [0.005505189486018053, 0.0328173459623586, 0.02325042341076879, 0.005876374594149182], [0.006158886732236927, 0.026221922329841843, 0.018245670465015046, 0.006195645643767024], [0.005072598721302958, 0.021257191955174753, 0.01868125389325286, 0.0053969058960827735], [0.008676603615695064, 0.023054628406892045, 0.018629338320900794, 0.005224959917817353], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010146993788951312], [0.0058685452594453204, 0.02100540640213748, 0.010023375919230523, 0.00521158731479381], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.00513361804469043, 0.02134927080634289, 0.01401935227497709, 0.0053969058960827735], [0.00513361804469043, 0.02134927080634289, 0.015978291118772137, 0.005464940900052216], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.019591617944796483, 0.03405533490008292, 0.029855418050037247, 0.010152947909047371], [0.019591617944796483, 0.03405533490008292, 0.030769455153419865, 0.010152947909047371], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.005505189486018053, 0.0328173459623586, 0.02325042341076879, 0.005876374594149182], [0.011713640821069456, 0.03240627241289958, 0.025746437721910164, 0.006183420760308459], [0.011713640821069456, 0.03240627241289958, 0.0293658715253199, 0.007783782881697367], [0.005072598721302958, 0.021257191955174753, 0.01868125389325286, 0.0053969058960827735], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010146993788951312], [0.010434926148828991, 0.02134927080634289, 0.02514934143318481, 0.005363268322638707], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.010772961315373494, 0.030895328122972618, 0.025357180181193137, 0.006183420760308459], [0.00560760024692121, 0.03153024194530667, 0.02556473471108739, 0.005464940900052216], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.005177244797677447, 0.031746511084606056, 0.029170121740130246, 0.005464940900052216], [0.0058685452594453204, 0.02099749135971138, 0.011955587534010458, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.010023375919230523, 0.00521158731479381], [0.0058685452594453204, 0.02100540640213748, 0.010023375919230523, 0.00521158731479381], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.00513361804469043, 0.02134927080634289, 0.01401935227497709, 0.0053969058960827735], [0.00513361804469043, 0.02134927080634289, 0.015978291118772137, 0.005464940900052216], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.019591617944796483, 0.03405533490008292, 0.029855418050037247, 0.010152947909047371], [0.019591617944796483, 0.03405533490008292, 0.030769455153419865, 0.010152947909047371], [0.011713640821069456, 0.03240627241289958, 0.025746437721910164, 0.006183420760308459], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010146993788951312], [0.005405962936660657, 0.0328173459623586, 0.02325042341076879, 0.005876374594149182], [0.00560760024692121, 0.03153024194530667, 0.02556473471108739, 0.005464940900052216], [0.005072598721302958, 0.022653627453415058, 0.018655206276993144, 0.007174170129260174], [0.010434926148828991, 0.02134927080634289, 0.02514934143318481, 0.005363268322638707], [0.010772961315373494, 0.030223387002050916, 0.025357180181193137, 0.006183420760308459], [0.005177244797677447, 0.031746511084606056, 0.029170121740130246, 0.005464940900052216], [0.005505189486018053, 0.0328173459623586, 0.02325042341076879, 0.005876374594149182], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.010772961315373494, 0.030895328122972618, 0.025357180181193137, 0.006183420760308459], [0.0058685452594453204, 0.02100540640213748, 0.010023375919230523, 0.00521158731479381], [0.0058685452594453204, 0.02100540640213748, 0.010023375919230523, 0.00521158731479381], [0.0058685452594453204, 0.02100540640213748, 0.010023375919230523, 0.00521158731479381], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.00513361804469043, 0.02134927080634289, 0.015978291118772137, 0.005464940900052216], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.019591617944796483, 0.03405533490008292, 0.029855418050037247, 0.010152947909047371], [0.017649176838861812, 0.03443121543494573, 0.031168362074071858, 0.010152947909047371], [0.011713640821069456, 0.03240627241289958, 0.025746437721910164, 0.006183420760308459], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.010434926148828991, 0.02134927080634289, 0.02514934143318481, 0.005363268322638707], [0.005072598721302958, 0.022653627453415058, 0.018655206276993144, 0.007174170129260174], [0.005405962936660657, 0.0328173459623586, 0.02325042341076879, 0.005876374594149182], [0.005505189486018053, 0.0328173459623586, 0.02325042341076879, 0.005876374594149182], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010146993788951312], [0.008676603615695064, 0.03166101839250334, 0.01877032737276765, 0.005223562058813741], [0.00560760024692121, 0.03153024194530667, 0.02556473471108739, 0.005464940900052216], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.005976699353909485, 0.0328173459623586, 0.02610553428814283, 0.005876374594149182], [0.019591617944796483, 0.03405533490008292, 0.029855418050037247, 0.010152947909047371], [0.0058685452594453204, 0.02100540640213748, 0.010023375919230523, 0.00521158731479381], [0.0058685452594453204, 0.02100540640213748, 0.010023375919230523, 0.00521158731479381], [0.0058685452594453204, 0.02100540640213748, 0.010023375919230523, 0.00521158731479381], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.017649176838861812, 0.03416212693961355, 0.031168362074071858, 0.010152947909047371], [0.011713640821069456, 0.03240627241289958, 0.025746437721910164, 0.006183420760308459], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.010434926148828991, 0.02134927080634289, 0.02514934143318481, 0.005363268322638707], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010146993788951312], [0.005072598721302958, 0.022653627453415058, 0.018655206276993144, 0.007174170129260174], [0.008676603615695064, 0.03166101839250334, 0.01877032737276765, 0.005223562058813741], [0.005505189486018053, 0.0328173459623586, 0.026806110878316577, 0.0052387177594520025], [0.005405962936660657, 0.0328173459623586, 0.02325042341076879, 0.005876374594149182], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.005505189486018053, 0.0328173459623586, 0.02325042341076879, 0.005876374594149182], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.00560760024692121, 0.03153024194530667, 0.02556473471108739, 0.005464940900052216], [0.0056066090888062005, 0.03225006721807081, 0.02593678753349369, 0.005876374594149182], [0.00513361804469043, 0.02134927080634289, 0.015978291118772137, 0.005092134940348099], [0.0058685452594453204, 0.02100540640213748, 0.010023375919230523, 0.00521158731479381], [0.0058685452594453204, 0.02100540640213748, 0.010023375919230523, 0.00521158731479381], [0.0058685452594453204, 0.02100540640213748, 0.010023375919230523, 0.00521158731479381], [0.0058685452594453204, 0.02100540640213748, 0.010023375919230523, 0.00521158731479381], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.017649176838861812, 0.03416212693961355, 0.031168362074071858, 0.010152947909047371], [0.017649176838861812, 0.03416212693961355, 0.031168362074071858, 0.010152947909047371], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010146993788951312], [0.010434926148828991, 0.02134927080634289, 0.02514934143318481, 0.005363268322638707], [0.005505189486018053, 0.0328173459623586, 0.026806110878316577, 0.0052387177594520025], [0.008676603615695064, 0.031504543344053934, 0.0176883555621922, 0.005223562058813741], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.005076410639324841, 0.022653627453415058, 0.019123090088441166, 0.007174170129260174], [0.005072598721302958, 0.022653627453415058, 0.018655206276993144, 0.007174170129260174], [0.011713640821069456, 0.032378074062711114, 0.03135585591411297, 0.010146993788951312], [0.008676603615695064, 0.03166101839250334, 0.01877032737276765, 0.005223562058813741], [0.0056066090888062005, 0.03225006721807081, 0.02593678753349369, 0.005876374594149182], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011955587534010458, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.017649176838861812, 0.03416212693961355, 0.031168362074071858, 0.010152947909047371], [0.017649176838861812, 0.03416212693961355, 0.031168362074071858, 0.010152947909047371], [0.019336445662171253, 0.03416212693961355, 0.031168362074071858, 0.011159589498857689], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.008306793799146847, 0.03282992806133637, 0.029448118839112032, 0.00568457411249715], [0.008676603615695064, 0.03166101839250334, 0.01877032737276765, 0.005223562058813741], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010146993788951312], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010146993788951312], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.008676603615695064, 0.023754930504569987, 0.018629338320900794, 0.005224959917817353], [0.005076410639324841, 0.022653627453415058, 0.019123090088441166, 0.007174170129260174], [0.011713640821069456, 0.032378074062711114, 0.03135585591411297, 0.010146993788951312], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.017649176838861812, 0.03416212693961355, 0.031168362074071858, 0.010152947909047371], [0.005072598721302958, 0.022653627453415058, 0.018655206276993144, 0.007174170129260174], [0.008676603615695064, 0.03166101839250334, 0.01877032737276765, 0.005223562058813741], [0.0058685452594453204, 0.02100540640213748, 0.018527810426278015, 0.007251444023643747], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011955587534010458, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.017649176838861812, 0.03416212693961355, 0.031168362074071858, 0.010152947909047371], [0.017649176838861812, 0.03416212693961355, 0.031168362074071858, 0.010152947909047371], [0.019336445662171253, 0.03416212693961355, 0.031168362074071858, 0.011159589498857689], [0.008676603615695064, 0.03166101839250334, 0.01877032737276765, 0.005223562058813741], [0.008306793799146847, 0.03282992806133637, 0.029448118839112032, 0.00568457411249715], [0.008676603615695064, 0.023754930504569987, 0.018629338320900794, 0.005224959917817353], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010146993788951312], [0.008306793799146847, 0.03282992806133637, 0.029448118839112032, 0.00568457411249715], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.005072598721302958, 0.022653627453415058, 0.018655206276993144, 0.007174170129260174], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010146993788951312], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010146993788951312], [0.005076410639324841, 0.022653627453415058, 0.019123090088441166, 0.007174170129260174], [0.017649176838861812, 0.03416212693961355, 0.031168362074071858, 0.010152947909047371], [0.017649176838861812, 0.03416212693961355, 0.031168362074071858, 0.010152947909047371], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011955587534010458, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.017031319276910872, 0.03349473704721639, 0.03135585591411297, 0.010892812681816539], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010146993788951312], [0.008676603615695064, 0.02337307429707984, 0.018629338320900794, 0.005224959917817353], [0.008306793799146847, 0.03282992806133637, 0.029448118839112032, 0.00564867844681966], [0.005076410639324841, 0.022653627453415058, 0.019123090088441166, 0.007174170129260174], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010146993788951312], [0.008306793799146847, 0.03282992806133637, 0.029448118839112032, 0.00568457411249715], [0.008676603615695064, 0.023754930504569987, 0.018629338320900794, 0.005224959917817353], [0.008306793799146847, 0.03282992806133637, 0.029448118839112032, 0.00568457411249715], [0.008306793799146847, 0.03282992806133637, 0.029448118839112032, 0.010372673723541797], [0.0058685452594453204, 0.019924786260881105, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.020962835873204952, 0.011955587534010458, 0.008162785679510096], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010146993788951312], [0.005069771151745019, 0.022653627453415058, 0.018655206276993144, 0.007174170129260174], [0.008676603615695064, 0.023754930504569987, 0.018629338320900794, 0.005224959917817353], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011955587534010458, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.017031319276910872, 0.03349473704721639, 0.03135585591411297, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.007453600928282454, 0.03282992806133637, 0.029448118839112032, 0.00568457411249715], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.011713640821069456, 0.03240026594107086, 0.03135585591411297, 0.010932385080500253], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010146993788951312], [0.008306793799146847, 0.03282992806133637, 0.029448118839112032, 0.010372673723541797], [0.0058685452594453204, 0.020962835873204952, 0.011955587534010458, 0.008162785679510096], [0.011713640821069456, 0.03238092958307968, 0.03135585591411297, 0.010146993788951312], [0.008306793799146847, 0.03282992806133637, 0.029448118839112032, 0.00564867844681966], [0.0058685452594453204, 0.019924786260881105, 0.011410484359548091, 0.005492685170680188], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010146993788951312], [0.008306793799146847, 0.03282992806133637, 0.029448118839112032, 0.00568457411249715], [0.008306793799146847, 0.03282992806133637, 0.029448118839112032, 0.00568457411249715], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011955587534010458, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.017031319276910872, 0.03349473704721639, 0.03135585591411297, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.011699144222216028, 0.03282992806133637, 0.029493506297000423, 0.005720905521099993], [0.0058685452594453204, 0.020962835873204952, 0.011955587534010458, 0.008162785679510096], [0.008306793799146847, 0.03282992806133637, 0.028536304793467953, 0.005322001932977224], [0.011639448627040415, 0.033441879561074264, 0.03149230099758477, 0.010135567292587268], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010146993788951312], [0.019591617944796483, 0.03261051527105353, 0.029449293583357298, 0.010819492948497086], [0.0058685452594453204, 0.019924786260881105, 0.011410484359548091, 0.005492685170680188], [0.011713640821069456, 0.03238092958307968, 0.03135585591411297, 0.00828793940970218], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010146993788951312], [0.008306793799146847, 0.03282992806133637, 0.029448118839112032, 0.00564867844681966], [0.008306793799146847, 0.03282992806133637, 0.029448118839112032, 0.00568457411249715], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011955587534010458, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.017031319276910872, 0.03349473704721639, 0.03135585591411297, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.0058685452594453204, 0.020962835873204952, 0.011955587534010458, 0.008162785679510096], [0.011428302805086119, 0.0327076256644521, 0.011410484359548091, 0.005492685170680188], [0.011699144222216028, 0.03282992806133637, 0.029493506297000423, 0.005720905521099993], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010261857446045594], [0.011713640821069456, 0.03238092958307968, 0.03135585591411297, 0.00828793940970218], [0.0058685452594453204, 0.020962835873204952, 0.011955587534010458, 0.008162785679510096], [0.008306793799146847, 0.03282992806133637, 0.029448118839112032, 0.006014596502623002], [0.0058685452594453204, 0.019924786260881105, 0.011410484359548091, 0.005492685170680188], [0.008306793799146847, 0.03282992806133637, 0.028536304793467953, 0.005322001932977224], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010146993788951312], [0.011639448627040415, 0.033441879561074264, 0.03149230099758477, 0.010135567292587268], [0.0058685452594453204, 0.01841012676420941, 0.011148314367264676, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011955587534010458, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.017031319276910872, 0.03349473704721639, 0.03135585591411297, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.008388637106475779, 0.020991125017491112, 0.011955587534010458, 0.01002463011482834], [0.0058685452594453204, 0.020962835873204952, 0.011955587534010458, 0.008162785679510096], [0.011428302805086119, 0.0327076256644521, 0.011410484359548091, 0.005492685170680188], [0.011713640821069456, 0.03346962743720302, 0.03135585591411297, 0.01001411272123214], [0.011699144222216028, 0.03282992806133637, 0.029493506297000423, 0.005720905521099993], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010261857446045594], [0.0058685452594453204, 0.020962835873204952, 0.011955587534010458, 0.008162785679510096], [0.008306793799146847, 0.03282992806133637, 0.029448118839112032, 0.006014596502623002], [0.008306793799146847, 0.03282992806133637, 0.028536304793467953, 0.005322001932977224], [0.011713640821069456, 0.03238092958307968, 0.03135585591411297, 0.00828793940970218], [0.0058685452594453204, 0.01841012676420941, 0.011148314367264676, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011955587534010458, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.017031319276910872, 0.03349473704721639, 0.03135585591411297, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.008388637106475779, 0.020991125017491112, 0.011955587534010458, 0.01002463011482834], [0.008388637106475779, 0.020991125017491112, 0.011955587534010458, 0.01002463011482834], [0.012896130240056609, 0.03240627241289958, 0.03135585591411297, 0.010261857446045594], [0.0058685452594453204, 0.020962835873204952, 0.011955587534010458, 0.008162785679510096], [0.008306793799146847, 0.03282992806133637, 0.028536304793467953, 0.005322001932977224], [0.008306793799146847, 0.03282992806133637, 0.029448118839112032, 0.006014596502623002], [0.012087690757123012, 0.03294216317906973, 0.027488789600407203, 0.006014596502623002], [0.011713640821069456, 0.03240627241289958, 0.03135585591411297, 0.010261857446045594], [0.011699144222216028, 0.03282992806133637, 0.029493506297000423, 0.005720905521099993], [0.0058685452594453204, 0.01841012676420941, 0.011148314367264676, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.017031319276910872, 0.03349473704721639, 0.03135585591411297, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.012896130240056609, 0.03240627241289958, 0.03135585591411297, 0.010261857446045594], [0.008388637106475779, 0.020991125017491112, 0.011955587534010458, 0.01002463011482834], [0.008388637106475779, 0.020991125017491112, 0.011955587534010458, 0.01002463011482834], [0.0058685452594453204, 0.020962835873204952, 0.011955587534010458, 0.008162785679510096], [0.012087690757123012, 0.03294216317906973, 0.027488789600407203, 0.006014596502623002], [0.008388637106475779, 0.02100970486697, 0.011955587534010458, 0.014317246208279122], [0.0058685452594453204, 0.0332785827762237, 0.02811410802772274, 0.00550374396234912], [0.008306793799146847, 0.03282992806133637, 0.029448118839112032, 0.006014596502623002], [0.008306793799146847, 0.03282992806133637, 0.028536304793467953, 0.005322001932977224], [0.0058685452594453204, 0.01735203978655234, 0.011148314367264676, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011148314367264676, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.017031319276910872, 0.03349473704721639, 0.03135585591411297, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.008388637106475779, 0.02100970486697, 0.011955587534010458, 0.014317246208279122], [0.012087690757123012, 0.03294216317906973, 0.027488789600407203, 0.006014596502623002], [0.008306793799146847, 0.03282992806133637, 0.026558047602669943, 0.005322001932977224], [0.012896130240056609, 0.03240627241289958, 0.03135585591411297, 0.010261857446045594], [0.008306793799146847, 0.03282992806133637, 0.029448118839112032, 0.006014596502623002], [0.012087690757123012, 0.03294216317906973, 0.027488789600407203, 0.006014596502623002], [0.008388637106475779, 0.020991125017491112, 0.011955587534010458, 0.01002463011482834], [0.007459764400826975, 0.01841012676420941, 0.01426167390114414, 0.005492685170680188], [0.0058685452594453204, 0.01735203978655234, 0.011148314367264676, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011148314367264676, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.017031319276910872, 0.03349473704721639, 0.03135585591411297, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.017031319276910872, 0.03349473704721639, 0.03135585591411297, 0.013465936605514033], [0.008409950244274718, 0.03282992806133637, 0.026558047602669943, 0.0051734258075367585], [0.016906371640734365, 0.01840063435549845, 0.031368893930963516, 0.005492685170680188], [0.008306793799146847, 0.03282992806133637, 0.029448118839112032, 0.006014596502623002], [0.008388637106475779, 0.02100970486697, 0.011968045771887368, 0.014317246208279122], [0.012896130240056609, 0.03240627241289958, 0.03135585591411297, 0.010261857446045594], [0.008388637106475779, 0.02100970486697, 0.011955587534010458, 0.014317246208279122], [0.007459764400826975, 0.01841012676420941, 0.01426167390114414, 0.005492685170680188], [0.0058685452594453204, 0.01735203978655234, 0.011148314367264676, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011148314367264676, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.01841012676420941, 0.011410484359548091, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.017031319276910872, 0.03349473704721639, 0.03135585591411297, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.017031319276910872, 0.03349473704721639, 0.03135585591411297, 0.013465936605514033], [0.017031319276910872, 0.03349639991480273, 0.03135585591411297, 0.013465936605514033], [0.012896130240056609, 0.03240627241289958, 0.03135585591411297, 0.010261857446045594], [0.008388637106475779, 0.02100970486697, 0.011968045771887368, 0.014317246208279122], [0.016906371640734365, 0.018410100130826855, 0.031368893930963516, 0.005492685170680188], [0.007230438016737756, 0.03282992806133637, 0.02998201118318086, 0.006014596502623002], [0.008388637106475779, 0.02100970486697, 0.011955587534010458, 0.014317246208279122], [0.008409950244274718, 0.03282992806133637, 0.026558047602669943, 0.0051734258075367585], [0.0058685452594453204, 0.016984818729168707, 0.011328844431827266, 0.005492685170680188], [0.0058685452594453204, 0.01735203978655234, 0.011148314367264676, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.019591617944796483, 0.03279320952222805, 0.029237991497772925, 0.007820606202126379], [0.017031319276910872, 0.03349473704721639, 0.03135585591411297, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.017031319276910872, 0.03349473704721639, 0.03135585591411297, 0.013465936605514033], [0.017031319276910872, 0.03349639991480273, 0.03135585591411297, 0.013465936605514033], [0.012896130240056609, 0.03240627241289958, 0.03135585591411297, 0.010261857446045594], [0.016906371640734365, 0.018410100130826855, 0.031368893930963516, 0.005492685170680188], [0.008388637106475779, 0.02100970486697, 0.012360237926444967, 0.014317246208279122], [0.0072920514996382465, 0.03282992806133637, 0.02998201118318086, 0.0092932236246343], [0.017140643838909047, 0.018410100130826855, 0.031368893930963516, 0.005492685170680188], [0.006913141489363637, 0.03143336657162866, 0.029768149169482793, 0.006014596502623002], [0.008388637106475779, 0.02100970486697, 0.011968045771887368, 0.014317246208279122], [0.007230438016737756, 0.03282992806133637, 0.02998201118318086, 0.006014596502623002], [0.008388637106475779, 0.02100970486697, 0.011955587534010458, 0.014317246208279122], [0.008388637106475779, 0.01947454487244108, 0.011968045771887368, 0.014317246208279122], [0.0058685452594453204, 0.016984818729168707, 0.011328844431827266, 0.005492685170680188], [0.0058685452594453204, 0.01735203978655234, 0.011148314367264676, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.017031319276910872, 0.03349473704721639, 0.03135585591411297, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.017031319276910872, 0.03352801731726022, 0.030864256452802382, 0.013465936605514033], [0.016906371640734365, 0.018410100130826855, 0.031368893930963516, 0.005492685170680188], [0.007230438016737756, 0.03282992806133637, 0.028172995365947327, 0.006014596502623002], [0.0058685452594453204, 0.016984818729168707, 0.02859051198735591, 0.005459886118225587], [0.008388637106475779, 0.01947454487244108, 0.011968045771887368, 0.014317246208279122], [0.008388637106475779, 0.02100970486697, 0.012360237926444967, 0.014317246208279122], [0.008388637106475779, 0.02100970486697, 0.011955587534010458, 0.014317246208279122], [0.017140643838909047, 0.018410100130826855, 0.031368893930963516, 0.005492685170680188], [0.008388637106475779, 0.02100970486697, 0.011968045771887368, 0.014317246208279122], [0.008388637106475779, 0.017653430813811172, 0.012874870041319675, 0.014317246208279122], [0.007230438016737756, 0.03282992806133637, 0.02998201118318086, 0.005836056102570387], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.008388637106475779, 0.02100970486697, 0.011968045771887368, 0.014317246208279122], [0.0058685452594453204, 0.016984818729168707, 0.011328844431827266, 0.005492685170680188], [0.0058685452594453204, 0.01735203978655234, 0.011148314367264676, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.017031319276910872, 0.03349473704721639, 0.03135585591411297, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.017031319276910872, 0.03352801731726022, 0.030864256452802382, 0.013465936605514033], [0.007230438016737756, 0.03282992806133637, 0.028172995365947327, 0.006014596502623002], [0.007230438016737756, 0.03282992806133637, 0.02998201118318086, 0.005637068890137671], [0.008388637106475779, 0.017653430813811172, 0.012874870041319675, 0.014317246208279122], [0.008388637106475779, 0.01947454487244108, 0.011968045771887368, 0.014317246208279122], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.010892812681816539], [0.0058685452594453204, 0.02100540640213748, 0.011268011400215541, 0.005492685170680188], [0.006467115853554207, 0.01947454487244108, 0.014304207346144852, 0.014317246208279122], [0.0058685452594453204, 0.016984818729168707, 0.02859051198735591, 0.005459886118225587], [0.005907492210610585, 0.02100540640213748, 0.031020107641927277, 0.005492685170680188], [0.016906371640734365, 0.018410100130826855, 0.031368893930963516, 0.005492685170680188], [0.017031319276910872, 0.03417336495386164, 0.031461129623435426, 0.01092759035403472], [0.016779406846928486, 0.018410100130826855, 0.031368893930963516, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.01735203978655234, 0.011148314367264676, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.017023273273193522, 0.03394493711748932, 0.031368893930963516, 0.005492685170680188], [0.007230438016737756, 0.03282992806133637, 0.028172995365947327, 0.006014596502623002], [0.007230438016737756, 0.03282992806133637, 0.030603014753190626, 0.005637068890137671], [0.0058685452594453204, 0.016984818729168707, 0.02859051198735591, 0.005441673921376888], [0.006467115853554207, 0.01947454487244108, 0.014304207346144852, 0.014317246208279122], [0.008388637106475779, 0.017653430813811172, 0.012874870041319675, 0.014317246208279122], [0.007230438016737756, 0.03282992806133637, 0.02998201118318086, 0.005637068890137671], [0.0058685452594453204, 0.02100540640213748, 0.011268011400215541, 0.005492685170680188], [0.005907492210610585, 0.02100540640213748, 0.031020107641927277, 0.005492685170680188], [0.008388637106475779, 0.01947454487244108, 0.011968045771887368, 0.014317246208279122], [0.006379470366318703, 0.03282992806133637, 0.02998201118318086, 0.005637068890137671], [0.0058685452594453204, 0.02100540640213748, 0.011268011400215541, 0.005492685170680188], [0.006295820347289073, 0.019407846229103427, 0.014304207346144852, 0.014317246208279122], [0.006467115853554207, 0.01947454487244108, 0.014304207346144852, 0.014317246208279122], [0.007926069348774043, 0.03282992806133637, 0.02998201118318086, 0.005637068890137671], [0.005907492210610585, 0.02100540640213748, 0.031020107641927277, 0.005492685170680188], [0.0058685452594453204, 0.016984818729168707, 0.011328844431827266, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.01735203978655234, 0.011148314367264676, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.017023273273193522, 0.03394493711748932, 0.031368893930963516, 0.005492685170680188], [0.007230438016737756, 0.03282992806133637, 0.028172995365947327, 0.006014596502623002], [0.007230438016737756, 0.03282992806133637, 0.030603014753190626, 0.005637068890137671], [0.007230438016737756, 0.03201607749859891, 0.025927691172547174, 0.006014596502623002], [0.005907492210610585, 0.02133900349401415, 0.031020107641927277, 0.005492685170680188], [0.0058685452594453204, 0.016984818729168707, 0.02859051198735591, 0.005441673921376888], [0.006467115853554207, 0.01947454487244108, 0.014304207346144852, 0.014317246208279122], [0.0058685452594453204, 0.02100540640213748, 0.011268011400215541, 0.005492685170680188], [0.008388637106475779, 0.017653430813811172, 0.012874870041319675, 0.014317246208279122], [0.008388637106475779, 0.01947454487244108, 0.011968045771887368, 0.014317246208279122], [0.007230438016737756, 0.03282992806133637, 0.02998201118318086, 0.005637068890137671], [0.005907492210610585, 0.02185499942008134, 0.02628102510105217, 0.007316268904523055], [0.007033115793046243, 0.03282992806133637, 0.02998201118318086, 0.005637068890137671], [0.0058685452594453204, 0.021108500353056434, 0.011328844431827266, 0.005492685170680188], [0.0058685452594453204, 0.016984818729168707, 0.02859051198735591, 0.005441673921376888], [0.0058685452594453204, 0.02100540640213748, 0.011268011400215541, 0.005492685170680188], [0.006295820347289073, 0.019407846229103427, 0.014304207346144852, 0.014317246208279122], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.01735203978655234, 0.011148314367264676, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.016898253710663905, 0.03394493711748932, 0.031368893930963516, 0.005492685170680188], [0.005907492210610585, 0.02133900349401415, 0.031020107641927277, 0.005492685170680188], [0.007230438016737756, 0.03282992806133637, 0.030603014753190626, 0.005637068890137671], [0.007230438016737756, 0.03282992806133637, 0.028172995365947327, 0.006014596502623002], [0.007230438016737756, 0.03282992806133637, 0.030603014753190626, 0.005637068890137671], [0.007230438016737756, 0.03201607749859891, 0.025927691172547174, 0.006014596502623002], [0.007230438016737756, 0.03201607749859891, 0.025927691172547174, 0.006014596502623002], [0.0058685452594453204, 0.02100540640213748, 0.011268011400215541, 0.005492685170680188], [0.0058685452594453204, 0.01934926115311971, 0.02859051198735591, 0.005441673921376888], [0.007230438016737756, 0.03282992806133637, 0.02998201118318086, 0.005637068890137671], [0.008388637106475779, 0.017653430813811172, 0.012874870041319675, 0.014317246208279122], [0.005629012052852868, 0.02185499942008134, 0.02628102510105217, 0.007316268904523055], [0.005907492210610585, 0.02133900349401415, 0.031020107641927277, 0.005492685170680188], [0.008052567394035615, 0.017653430813811172, 0.012874870041319675, 0.014317246208279122], [0.008388637106475779, 0.01947454487244108, 0.011968045771887368, 0.014317246208279122], [0.005907075105803402, 0.016984818729168707, 0.02859051198735591, 0.005242953148248677], [0.005061932728730493, 0.015193421734091711, 0.010782590331802929, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.01735203978655234, 0.011148314367264676, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.016898253710663905, 0.03394493711748932, 0.031368893930963516, 0.005492685170680188], [0.007230438016737756, 0.03201607749859891, 0.025927691172547174, 0.006014596502623002], [0.007790020452161841, 0.03282992806133637, 0.030603014753190626, 0.005637068890137671], [0.016898253710663905, 0.03394493711748932, 0.031023962504453745, 0.005492685170680188], [0.005907492210610585, 0.02133900349401415, 0.031020107641927277, 0.005492685170680188], [0.005907492210610585, 0.02133900349401415, 0.031020107641927277, 0.005492685170680188], [0.007230438016737756, 0.03282992806133637, 0.030603014753190626, 0.005637068890137671], [0.007230438016737756, 0.03201607749859891, 0.025927691172547174, 0.006014596502623002], [0.0058685452594453204, 0.02100540640213748, 0.011268011400215541, 0.005492685170680188], [0.0058685452594453204, 0.01934926115311971, 0.02859051198735591, 0.005441673921376888], [0.007230438016737756, 0.03282992806133637, 0.030603014753190626, 0.005637068890137671], [0.00586057907612748, 0.02133900349401415, 0.031020107641927277, 0.005492685170680188], [0.008052567394035615, 0.017653430813811172, 0.012874870041319675, 0.014317246208279122], [0.005088777170901917, 0.01578179377208946, 0.011268678682758312, 0.005492685170680188], [0.005629012052852868, 0.02185499942008134, 0.026113475224344682, 0.007316268904523055], [0.005061932728730493, 0.015193421734091711, 0.010782590331802929, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.01735203978655234, 0.011148314367264676, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.016350167612903685, 0.03394493711748932, 0.031023962504453745, 0.005492685170680188], [0.007790020452161841, 0.03282992806133637, 0.030603014753190626, 0.005637068890137671], [0.005905283381847089, 0.031053091199729176, 0.02599326236028235, 0.005479967382742219], [0.007230438016737756, 0.03201607749859891, 0.025927691172547174, 0.006014596502623002], [0.007230438016737756, 0.03201607749859891, 0.025927691172547174, 0.006014596502623002], [0.005907492210610585, 0.02133900349401415, 0.031020107641927277, 0.005492685170680188], [0.005907492210610585, 0.02133900349401415, 0.031020107641927277, 0.005492685170680188], [0.008052567394035615, 0.017653430813811172, 0.012874870041319675, 0.014317246208279122], [0.007230438016737756, 0.03282992806133637, 0.030603014753190626, 0.005637068890137671], [0.0050869406516949175, 0.021005316252046335, 0.011268678682758312, 0.005492685170680188], [0.00586057907612748, 0.02133900349401415, 0.031020107641927277, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011268011400215541, 0.005492685170680188], [0.008283589991720623, 0.03282992806133637, 0.031015919936038118, 0.005637068890137671], [0.007230438016737756, 0.03282992806133637, 0.030603014753190626, 0.005637068890137671], [0.006485840552498525, 0.03201607749859891, 0.025426802317260425, 0.006014596502623002], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010782590331802929, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.01735203978655234, 0.011148314367264676, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.016350167612903685, 0.03394493711748932, 0.031023962504453745, 0.005492685170680188], [0.005905283381847089, 0.031053091199729176, 0.02599326236028235, 0.005479967382742219], [0.016350167612903685, 0.0335034413954995, 0.031023962504453745, 0.005492685170680188], [0.008283589991720623, 0.03282992806133637, 0.031015919936038118, 0.005637068890137671], [0.007790020452161841, 0.03282992806133637, 0.030603014753190626, 0.005637068890137671], [0.00586057907612748, 0.021475564281275256, 0.03137012601528287, 0.005637055901045405], [0.006485840552498525, 0.03201607749859891, 0.025426802317260425, 0.006014596502623002], [0.0058685452594453204, 0.02100540640213748, 0.011268011400215541, 0.005492685170680188], [0.0050869406516949175, 0.021005316252046335, 0.011268678682758312, 0.005492685170680188], [0.007230438016737756, 0.03201607749859891, 0.025927691172547174, 0.006014596502623002], [0.007230438016737756, 0.03201607749859891, 0.025927691172547174, 0.006014596502623002], [0.008052567394035615, 0.017653430813811172, 0.012874870041319675, 0.014317246208279122], [0.007230438016737756, 0.03282992806133637, 0.030603014753190626, 0.005637068890137671], [0.005907492210610585, 0.02133900349401415, 0.031020107641927277, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010782590331802929, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.01735203978655234, 0.011148314367264676, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.016350167612903685, 0.03394493711748932, 0.031023962504453745, 0.005492685170680188], [0.016350167612903685, 0.03394493711748932, 0.031491381677765475, 0.005492685170680188], [0.008052567394035615, 0.017653430813811172, 0.012874870041319675, 0.014317246208279122], [0.016350167612903685, 0.0335034413954995, 0.031023962504453745, 0.005492685170680188], [0.011498309335143361, 0.03282992806133637, 0.031015919936038118, 0.005637068890137671], [0.00586057907612748, 0.021475564281275256, 0.03137012601528287, 0.005637055901045405], [0.005905283381847089, 0.031053091199729176, 0.02599326236028235, 0.005479967382742219], [0.007790020452161841, 0.03282992806133637, 0.030603014753190626, 0.005637068890137671], [0.005905283381847089, 0.031053091199729176, 0.02599326236028235, 0.005479967382742219], [0.008283589991720623, 0.03282992806133637, 0.031015919936038118, 0.005637068890137671], [0.006485840552498525, 0.03201607749859891, 0.025426802317260425, 0.006014596502623002], [0.008283589991720623, 0.03282992806133637, 0.031015919936038118, 0.005637068890137671], [0.005907492210610585, 0.02133900349401415, 0.031020107641927277, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011268011400215541, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.016350167612903685, 0.03394493711748932, 0.031023962504453745, 0.005492685170680188], [0.016350167612903685, 0.03394493711748932, 0.031491381677765475, 0.005492685170680188], [0.016350167612903685, 0.03394493711748932, 0.031491381677765475, 0.005492685170680188], [0.011498309335143361, 0.03282992806133637, 0.031015919936038118, 0.005637068890137671], [0.016350167612903685, 0.0335034413954995, 0.031023962504453745, 0.005492685170680188], [0.00586057907612748, 0.021475564281275256, 0.03137012601528287, 0.005637055901045405], [0.008052567394035615, 0.017653430813811172, 0.012874870041319675, 0.014317246208279122], [0.005905283381847089, 0.031053091199729176, 0.02599326236028235, 0.005479967382742219], [0.0058685452594453204, 0.02100540640213748, 0.011268011400215541, 0.005492685170680188], [0.007790020452161841, 0.03282992806133637, 0.030603014753190626, 0.005637068890137671], [0.008283589991720623, 0.03282992806133637, 0.031198564559886986, 0.005637068890137671], [0.005905283381847089, 0.031053091199729176, 0.02599326236028235, 0.005479967382742219], [0.005905283381847089, 0.031053091199729176, 0.02599326236028235, 0.005479967382742219], [0.005907492210610585, 0.02133900349401415, 0.030596238203611178, 0.005492685170680188], [0.008283589991720623, 0.03282992806133637, 0.031015919936038118, 0.005637068890137671], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.016350167612903685, 0.03394493711748932, 0.031023962504453745, 0.005492685170680188], [0.01622236287593924, 0.03394493711748932, 0.031491381677765475, 0.005492685170680188], [0.00586057907612748, 0.021475564281275256, 0.03137012601528287, 0.005637055901045405], [0.011498309335143361, 0.03282992806133637, 0.031015919936038118, 0.005637068890137671], [0.008052567394035615, 0.021481227173460033, 0.01229825285909278, 0.014264197428768133], [0.008283589991720623, 0.03282992806133637, 0.031198564559886986, 0.005637068890137671], [0.008052567394035615, 0.017653430813811172, 0.012874870041319675, 0.014317246208279122], [0.005905283381847089, 0.031053091199729176, 0.02599326236028235, 0.005479967382742219], [0.016350167612903685, 0.0335034413954995, 0.031023962504453745, 0.005492685170680188], [0.016350167612903685, 0.03394493711748932, 0.030349600724752932, 0.005492685170680188], [0.011498309335143361, 0.03282992806133637, 0.031015919936038118, 0.005637068890137671], [0.007041627625192543, 0.03282992806133637, 0.03111745518442457, 0.005637068890137671], [0.0058685452594453204, 0.02100540640213748, 0.011268011400215541, 0.005492685170680188], [0.005905283381847089, 0.031053091199729176, 0.02599326236028235, 0.005479967382742219], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.011431965455858274, 0.03389601747871837, 0.03052798637127772, 0.005637068890137671], [0.005905283381847089, 0.031053091199729176, 0.02599326236028235, 0.005479967382742219], [0.005022345825824525, 0.01735203978655234, 0.012088114892597172, 0.005492685170680188], [0.00586057907612748, 0.021475564281275256, 0.03137012601528287, 0.005637055901045405], [0.008283589991720623, 0.03282992806133637, 0.031198564559886986, 0.005637068890137671], [0.008052567394035615, 0.017653430813811172, 0.012874870041319675, 0.014317246208279122], [0.0058685452594453204, 0.02100540640213748, 0.011268011400215541, 0.005492685170680188], [0.005905283381847089, 0.031053091199729176, 0.02599326236028235, 0.005479967382742219], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.00622672822686162], [0.005061932728730493, 0.01564424297781163, 0.010708770330112404, 0.006434872932855909], [0.007041627625192543, 0.03410025567678098, 0.03111745518442457, 0.005637068890137671], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005887370218589653], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.011431965455858274, 0.03389601747871837, 0.03052798637127772, 0.005637068890137671], [0.005905283381847089, 0.031053091199729176, 0.02599326236028235, 0.0050462387702632176], [0.008052567394035615, 0.017653430813811172, 0.012874870041319675, 0.014317246208279122], [0.007041627625192543, 0.03410025567678098, 0.03111745518442457, 0.005637068890137671], [0.005022345825824525, 0.01735203978655234, 0.012088114892597172, 0.005492685170680188], [0.008269794931449753, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.008283589991720623, 0.03282992806133637, 0.031198564559886986, 0.005637068890137671], [0.005065174210140535, 0.01735203978655234, 0.012457853824706313, 0.005889589645043986], [0.0058685452594453204, 0.02100540640213748, 0.011268011400215541, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005887370218589653], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.00622672822686162], [0.0058685452594453204, 0.02100540640213748, 0.011268011400215541, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.011431965455858274, 0.03389601747871837, 0.03052798637127772, 0.005637068890137671], [0.012312697683569223, 0.03410025567678098, 0.03111745518442457, 0.005637068890137671], [0.007041627625192543, 0.03410025567678098, 0.03111745518442457, 0.005637068890137671], [0.005905283381847089, 0.031053091199729176, 0.024539862889139258, 0.0050462387702632176], [0.005905283381847089, 0.031053091199729176, 0.02599326236028235, 0.0050462387702632176], [0.008269794931449753, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.00622672822686162], [0.0058685452594453204, 0.02100540640213748, 0.011268011400215541, 0.005492685170680188], [0.0060119618341100125, 0.02100540640213748, 0.011299859349178069, 0.005492685170680188], [0.00590042639637619, 0.017454206829831426, 0.012457853824706313, 0.005889589645043986], [0.005022345825824525, 0.01735203978655234, 0.012088114892597172, 0.005492685170680188], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.011431965455858274, 0.03389601747871837, 0.03052798637127772, 0.005637068890137671], [0.012312697683569223, 0.03410025567678098, 0.03111745518442457, 0.005637068890137671], [0.0058685452594453204, 0.02100540640213748, 0.01249659668387473, 0.005902056550013307], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.0058685452594453204, 0.03319597456197941, 0.012141962700214169, 0.0054910725126983894], [0.007041627625192543, 0.03410025567678098, 0.03111745518442457, 0.005405586845717445], [0.00590042639637619, 0.017454206829831426, 0.012457853824706313, 0.005889589645043986], [0.005905283381847089, 0.031053091199729176, 0.02599326236028235, 0.0050462387702632176], [0.0058685452594453204, 0.02100540640213748, 0.011268011400215541, 0.005492685170680188], [0.005905283381847089, 0.031053091199729176, 0.024539862889139258, 0.0050462387702632176], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.00622672822686162], [0.0060119618341100125, 0.02100540640213748, 0.011299859349178069, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.011431965455858274, 0.03389601747871837, 0.03052798637127772, 0.005637068890137671], [0.011381106658985368, 0.034099140251107246, 0.031150704145775532, 0.005637068890137671], [0.007041627625192543, 0.03410025567678098, 0.03111745518442457, 0.005405586845717445], [0.00590042639637619, 0.017454206829831426, 0.012457853824706313, 0.005889589645043986], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.0058685452594453204, 0.03319597456197941, 0.012141962700214169, 0.0054910725126983894], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.0058685452594453204, 0.02100540640213748, 0.011268011400215541, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.01249659668387473, 0.005902056550013307], [0.005905283381847089, 0.031053091199729176, 0.02599326236028235, 0.0050462387702632176], [0.005061932728730493, 0.015193421734091711, 0.011796488756889198, 0.0061994821484742686], [0.0058685452594453204, 0.03319597456197941, 0.012141962700214169, 0.0054910725126983894], [0.005061932728730493, 0.01506784173016169, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.011431965455858274, 0.03389601747871837, 0.03052798637127772, 0.005637068890137671], [0.011381106658985368, 0.034099140251107246, 0.031150704145775532, 0.005637068890137671], [0.011431203101656965, 0.03439870118515532, 0.03111745518442457, 0.005405586845717445], [0.007041627625192543, 0.03410025567678098, 0.03111745518442457, 0.005405586845717445], [0.0058685452594453204, 0.02100540640213748, 0.01249659668387473, 0.005902056550013307], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.005900525889689986, 0.03319597456197941, 0.01165542828265833, 0.0054910725126983894], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.006260367055383068], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.0058685452594453204, 0.03319597456197941, 0.012141962700214169, 0.0054910725126983894], [0.005061932728730493, 0.01506784173016169, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.011431965455858274, 0.03389601747871837, 0.03052798637127772, 0.005637068890137671], [0.011431203101656965, 0.03439870118515532, 0.03111745518442457, 0.005405586845717445], [0.007041627625192543, 0.03410025567678098, 0.03111745518442457, 0.005405586845717445], [0.005571965965695449, 0.03322060206505201, 0.031134966100672144, 0.005312333909851677], [0.005088777170901917, 0.01744344739510577, 0.011318205496567119, 0.005492685170680188], [0.005900525889689986, 0.03319597456197941, 0.01165542828265833, 0.0054910725126983894], [0.0058685452594453204, 0.02100540640213748, 0.01249659668387473, 0.00658668668065538], [0.0058685452594453204, 0.02100540640213748, 0.01249659668387473, 0.005902056550013307], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.005061932728730493, 0.015045948992856537, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01506784173016169, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.011431203101656965, 0.03439870118515532, 0.03111745518442457, 0.005405586845717445], [0.007041627625192543, 0.03410025567678098, 0.03111745518442457, 0.005405586845717445], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.007721397816736449, 0.02100540640213748, 0.01249659668387473, 0.00658668668065538], [0.005571965965695449, 0.03322060206505201, 0.031134966100672144, 0.005312333909851677], [0.005088777170901917, 0.01744344739510577, 0.011318205496567119, 0.005492685170680188], [0.005088777170901917, 0.01744344739510577, 0.011307730765828933, 0.009175477162069744], [0.005061932728730493, 0.03386706294950894, 0.010738255914706249, 0.005646341645061798], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.0058685452594453204, 0.02100540640213748, 0.01249659668387473, 0.00658668668065538], [0.005061932728730493, 0.015045948992856537, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01506784173016169, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.011431203101656965, 0.03439870118515532, 0.03111745518442457, 0.005222001532853577], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.005061932728730493, 0.03386706294950894, 0.010738255914706249, 0.005646341645061798], [0.011431203101656965, 0.03439870118515532, 0.030820170512619302, 0.005405586845717445], [0.007041627625192543, 0.03410025567678098, 0.03111745518442457, 0.005405089625735065], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.005088777170901917, 0.01744344739510577, 0.011307730765828933, 0.009175477162069744], [0.005088777170901917, 0.01744344739510577, 0.011318205496567119, 0.005492685170680188], [0.006556263845537005, 0.03410025567678098, 0.03111745518442457, 0.005405586845717445], [0.0058685452594453204, 0.02039844113378973, 0.012881749552679485, 0.00658668668065538], [0.0051780140374396945, 0.015095098383136147, 0.010248881669894343, 0.005492685170680188], [0.005061932728730493, 0.01506784173016169, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.011431203101656965, 0.03439870118515532, 0.03111745518442457, 0.005222001532853577], [0.011431203101656965, 0.03439870118515532, 0.030820170512619302, 0.005405586845717445], [0.005010993848689053, 0.03289452317868274, 0.011955587534010458, 0.005492685170680188], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.00793141665004653, 0.03410025567678098, 0.03111745518442457, 0.005493756526273747], [0.0058685452594453204, 0.02039844113378973, 0.012881749552679485, 0.00658668668065538], [0.005088777170901917, 0.01744344739510577, 0.011307730765828933, 0.009175477162069744], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.007041627625192543, 0.03410025567678098, 0.03111745518442457, 0.005405089625735065], [0.006556263845537005, 0.03410025567678098, 0.03111745518442457, 0.005405586845717445], [0.0051780140374396945, 0.015095098383136147, 0.010248881669894343, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.011431203101656965, 0.03439870118515532, 0.03111745518442457, 0.005222001532853577], [0.00793141665004653, 0.03410025567678098, 0.03111745518442457, 0.005493756526273747], [0.0058685452594453204, 0.02039844113378973, 0.012881749552679485, 0.00658668668065538], [0.005010993848689053, 0.03289452317868274, 0.011955587534010458, 0.005492685170680188], [0.0058685452594453204, 0.022742495227648683, 0.011955587534010458, 0.011737613146101729], [0.007041627625192543, 0.03410025567678098, 0.03111745518442457, 0.005405089625735065], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.006556263845537005, 0.03410025567678098, 0.03111745518442457, 0.005405586845717445], [0.005088777170901917, 0.017806451032326195, 0.011307730765828933, 0.009175477162069744], [0.0058685452594453204, 0.02100540640213748, 0.011929458950747481, 0.005492685170680188], [0.005088777170901917, 0.01744344739510577, 0.011307730765828933, 0.009175477162069744], [0.0051780140374396945, 0.015095098383136147, 0.010248881669894343, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.011431203101656965, 0.03439870118515532, 0.03111745518442457, 0.005222001532853577], [0.0058685452594453204, 0.022742495227648683, 0.011955587534010458, 0.011737613146101729], [0.005010993848689053, 0.03289452317868274, 0.011955587534010458, 0.005492685170680188], [0.007041627625192543, 0.03445885724201088, 0.03111745518442457, 0.005059373615493241], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.0058685452594453204, 0.02100540640213748, 0.011929458950747481, 0.005492685170680188], [0.006556263845537005, 0.03410025567678098, 0.03111745518442457, 0.005405586845717445], [0.0058685452594453204, 0.02039844113378973, 0.012881749552679485, 0.00658668668065538], [0.005088777170901917, 0.01735203978655234, 0.01330555555692502, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.006595865662928843], [0.007041627625192543, 0.03410025567678098, 0.03111745518442457, 0.0053520273793923575], [0.0050268297170800655, 0.015095098383136147, 0.010230538597223324, 0.005492685170680188], [0.0051780140374396945, 0.015095098383136147, 0.010248881669894343, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.011431203101656965, 0.03439870118515532, 0.03111745518442457, 0.005222001532853577], [0.011431203101656965, 0.03439870118515532, 0.03111745518442457, 0.005222001532853577], [0.0058685452594453204, 0.02100540640213748, 0.03148687230050663, 0.005342289955751007], [0.0058685452594453204, 0.022742495227648683, 0.011955587534010458, 0.011737613146101729], [0.007041627625192543, 0.03445885724201088, 0.031160719730233592, 0.005059373615493241], [0.00505175276532252, 0.031851778205818755, 0.03052366602505172, 0.0056370687256551665], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.005037452596854414, 0.03278769430764436, 0.01198752519997167, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.01330555555692502, 0.005492685170680188], [0.006556263845537005, 0.03410025567678098, 0.03111745518442457, 0.005405586845717445], [0.0050268297170800655, 0.015095098383136147, 0.010230538597223324, 0.005492685170680188], [0.0051780140374396945, 0.015095098383136147, 0.010248881669894343, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.005088777170901917, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.011431203101656965, 0.03439870118515532, 0.03111745518442457, 0.005222001532853577], [0.011431203101656965, 0.03439870118515532, 0.03111745518442457, 0.005222001532853577], [0.011431203101656965, 0.03439870118515532, 0.03111745518442457, 0.005222001532853577], [0.007041627625192543, 0.03445885724201088, 0.031160719730233592, 0.005059373615493241], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005707166202155668], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.0058685452594453204, 0.02100540640213748, 0.03148687230050663, 0.005342289955751007], [0.006556263845537005, 0.03410025567678098, 0.03111745518442457, 0.005405586845717445], [0.00505175276532252, 0.03162755346964442, 0.03052366602505172, 0.0056370687256551665], [0.005088777170901917, 0.016633003463663566, 0.03038452243748018, 0.007122796727665915], [0.0050268297170800655, 0.015095098383136147, 0.010230538597223324, 0.005492685170680188], [0.0051780140374396945, 0.015095098383136147, 0.010248881669894343, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.011431203101656965, 0.03439870118515532, 0.03111745518442457, 0.005222001532853577], [0.011431203101656965, 0.03439870118515532, 0.03111745518442457, 0.005222001532853577], [0.011431203101656965, 0.03439870118515532, 0.03111745518442457, 0.005222001532853577], [0.011431203101656965, 0.03445396684829989, 0.031121057480237843, 0.005222001532853577], [0.006556263845537005, 0.03410025567678098, 0.03111745518442457, 0.005405586845717445], [0.007041627625192543, 0.03445885724201088, 0.031160719730233592, 0.005059373615493241], [0.006383382443470326, 0.031851778205818755, 0.03137012601528287, 0.0056370687256551665], [0.005616784423980166, 0.029388437967006537, 0.031160719730233592, 0.005059373615493241], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005707166202155668], [0.00505175276532252, 0.03162755346964442, 0.03052366602505172, 0.0056370687256551665], [0.006383382443470326, 0.031409463659906815, 0.03137012601528287, 0.0056370687256551665], [0.0050268297170800655, 0.015027893177736442, 0.010230538597223324, 0.005087330324623459], [0.0050268297170800655, 0.015095098383136147, 0.010230538597223324, 0.005492685170680188], [0.0051780140374396945, 0.015095098383136147, 0.010248881669894343, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.011295270096288279, 0.03445885724201088, 0.031160719730233592, 0.005059373615493241], [0.007041627625192543, 0.03445885724201088, 0.031160719730233592, 0.005059373615493241], [0.005616784423980166, 0.029388437967006537, 0.031160719730233592, 0.005059373615493241], [0.007041627625192543, 0.03445885724201088, 0.031160719730233592, 0.005059373615493241], [0.005616784423980166, 0.029388437967006537, 0.030704126063068788, 0.005059373615493241], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005707166202155668], [0.006383382443470326, 0.031409463659906815, 0.03137012601528287, 0.0056370687256551665], [0.00505175276532252, 0.03162755346964442, 0.03052366602505172, 0.0056370687256551665], [0.005059771007954962, 0.01735203978655234, 0.012258842535878174, 0.005049045397022951], [0.0071610168765452705, 0.03445885724201088, 0.029819858508138335, 0.005060571188300924], [0.0050268297170800655, 0.015027893177736442, 0.010230538597223324, 0.005087330324623459], [0.0050268297170800655, 0.015095098383136147, 0.010230538597223324, 0.005492685170680188], [0.0051780140374396945, 0.015095098383136147, 0.010248881669894343, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.011295270096288279, 0.03445885724201088, 0.031160719730233592, 0.005059373615493241], [0.011295270096288279, 0.03445885724201088, 0.031160719730233592, 0.005059373615493241], [0.007041627625192543, 0.03445885724201088, 0.031160719730233592, 0.005059373615493241], [0.005616784423980166, 0.029388437967006537, 0.030704126063068788, 0.005059373615493241], [0.006383382443470326, 0.031409463659906815, 0.03137012601528287, 0.0056370687256551665], [0.0051780140374396945, 0.02078570668779144, 0.010248881669894343, 0.005492685170680188], [0.005059771007954962, 0.01735203978655234, 0.012258842535878174, 0.005049045397022951], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.00600534463808118], [0.011786727858322344, 0.015193421734091711, 0.030317633351563414, 0.005492685170680188], [0.00505175276532252, 0.03162755346964442, 0.03052366602505172, 0.0056370687256551665], [0.0050268297170800655, 0.015027893177736442, 0.010230538597223324, 0.005087330324623459], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.011295270096288279, 0.03445885724201088, 0.031160719730233592, 0.005059373615493241], [0.011295270096288279, 0.03445885724201088, 0.031160719730233592, 0.005059373615493241], [0.007041627625192543, 0.03445885724201088, 0.031160719730233592, 0.005059373615493241], [0.006383382443470326, 0.031409463659906815, 0.03137012601528287, 0.0056370687256551665], [0.005059271612634917, 0.03142539692005131, 0.010738255914706249, 0.005039069109868905], [0.0051780140374396945, 0.02078570668779144, 0.010248881669894343, 0.005492685170680188], [0.005059771007954962, 0.01735203978655234, 0.012258842535878174, 0.005049045397022951], [0.005616784423980166, 0.029388437967006537, 0.030704126063068788, 0.005059373615493241], [0.00505175276532252, 0.03162755346964442, 0.031190837592775095, 0.0056370687256551665], [0.00505175276532252, 0.03162755346964442, 0.03052366602505172, 0.0056370687256551665], [0.005061932728730493, 0.015193421734091711, 0.011962324091527336, 0.005492685170680188], [0.0075647088670546606, 0.031796617020426104, 0.031049828596777407, 0.0056370687256551665], [0.0050268297170800655, 0.015027893177736442, 0.010230538597223324, 0.005087330324623459], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005059271612634917, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.015193421734091711, 0.010738255914706249, 0.005492685170680188], [0.005061932728730493, 0.01735203978655234, 0.011268678682758312, 0.005492685170680188], [0.0058685452594453204, 0.02100540640213748, 0.011955587534010458, 0.005492685170680188], [0.011295270096288279, 0.03445885724201088, 0.031160719730233592, 0.005059373615493241], [0.011295270096288279, 0.03445885724201088, 0.031160719730233592, 0.005059373615493241], [0.005059271612634917, 0.03142539692005131, 0.010738255914706249, 0.005039069109868905], [0.007041627625192543, 0.03445885724201088, 0.031160719730233592, 0.005059373615493241], [0.006383382443470326, 0.031409463659906815, 0.03137012601528287, 0.0056370687256551665], [0.007363844358816061, 0.03445885724201088, 0.031160719730233592, 0.005059373615493241], [0.0051780140374396945, 0.02078570668779144, 0.010248881669894343, 0.005492685170680188], [0.005059771007954962, 0.01735203978655234, 0.012258842535878174, 0.005049045397022951], [0.005061932728730493, 0.031745678318059495, 0.030896745858946653, 0.005170022302926589], [0.005616784423980166, 0.029388437967006537, 0.030704126063068788, 0.005059373615493241], [0.00505175276532252, 0.03162755346964442, 0.03052366602505172, 0.0056370687256551665], [0.0058685452594453204, 0.020080565820133307, 0.011955587534010458, 0.005492685170680188]]
    multi_nsga_kpls_y = [[(-x[0] * N0LV * 1e-9), x[1]] for x in [[-1.2689571134136741, 0.6244165033122802], [-1.3038517177208198, 0.7396618598465131], [-1.147095507072986, 0.4375943218343832], [-1.2472044527342736, 0.5192135915649068], [-1.3827591464483164, 0.6053889713247951], [-1.3990656032666637, 0.7079469774481327], [-1.334954996547624, 0.7560288874723599], [-1.413499309142585, 0.6443113019463702], [-1.4100922491023815, 0.7263402293193771], [-1.4420112662234084, 0.8908090528975411], [-1.2076522996588914, 0.506000340848033], [-1.3394148735146443, 0.6345045784384625], [-1.1000493680235808, 0.7257574151260641], [-1.3967672325307425, 0.6294213244228817], [-1.2217205819869226, 0.5843158337112597], [-1.376494094562525, 0.5935367874979155], [-1.3573174041150589, 0.6299982596865922], [-1.2962319049746935, 0.715169043940356], [-1.2367799628594882, 0.6145274412543943], [-0.9509598324638487, 0.40260897484381886], [-1.3665169120959997, 0.6130137665558946], [-1.2876630913072962, 0.4931556598967067], [-1.3553702673797343, 0.6510256796078441], [-1.2316718868897871, 0.6933123338557624], [-1.1882089625087726, 0.4564884249144718], [-1.384843264874394, 0.6711751398130975], [-1.2657325624697606, 0.6847979445912701], [-1.2243525971212113, 0.5179619529979066], [-1.1983307708263269, 0.5805815465296843], [-1.1170872485487635, 0.5265207855664643], [-1.3110120460683807, 0.680615665166488], [-1.17934532132684, 0.47334229568173686], [-1.1100537651829903, 0.5290515910396686], [-1.3799787110136803, 0.6794551735285669], [-1.3898700367221932, 0.574754224209836], [-1.4077044809732726, 0.7489622624876133], [-1.2317420848687826, 0.5166290285836481], [-1.3023579131976404, 0.6828663945275771], [-1.3224647985837101, 0.6752739722035], [-1.2801811919153339, 0.5949620619731352], [-1.3672571906453248, 0.8058986355292264], [-1.4054449530554372, 0.6524729261494334], [-1.3586244528990823, 0.4928739692894987], [-1.3408605778307714, 0.6609608283565579], [-1.3762877577127182, 0.838888251677414], [-1.3551913977817502, 0.8192151094314973], [-1.2261365950820342, 0.6099708287769456], [-1.4345180976024117, 0.8042832811356924], [-1.148753197058564, 0.5374384332993067], [-1.2909766526861586, 0.6865217749994639], [-1.457703567033487, 0.797445254427495], [-1.4353108288718872, 0.7299905359569061], [-1.3786180461518858, 0.6402361433477087], [-1.3114853832325037, 0.6327269534396075], [-1.2952477778066354, 0.6366415690148853], [-1.358211147731086, 0.6926608574428564], [-1.2744788409513936, 0.5334982732898967], [-1.206468633802778, 0.7152194143635638], [-1.4649702849592003, 0.6260622285501143], [-1.3724499785127398, 0.6807565761577007], [-1.2132999909027105, 0.5049059757580039], [-1.1307220023032463, 0.5244007598218815], [-1.2043790646729475, 0.5355161164440878], [-1.2289290973072295, 0.5763723176726728], [-1.2437525534917555, 0.5775354414204766], [-1.2524988801136268, 0.5797705807977113], [-1.3374372071450908, 0.6195832089412937], [-1.3799251935008947, 0.6287559690455403], [-1.2737348778201028, 0.6344636318424515], [-1.3140104830081132, 0.6442805035509993], [-1.3023610935243588, 0.6603841685614844], [-1.3299999705609162, 0.6659350257520622], [-1.4078584329765826, 0.6795819364205685], [-1.3856258893018367, 0.7227994218374332], [-1.3369756433905085, 0.7351371505917226], [-1.384065277623368, 0.743387909281757], [-1.430675688171494, 0.7546982106102411], [-1.4184593828237229, 0.7643395201316656], [-1.380514988400613, 0.8003607759656122], [-1.427560887950017, 0.8880572299972909], [-1.2133559470105315, 0.5045101346242267], [-1.430675688171494, 0.7546982106102411], [-1.4486271921527352, 0.8986420659739992], [-1.2437525534917555, 0.5775354414204766], [-1.4078584329765826, 0.6795819364205685], [-1.3799251935008947, 0.6287559690455403], [-1.2684225946543728, 0.602390672506377], [-1.3275114112365163, 0.6153298133550186], [-1.2289290973072295, 0.5763723176726728], [-1.426696980398686, 0.7451496487315438], [-1.25452807844673, 0.5792322420990719], [-1.3374372071450908, 0.6195832089412937], [-1.3799251935008947, 0.6287559690455403], [-1.2132999909027105, 0.5049059757580039], [-1.427560887950017, 0.8880572299972909], [-1.2737348778201028, 0.6344636318424515], [-1.4184593828237229, 0.7643395201316656], [-1.3299999705609162, 0.6659350257520622], [-1.3290710032099762, 0.6626530400899534], [-1.4260443548917963, 0.8828585393497335], [-1.2133559470105315, 0.5045101346242267], [-1.4578207984221077, 0.7415384691713905], [-1.4078584329765826, 0.6795819364205685], [-1.3799251935008947, 0.6287559690455403], [-1.2450008518658509, 0.5775176225104045], [-1.3799251935008947, 0.6287559690455403], [-1.3799251935008947, 0.6287559690455403], [-1.25452807844673, 0.5792322420990719], [-1.25452807844673, 0.5792322420990719], [-1.3789658564252187, 0.5952644471273828], [-1.2474356919046332, 0.5782313555649407], [-1.2289290973072295, 0.5763723176726728], [-1.3803536719517129, 0.6329049281891582], [-1.2132999909027105, 0.5049059757580039], [-1.430675688171494, 0.7546982106102411], [-1.4486271921527352, 0.8986420659739992], [-1.2437525534917555, 0.5775354414204766], [-1.426945953797286, 0.7454371976243775], [-1.3374372071450908, 0.6195832089412937], [-1.3801679918344438, 0.6393406158995882], [-1.2133559470105315, 0.5045101346242267], [-1.4578207984221077, 0.7415384691713905], [-1.4104591825999666, 0.6758553941861074], [-1.3799251935008947, 0.6287559690455403], [-1.2579722692403557, 0.5327058072632541], [-1.3996184327901777, 0.6435929722339843], [-1.3799251935008947, 0.6287559690455403], [-1.3799251935008947, 0.6287559690455403], [-1.3799251935008947, 0.6287559690455403], [-1.4313582866591679, 0.6936306365332874], [-1.3802986741706922, 0.6323955965677194], [-1.4501060073672976, 0.7036595762240327], [-1.3789658564252187, 0.5952644471273828], [-1.3803536719517129, 0.6329049281891582], [-1.4039449368004002, 0.6687226326985724], [-1.2132999909027105, 0.5049059757580039], [-1.2450008518658509, 0.5775176225104045], [-1.3801679918344438, 0.6393406158995882], [-1.25452807844673, 0.5792322420990719], [-1.25452807844673, 0.5792322420990719], [-1.2133559470105315, 0.5045101346242267], [-1.4578207984221077, 0.7415384691713905], [-1.2579722692403557, 0.5327058072632541], [-1.385727427017057, 0.6317147920213332], [-1.4501060073672976, 0.7036595762240327], [-1.379897412061667, 0.6284708474519793], [-1.3799251935008947, 0.6287559690455403], [-1.3789658564252187, 0.5952644471273828], [-1.4313582866591679, 0.6936306365332874], [-1.3799251935008947, 0.6287559690455403], [-1.3799251935008947, 0.6287559690455403], [-1.270717273182516, 0.5453707090880555], [-1.3799251935008947, 0.6287559690455403], [-1.3799198974622418, 0.6287017877210145], [-1.3799251935008947, 0.6287559690455403], [-1.3996184327901777, 0.6435929722339843], [-1.3796200043011706, 0.6156679800287881], [-1.3698457110595514, 0.5908377355921116], [-1.4039449368004002, 0.6687226326985724], [-1.4179850132753327, 0.6713501790610928], [-1.2133559470105315, 0.5045101346242267], [-1.4578207984221077, 0.7415384691713905], [-1.4501060073672976, 0.7036595762240327], [-1.4313582866591679, 0.6936306365332874], [-1.4179850132753327, 0.6713501790610928], [-1.3996184327901777, 0.6435929722339843], [-1.2579722692403557, 0.5327058072632541], [-1.2504579922859331, 0.5268047081914793], [-1.3794012886247868, 0.5928956001278083], [-1.270717273182516, 0.5453707090880555], [-1.3996184327901777, 0.6435929722339843], [-1.2537680666354845, 0.5297198785712214], [-1.3836015406077302, 0.5947849339087179], [-1.2707788564395148, 0.5454444840309243], [-1.4501060073672976, 0.7036595762240327], [-1.3698457110595514, 0.5908377355921116], [-1.27576291789251, 0.5460279746883162], [-1.3932003872543859, 0.6218021806763689], [-1.4270535041750652, 0.6720701733798173], [-1.4039449368004002, 0.6687226326985724], [-1.2133559470105315, 0.5045101346242267], [-1.4606873902139208, 0.6847091059415705], [-1.4270535041750652, 0.6720701733798173], [-1.3836015406077302, 0.5947849339087179], [-1.3996184327901777, 0.6435929722339843], [-1.3988145393020035, 0.6044571123953453], [-1.3996184327901777, 0.6435929722339843], [-1.4249869415943115, 0.6617049844741351], [-1.2579722692403557, 0.5327058072632541], [-1.3996184327901777, 0.6435929722339843], [-1.2504579922859331, 0.5268047081914793], [-1.270717273182516, 0.5453707090880555], [-1.2537680666354845, 0.5297198785712214], [-1.270717273182516, 0.5453707090880555], [-1.2504579922859331, 0.5268047081914793], [-1.2707788564395148, 0.5454444840309243], [-1.2707788564395148, 0.5454444840309243], [-1.399815103023462, 0.6586904639618529], [-1.3832846128109764, 0.5919023403283787], [-1.3698457110595514, 0.5908377355921116], [-1.2133559470105315, 0.5045101346242267], [-1.2133559470105315, 0.5045101346242267], [-1.4606873902139208, 0.6847091059415705], [-1.4270535041750652, 0.6720701733798173], [-1.3698457110595514, 0.5908377355921116], [-1.3836015406077302, 0.5947849339087179], [-1.3836015406077302, 0.5947849339087179], [-1.2537680666354845, 0.5297198785712214], [-1.239453869424662, 0.5240580153398868], [-1.2711123945546774, 0.540345285336816], [-1.2504579922859331, 0.5268047081914793], [-1.4249869415943115, 0.6617049844741351], [-1.2579722692403557, 0.5327058072632541], [-1.2504579922859331, 0.5268047081914793], [-1.2504579922859331, 0.5268047081914793], [-1.2748227613405456, 0.5463720219525248], [-1.257151076679519, 0.5320209474909454], [-1.4157920164258788, 0.6021612673299115], [-1.3616819925927783, 0.5844492842865326], [-1.2812526243422533, 0.572678456650736], [-1.2133559470105315, 0.5045101346242267], [-1.2133559470105315, 0.5045101346242267], [-1.4606873902139208, 0.6847091059415705], [-1.3958103330401135, 0.5911865894552484], [-1.3698457110595514, 0.5908377355921116], [-1.4270535041750652, 0.6720701733798173], [-1.4499405340691744, 0.6734305699632785], [-1.3616819925927783, 0.5844492842865326], [-1.239453869424662, 0.5240580153398868], [-1.2537680666354845, 0.5297198785712214], [-1.425051968210731, 0.6621624860195434], [-1.4554949030936826, 0.6815478682003129], [-1.2342839221875843, 0.5163852114656641], [-1.3606203750527854, 0.5748025715701714], [-1.239453869424662, 0.5240580153398868], [-1.4249869415943115, 0.6617049844741351], [-1.2826957528755585, 0.5389588728900986], [-1.2504579922859331, 0.5268047081914793], [-1.2504579922859331, 0.5268047081914793], [-1.2504579922859331, 0.5268047081914793], [-1.104229703399312, 0.47649654071474], [-1.2133559470105315, 0.5045101346242267], [-1.2133559470105315, 0.5045101346242267], [-1.4606873902139208, 0.6847091059415705], [-1.3958103330401135, 0.5911865894552484], [-1.4554949030936826, 0.6815478682003129], [-1.374218160687231, 0.5819820276285833], [-1.2537680666354845, 0.5297198785712214], [-1.2504579922859331, 0.5268047081914793], [-1.2352916298998724, 0.5078779553945463], [-1.2480819115153012, 0.518426712472109], [-1.207741447681216, 0.5014373432424907], [-1.4521651369584792, 0.6523366509708252], [-1.3624754990143404, 0.5756486949761461], [-1.2504768308669822, 0.5271749850283298], [-1.2504579922859331, 0.5268047081914793], [-1.2504579922859331, 0.5268047081914793], [-1.2504579922859331, 0.5268047081914793], [-1.2826957528755585, 0.5389588728900986], [-1.3606203750527854, 0.5748025715701714], [-1.104229703399312, 0.47649654071474], [-1.2133559470105315, 0.5045101346242267], [-1.2133559470105315, 0.5045101346242267], [-1.4606873902139208, 0.6847091059415705], [-1.3864288587629472, 0.5869405425009456], [-1.3624754990143404, 0.5756486949761461], [-1.3958103330401135, 0.5911865894552484], [-1.4521651369584792, 0.6523366509708252], [-1.374218160687231, 0.5819820276285833], [-1.2612433645423797, 0.5305286921564967], [-1.207741447681216, 0.5014373432424907], [-1.2352916298998724, 0.5078779553945463], [-1.4432010192827933, 0.6443603249629032], [-1.207741447681216, 0.5014373432424907], [-1.2826957528755585, 0.5389588728900986], [-1.4581690427830338, 0.6565284194220629], [-1.4498522953988209, 0.6505921767543847], [-1.1750911869544418, 0.4840958137756588], [-1.2128005294270123, 0.5043602487919165], [-1.2564107705932948, 0.5175109236760789], [-1.104229703399312, 0.47649654071474], [-1.4606873902139208, 0.6847091059415705], [-1.3864288587629472, 0.5869405425009456], [-1.3742343244138606, 0.5821319932262542], [-1.4581690427830338, 0.6565284194220629], [-1.4521651369584792, 0.6523366509708252], [-1.3958103330401135, 0.5911865894552484], [-1.2564107705932948, 0.5175109236760789], [-1.374218160687231, 0.5819820276285833], [-1.2352916298998724, 0.5078779553945463], [-1.1750911869544418, 0.4840958137756588], [-1.2149939234429106, 0.49938073625274426], [-1.4417271017668774, 0.6327807962222837], [-1.374218160687231, 0.5819820276285833], [-1.4530706889975604, 0.6533071380509774], [-1.4498522974829269, 0.6505921386811039], [-1.4187384888492476, 0.6281727707257884], [-1.4433123085764548, 0.6449119932764377], [-1.4432010192827933, 0.6443603249629032], [-1.2352916298998724, 0.5078779553945463], [-1.104229703399312, 0.47649654071474], [-1.4663966321000457, 0.6726311540405917], [-1.3694186561137707, 0.5801505294399686], [-1.3741304738215874, 0.5811718046795581], [-1.2352916298998724, 0.5078779553945463], [-1.2352916298998724, 0.5078779553945463], [-1.4417271017668774, 0.6327807962222837], [-1.4432010192827933, 0.6443603249629032], [-1.4022031499751142, 0.5994315835495192], [-1.4498522974829269, 0.6505921386811039], [-1.2149939234429106, 0.49938073625274426], [-1.4530706889975604, 0.6533071380509774], [-1.3958103330401135, 0.5911865894552484], [-1.374218160687231, 0.5819820276285833], [-1.4521651369584792, 0.6523366509708252], [-1.3959587206006197, 0.591341767628295], [-1.4433123085764548, 0.6449119932764377], [-1.268691566978776, 0.5517880231891948], [-1.374218160687231, 0.5819820276285833], [-1.374218160687231, 0.5819820276285833], [-1.104229703399312, 0.47649654071474], [-1.4663966321000457, 0.6726311540405917], [-1.268691566978776, 0.5517880231891948], [-1.3741304738215874, 0.5811718046795581], [-1.3711601277270014, 0.5772494906671155], [-1.3659444254617223, 0.5585589524977489], [-1.268691566978776, 0.5517880231891948], [-1.4022031499751142, 0.5994315835495192], [-1.4417271017668774, 0.6327807962222837], [-1.3830638215167508, 0.5858160791450749], [-1.4530706889975604, 0.6533071380509774], [-1.4498522974829269, 0.6505921386811039], [-1.449316978724597, 0.6399549231432182], [-1.374218160687231, 0.5819820276285833], [-1.4023889003533856, 0.5996866058902572], [-1.4127993328214346, 0.5997277689000766], [-1.4521651369584792, 0.6523366509708252], [-1.374218160687231, 0.5819820276285833], [-1.2149939234429106, 0.49938073625274426], [-1.374218160687231, 0.5819820276285833], [-1.104229703399312, 0.47649654071474], [-1.4663966321000457, 0.6726311540405917], [-1.3113242513379408, 0.5380272483125965], [-1.384425195316026, 0.5855280901998842], [-1.3863732121597327, 0.5899351014259487], [-1.2149939234429106, 0.49938073625274426], [-1.449316978724597, 0.6399549231432182], [-1.4426177091097012, 0.6273892248489884], [-1.4534951766733513, 0.6590213798777552], [-1.4530706889975604, 0.6533071380509774], [-1.4498522974829269, 0.6505921386811039], [-1.449316978724597, 0.6399549231432182], [-1.3841188416706396, 0.5854263773090417], [-1.4023889003533856, 0.5996866058902572], [-1.4127993328214346, 0.5997277689000766], [-1.4521651369584792, 0.6523366509708252], [-1.4498522974829269, 0.6505921386811039], [-1.415536232394105, 0.6020612462639412], [-1.4521651369584792, 0.6523366509708252], [-1.2149939234429106, 0.49938073625274426], [-1.104229703399312, 0.47649654071474], [-1.4666825696127588, 0.6712972220669505], [-1.3113242513379408, 0.5380272483125965], [-1.384425195316026, 0.5855280901998842], [-1.3863274469226385, 0.589353152548291], [-1.3863732121597327, 0.5899351014259487], [-1.2149939234429106, 0.49938073625274426], [-1.4426177091097012, 0.6273892248489884], [-1.449316978724597, 0.6399549231432182], [-1.2160889866954798, 0.5000099262451874], [-1.415536232394105, 0.6020612462639412], [-1.4475488651004857, 0.63819533818939], [-1.4606455732151769, 0.6606943620328368], [-1.4070349115069105, 0.592923299625291], [-1.4497718484677824, 0.642940527131616], [-1.449316978724597, 0.6399549231432182], [-1.4561522409881882, 0.657117622271022], [-1.2149939234429106, 0.49938073625274426], [-1.3841188416706396, 0.5854263773090417], [-1.4530945289615254, 0.6440895716068351], [-1.104229703399312, 0.47649654071474], [-1.4666825696127588, 0.6712972220669505], [-1.3113242513379408, 0.5380272483125965], [-1.449316978724597, 0.6399549231432182], [-1.2149939234429106, 0.49938073625274426], [-1.4426177091097012, 0.6273892248489884], [-1.2257957868669997, 0.5344770583179117], [-1.449316978724597, 0.6399549231432182], [-1.2160889866954798, 0.5000099262451874], [-1.4426177091097012, 0.6273892248489884], [-1.415536232394105, 0.6020612462639412], [-1.4606455732151769, 0.6606943620328368], [-1.4151142011019395, 0.6013818422543429], [-1.4606455732151769, 0.6606943620328368], [-1.4530945289615254, 0.6440895716068351], [-1.415536232394105, 0.6020612462639412], [-1.4495653264355617, 0.6408933579373721], [-1.4561522409881882, 0.657117622271022], [-1.4497718484677824, 0.642940527131616], [-1.4475488651004857, 0.63819533818939], [-1.104229703399312, 0.47649654071474], [-1.104229703399312, 0.47649654071474], [-1.4666825696127588, 0.6712972220669505], [-1.4666825696127588, 0.6712972220669505], [-1.3113242513379408, 0.5380272483125965], [-1.2257957868669997, 0.5344770583179117], [-1.4426177091097012, 0.6273892248489884], [-1.4085819841004439, 0.5986784074756679], [-1.4606455732151769, 0.6606943620328368], [-1.4530521153552591, 0.6439354975368157], [-1.4475488651004857, 0.63819533818939], [-1.4607737297708663, 0.6619335944281041], [-1.4426177091097012, 0.6273892248489884], [-1.415536232394105, 0.6020612462639412], [-1.2149939234429106, 0.49938073625274426], [-1.4606455732151769, 0.6606943620328368], [-1.4115185664043381, 0.5999998576510349], [-1.4151142011019395, 0.6013818422543429], [-1.4606455732151769, 0.6606943620328368], [-1.4530945289615254, 0.6440895716068351], [-1.0900494666959484, 0.45194266361371266], [-1.104229703399312, 0.47649654071474], [-1.104229703399312, 0.47649654071474], [-1.4666825696127588, 0.6712972220669505], [-1.4666825696127588, 0.6712972220669505], [-1.3113242513379408, 0.5380272483125965], [-1.2257957868669997, 0.5344770583179117], [-1.4426177091097012, 0.6273892248489884], [-1.2149939234429106, 0.49938073625274426], [-1.4607737297708663, 0.6619335944281041], [-1.4606455732151769, 0.6606943620328368], [-1.4475488651004857, 0.63819533818939], [-1.4530521153552591, 0.6439354975368157], [-1.2149939234429106, 0.49938073625274426], [-1.4426177091097012, 0.6273892248489884], [-1.4571461531856094, 0.6558168486685239], [-1.415536232394105, 0.6020612462639412], [-1.4606455732151769, 0.6606943620328368], [-1.4530521153552591, 0.6439354975368157], [-1.4606455732151769, 0.6606943620328368], [-1.0900494666959484, 0.45194266361371266], [-1.104229703399312, 0.47649654071474], [-1.104229703399312, 0.47649654071474], [-1.4679161075645473, 0.6492084071845371], [-1.4426177091097012, 0.6273892248489884], [-1.0969790118975844, 0.47507326156108565], [-1.2149939234429106, 0.49938073625274426], [-1.3113242513379408, 0.5380272483125965], [-1.415536232394105, 0.6020612462639412], [-1.2149939234429106, 0.49938073625274426], [-1.104229703399312, 0.47649654071474], [-1.4087328838763287, 0.5883334669555581], [-1.4479019533077517, 0.6330445583498168], [-1.456798273231737, 0.6454643251332982], [-1.4481086812660333, 0.63874222309748], [-1.453571333054662, 0.6418834879964583], [-1.4426177091097012, 0.6273892248489884], [-1.4165402989169542, 0.6176817155777348], [-1.2417505957029928, 0.5261023382624274], [-1.4666825696127588, 0.6712972220669505], [-1.0900494666959484, 0.45194266361371266], [-1.104229703399312, 0.47649654071474], [-1.104229703399312, 0.47649654071474], [-1.470858535592954, 0.6624666571375569], [-1.4426177091097012, 0.6273892248489884], [-1.0969790118975844, 0.47507326156108565], [-1.2149939234429106, 0.49938073625274426], [-1.3113242513379408, 0.5380272483125965], [-1.1097007141097681, 0.4876811235635332], [-1.415536232394105, 0.6020612462639412], [-1.4087328838763287, 0.5883334669555581], [-1.104229703399312, 0.47649654071474], [-1.2149939234429106, 0.49938073625274426], [-1.3113242513379408, 0.5380272483125965], [-1.456798273231737, 0.6454643251332982], [-1.2152258508586906, 0.49978947429302967], [-1.4481086812660333, 0.63874222309748], [-1.4479019533077517, 0.6330445583498168], [-1.2513142030543225, 0.5044608579449198], [-1.4426177091097012, 0.6273892248489884], [-1.0900494666959484, 0.45194266361371266], [-1.104229703399312, 0.47649654071474], [-1.104229703399312, 0.47649654071474], [-1.470858535592954, 0.6624666571375569], [-1.4426177091097012, 0.6273892248489884], [-1.0969790118975844, 0.47507326156108565], [-1.3113242513379408, 0.5380272483125965], [-1.4087328838763287, 0.5883334669555581], [-1.463042253208864, 0.643213314930702], [-1.4071222281347675, 0.5874555044304753], [-1.415536232394105, 0.6020612462639412], [-1.104229703399312, 0.47649654071474], [-1.4479019533077517, 0.6330445583498168], [-1.442620473752424, 0.6273986532327857], [-1.4426177091097012, 0.6273892248489884], [-1.4479019533077517, 0.6330445583498168], [-1.2153096092444686, 0.4982899557606815], [-1.3113242513379408, 0.5380272483125965], [-1.104229703399312, 0.47649654071474], [-1.4162035145516032, 0.6047247549574096], [-1.1177517404718946, 0.44709971897170775], [-1.470858535592954, 0.6624666571375569], [-1.118259756306137, 0.4780478991985396], [-1.4426177091097012, 0.6273892248489884], [-1.4400249721279454, 0.6220947485259909], [-1.3113242513379408, 0.5380272483125965], [-1.4431480317823957, 0.6286018216949842], [-1.4426177091097012, 0.6273892248489884], [-1.3113242513379408, 0.5380272483125965], [-1.463042253208864, 0.643213314930702], [-1.442620473752424, 0.6273986532327857], [-1.4426177091097012, 0.6273892248489884], [-1.2153096092444686, 0.4982899557606815], [-1.4632025779018065, 0.643720965095888], [-1.4180762969829568, 0.5791092240785036], [-1.4480987193555788, 0.6310461341345885], [-1.1205842507563442, 0.47980657153960304], [-1.0900494666959484, 0.45194266361371266], [-1.104229703399312, 0.47649654071474], [-1.104229703399312, 0.47649654071474], [-1.1177517404718946, 0.44709971897170775], [-1.1177517404718946, 0.44709971897170775], [-1.470858535592954, 0.6624666571375569], [-1.118259756306137, 0.4780478991985396], [-1.4400249721279454, 0.6220947485259909], [-1.4395253949763303, 0.6097904355157685], [-1.4398412964051799, 0.6214691710854187], [-1.4431480317823957, 0.6286018216949842], [-1.2144042833566229, 0.497769506890977], [-1.155000485194974, 0.4858761368217383], [-1.3113242513379408, 0.5380272483125965], [-1.4665538210077262, 0.6558664761518727], [-1.4632025779018065, 0.643720965095888], [-1.1207505600532153, 0.4799067849497642], [-1.4632826648194148, 0.6439837926562718], [-1.4685258563467551, 0.6608690170150496], [-1.4632025779018065, 0.643720965095888], [-1.4428405480572866, 0.6231525728633283], [-1.296879084528753, 0.530849075092688], [-1.2153096092444686, 0.4982899557606815], [-1.1171822819186006, 0.4458701261129736], [-1.1177517404718946, 0.44709971897170775], [-1.1177517404718946, 0.44709971897170775], [-1.470858535592954, 0.6624666571375569], [-1.4665538210077262, 0.6558664761518727], [-1.4632025779018065, 0.643720965095888], [-1.3364063459213849, 0.554891320917306], [-1.3924990691273083, 0.6085253691112048], [-1.4395253949763303, 0.6097904355157685], [-1.2153096092444686, 0.4982899557606815], [-1.4431480317823957, 0.6286018216949842], [-1.4632025779018065, 0.643720965095888], [-1.155000485194974, 0.4858761368217383], [-1.4665538210077262, 0.6558664761518727], [-1.4685258563467551, 0.6608690170150496], [-1.4632025779018065, 0.643720965095888], [-1.1192281329408746, 0.47988864923296204], [-1.444663229395262, 0.6292752068048999], [-1.155000485194974, 0.4858761368217383], [-1.1207505600532153, 0.4799067849497642], [-1.1171822819186006, 0.4458701261129736], [-1.1177517404718946, 0.44709971897170775], [-1.1177517404718946, 0.44709971897170775], [-1.470858535592954, 0.6624666571375569], [-1.3924990691273083, 0.6085253691112048], [-1.444663229395262, 0.6292752068048999], [-1.4395253949763303, 0.6097904355157685], [-1.3364063459213849, 0.554891320917306], [-1.4665538210077262, 0.6558664761518727], [-1.3697031163035391, 0.5891748089550578], [-1.1235459982634195, 0.4532367775452357], [-1.2177424942523765, 0.4989156158611026], [-1.4665538210077262, 0.6558664761518727], [-1.4632025779018065, 0.643720965095888], [-1.4431480317823957, 0.6286018216949842], [-1.3253897003495578, 0.5448298101789724], [-1.4632025779018065, 0.643720965095888], [-1.155000485194974, 0.4858761368217383], [-1.4683406016629874, 0.6600054576829341], [-1.2153096092444686, 0.4982899557606815], [-1.078967521457503, 0.4393611462362538], [-1.1171822819186006, 0.4458701261129736], [-1.1177517404718946, 0.44709971897170775], [-1.1177517404718946, 0.44709971897170775], [-1.470858535592954, 0.6624666571375569], [-1.155000485194974, 0.4858761368217383], [-1.1457149319540967, 0.4533235285971271], [-1.444663229395262, 0.6292752068048999], [-1.370137078616279, 0.5935976842908873], [-1.3364063459213849, 0.554891320917306], [-1.4431480317823957, 0.6286018216949842], [-1.3697031163035391, 0.5891748089550578], [-1.4395253949763303, 0.6097904355157685], [-1.4644592187247611, 0.64917987518248], [-1.1235459982634195, 0.4532367775452357], [-1.4632025779018065, 0.643720965095888], [-1.3948270351875949, 0.6075343621941734], [-1.2177424942523765, 0.4989156158611026], [-1.3253897003495578, 0.5448298101789724], [-1.2153096092444686, 0.4982899557606815], [-1.0268036145785462, 0.4207100553171897], [-1.078967521457503, 0.4393611462362538], [-1.1171822819186006, 0.4458701261129736], [-1.1177517404718946, 0.44709971897170775], [-1.1177517404718946, 0.44709971897170775], [-1.470858535592954, 0.6624666571375569], [-1.155000485194974, 0.4858761368217383], [-1.370137078616279, 0.5935976842908873], [-1.1467845634999745, 0.45557527640642675], [-1.3599039403650341, 0.5754868306230191], [-1.1457149319540967, 0.4533235285971271], [-1.444663229395262, 0.6292752068048999], [-1.3948270351875949, 0.6075343621941734], [-1.3697031163035391, 0.5891748089550578], [-1.4431480317823957, 0.6286018216949842], [-1.4644592187247611, 0.64917987518248], [-1.2153096092444686, 0.4982899557606815], [-1.3270682496950192, 0.5360369890428341], [-1.1177517404718946, 0.44709971897170775], [-1.4632025779018065, 0.643720965095888], [-1.0268036145785462, 0.4207100553171897], [-1.078967521457503, 0.4393611462362538], [-1.1171822819186006, 0.4458701261129736], [-1.470858535592954, 0.6624666571375569], [-1.155000485194974, 0.4858761368217383], [-1.370137078616279, 0.5935976842908873], [-1.1467845634999745, 0.45557527640642675], [-1.3948270351875949, 0.6075343621941734], [-1.3599039403650341, 0.5754868306230191], [-1.4431480317823957, 0.6286018216949842], [-1.155000485194974, 0.4858761368217383], [-1.1457149319540967, 0.4533235285971271], [-1.3697031163035391, 0.5891748089550578], [-1.3270682496950192, 0.5360369890428341], [-1.444663229395262, 0.6292752068048999], [-1.4644592187247611, 0.64917987518248], [-1.444663229395262, 0.6292752068048999], [-1.2153096092444686, 0.4982899557606815], [-1.4465586868178235, 0.6379797551786744], [-1.1182099918683044, 0.4462615783296503], [-1.0268036145785462, 0.4207100553171897], [-1.078967521457503, 0.4393611462362538], [-1.1171822819186006, 0.4458701261129736], [-1.470858535592954, 0.6624666571375569], [-1.1467845634999745, 0.45557527640642675], [-1.370137078616279, 0.5935976842908873], [-1.3948270351875949, 0.6075343621941734], [-1.1592050389279434, 0.49283905600571826], [-1.2153096092444686, 0.4982899557606815], [-1.3599039403650341, 0.5754868306230191], [-1.444663229395262, 0.6292752068048999], [-1.1457149319540967, 0.4533235285971271], [-1.3341274835032007, 0.5437477476996755], [-1.3697031163035391, 0.5891748089550578], [-1.1412298005984378, 0.4511840830365818], [-1.4644592187247611, 0.64917987518248], [-1.4667595086879601, 0.6570737973330153], [-1.4685258563467551, 0.6608690170150496], [-1.444663229395262, 0.6292752068048999], [-1.1182099918683044, 0.4462615783296503], [-1.0106993832013034, 0.4182447191250551], [-1.0268036145785462, 0.4207100553171897], [-1.078967521457503, 0.4393611462362538], [-1.1171822819186006, 0.4458701261129736], [-1.470858535592954, 0.6624666571375569], [-1.370137078616279, 0.5935976842908873], [-1.1467845634999745, 0.45557527640642675], [-1.3948270351875949, 0.6075343621941734], [-1.2153096092444686, 0.4982899557606815], [-1.1182099918683044, 0.4462615783296503], [-1.431191969400899, 0.6196156221637219], [-1.0358038330164665, 0.4319867533921955], [-1.0280463716863004, 0.4209070085296982], [-1.1457149319540967, 0.4533235285971271], [-1.3599039403650341, 0.5754868306230191], [-1.2135455326569522, 0.4867660615145986], [-1.444663229395262, 0.6292752068048999], [-1.3490713329856707, 0.5588218361743152], [-1.4685069991035435, 0.6607870256314989], [-1.3341274835032007, 0.5437477476996755], [-1.1081245660606112, 0.4161433472761561], [-1.1171822819186006, 0.4458701261129736], [-1.470858535592954, 0.6624666571375569], [-1.370137078616279, 0.5935976842908873], [-1.3948270351875949, 0.6075343621941734], [-1.1826824216636223, 0.45295914919494], [-1.2153096092444686, 0.4982899557606815], [-1.1182099918683044, 0.4462615783296503], [-1.3599039403650341, 0.5754868306230191], [-1.444663229395262, 0.6292752068048999], [-1.431191969400899, 0.6196156221637219], [-1.2135455326569522, 0.4867660615145986], [-1.3552501142535127, 0.5740429505632024], [-1.3490713329856707, 0.5588218361743152], [-1.4685069991035435, 0.6607870256314989], [-1.337121483880982, 0.5452159818237586], [-1.3342299942559126, 0.5432078718754086], [-1.3599039403650341, 0.5754868306230191], [-1.464249698529045, 0.6514620602758356], [-1.444663229395262, 0.6292752068048999], [-1.1081245660606112, 0.4161433472761561], [-1.1171822819186006, 0.4458701261129736], [-1.470858535592954, 0.6624666571375569], [-1.370137078616279, 0.5935976842908873], [-1.3948270351875949, 0.6075343621941734], [-1.1826824216636223, 0.45295914919494], [-1.1133893319459176, 0.42975169535593183], [-1.2153096092444686, 0.4982899557606815], [-1.3490427815430868, 0.5587841485456753], [-1.3599039403650341, 0.5754868306230191], [-1.464249698529045, 0.6514620602758356], [-1.352388885750443, 0.5726610429354276], [-1.444663229395262, 0.6292752068048999], [-1.2135455326569522, 0.4867660615145986], [-1.431191969400899, 0.6196156221637219], [-1.3552501142535127, 0.5740429505632024], [-1.3552501142535127, 0.5740429505632024], [-1.3601323463211306, 0.5823244962847308], [-1.3490713329856707, 0.5588218361743152], [-1.3552226924624984, 0.5737318981726364], [-1.1064189503537536, 0.4055482834608345], [-1.470858535592954, 0.6624666571375569], [-1.2135455326569522, 0.4867660615145986], [-1.3601323463211306, 0.5823244962847308], [-1.444663229395262, 0.6292752068048999], [-1.1907547046557294, 0.44066678218682204], [-1.1099310500234427, 0.41591161773630914], [-1.1133893319459176, 0.42975169535593183], [-1.464249698529045, 0.6514620602758356], [-1.3478292517418853, 0.5570974090304649], [-1.3988540387614634, 0.590703947688165], [-1.3599039403650341, 0.5754868306230191], [-1.3490427815430868, 0.5587841485456753], [-1.3552501142535127, 0.5740429505632024], [-1.352388885750443, 0.5726610429354276], [-1.3552501142535127, 0.5740429505632024], [-1.221411983196501, 0.4873370092728335], [-1.3552226924624984, 0.5737318981726364], [-1.3552226924624984, 0.5737318981726364], [-1.3490427815430868, 0.5587841485456753], [-1.1063096919179696, 0.4043871721469433], [-1.1064189503537536, 0.4055482834608345], [-1.470858535592954, 0.6624666571375569], [-1.3350953575544255, 0.5398124737019164], [-1.1907547046557294, 0.44066678218682204], [-1.444663229395262, 0.6292752068048999], [-1.3988540387614634, 0.590703947688165], [-1.3601323463211306, 0.5823244962847308], [-1.1099310500234427, 0.41591161773630914], [-1.1907547046557294, 0.44066678218682204], [-1.446882386167466, 0.6352563713996238], [-1.1133893319459176, 0.42975169535593183], [-1.352388885750443, 0.5726610429354276], [-1.3599039403650341, 0.5754868306230191], [-1.1133893319459176, 0.42975169535593183], [-1.3478292517418853, 0.5570974090304649], [-1.3490427815430868, 0.5587841485456753], [-1.3552501142535127, 0.5740429505632024], [-1.3552501142535127, 0.5740429505632024], [-1.3552501142535127, 0.5740429505632024], [-1.1063096919179696, 0.4043871721469433], [-1.1064189503537536, 0.4055482834608345], [-1.470858535592954, 0.6624666571375569], [-1.444663229395262, 0.6292752068048999], [-1.3988540387614634, 0.590703947688165], [-1.1954116408332753, 0.44011627913141393], [-1.1099310500234427, 0.41591161773630914], [-1.1064190112104406, 0.405548300163537], [-1.446882386167466, 0.6352563713996238], [-1.1133893319459176, 0.42975169535593183], [-1.1133893319459176, 0.42975169535593183], [-1.3988540387614634, 0.590703947688165], [-1.1133893319459176, 0.42975169535593183], [-1.446882386167466, 0.6352563713996238], [-1.3987063659618753, 0.5365472045167299], [-1.1917636076474434, 0.4560721874048922], [-1.3350953575544255, 0.5398124737019164], [-1.1907547046557294, 0.44066678218682204], [-1.3478292517418853, 0.5570974090304649], [-1.1907547046557294, 0.44066678218682204], [-1.0895924628121454, 0.4024681285653245], [-1.1063096919179696, 0.4043871721469433], [-1.1064189503537536, 0.4055482834608345], [-1.470858535592954, 0.6624666571375569], [-1.444663229395262, 0.6292752068048999], [-1.1564179334396367, 0.43141045335971695], [-1.4370469743305576, 0.5584227385409044], [-1.2557206464808564, 0.4598501396038609], [-1.1954116408332753, 0.44011627913141393], [-1.1064190112104406, 0.405548300163537], [-1.446882386167466, 0.6352563713996238], [-1.1413747241312253, 0.4117799864250602], [-1.1912274727721957, 0.43633745888787523], [-1.446882386167466, 0.6352563713996238], [-1.468487260100611, 0.6611392696788722], [-1.448677988902125, 0.6473943359112825], [-1.398454205543546, 0.5359795995290052], [-1.3987063659618753, 0.5365472045167299], [-1.3988540387614634, 0.590703947688165], [-1.1917636076474434, 0.4560721874048922], [-1.1061964351430111, 0.40203894573261556], [-1.1063096919179696, 0.4043871721469433], [-1.1064189503537536, 0.4055482834608345], [-1.470858535592954, 0.6624666571375569], [-1.2557206464808564, 0.4598501396038609], [-1.4372778751998703, 0.5558117414721095], [-1.1564179334396367, 0.43141045335971695], [-1.4620803227993275, 0.6445887776157392], [-1.32983596604793, 0.5157200454008104], [-1.3987063659618753, 0.5365472045167299], [-1.1917803949285883, 0.43333927477238443], [-1.1954116408332753, 0.44011627913141393], [-1.1496852261784136, 0.41308332665907876], [-1.1413747241312253, 0.4117799864250602], [-1.1064190112104406, 0.405548300163537], [-1.468487260100611, 0.6611392696788722], [-1.1064189503537536, 0.4055482834608345], [-1.4686345358520008, 0.662040809521961], [-1.468487260100611, 0.6611392696788722], [-1.1413747241312253, 0.4117799864250602], [-1.1444219852556352, 0.40076104256908013], [-1.470858535592954, 0.6624666571375569], [-1.4620803227993275, 0.6445887776157392], [-1.4372778751998703, 0.5558117414721095], [-1.2557206464808564, 0.4598501396038609], [-1.2768642999545805, 0.46921859817380734], [-1.3282888433240094, 0.5064659092034356], [-1.1496852261784136, 0.41308332665907876], [-1.32983596604793, 0.5157200454008104], [-1.1917803949285883, 0.43333927477238443], [-1.2288942082591205, 0.44880348406811826], [-1.468258091227196, 0.6597265796089545], [-1.1917803949285883, 0.43333927477238443], [-1.1954116408332753, 0.44011627913141393], [-1.3987063659618753, 0.5365472045167299], [-1.4686345358520008, 0.662040809521961], [-1.224954686593971, 0.4430815128509571], [-1.4685258563467551, 0.6608690170150496], [-1.1734563664104563, 0.4214418995697482], [-1.1063096919179696, 0.4043871721469433], [-1.1444219852556352, 0.40076104256908013], [-1.470858535592954, 0.6624666571375569], [-1.4372778751998703, 0.5558117414721095], [-1.2557206464808564, 0.4598501396038609], [-1.4615023773150153, 0.6421414982667967], [-1.4620803227993275, 0.6445887776157392], [-1.2768642999545805, 0.46921859817380734], [-1.2989655011185048, 0.4862453005047429], [-1.224954686593971, 0.4430815128509571], [-1.3282888433240094, 0.5064659092034356], [-1.32983596604793, 0.5157200454008104], [-1.4641063807147274, 0.6475203291143525], [-1.1496852261784136, 0.41308332665907876], [-1.3282888433240094, 0.5064659092034356], [-1.2011392174139848, 0.4282293528308573], [-1.2768642999545805, 0.46921859817380734], [-1.468258091227196, 0.6597265796089545], [-1.2288942082591205, 0.44880348406811826], [-1.2492737938256637, 0.45061488170799663], [-1.1734563664104563, 0.4214418995697482], [-1.1444219852556352, 0.40076104256908013], [-1.470858535592954, 0.6624666571375569], [-1.2557206464808564, 0.4598501396038609], [-1.4372778751998703, 0.5558117414721095], [-1.455423243850998, 0.5857224238929708], [-1.2989655011185048, 0.4862453005047429], [-1.3282888433240094, 0.5064659092034356], [-1.4615023773150153, 0.6421414982667967], [-1.2179014840496767, 0.4410453991287213], [-1.2492737938256637, 0.45061488170799663], [-1.4620803227993275, 0.6445887776157392], [-1.2011392174139848, 0.4282293528308573], [-1.4641063807147274, 0.6475203291143525], [-1.2768642999545805, 0.46921859817380734], [-1.2846374221823886, 0.4714064704453993], [-1.3282888433240094, 0.5064659092034356], [-1.2989655011185048, 0.4862453005047429], [-1.2288942082591205, 0.44880348406811826], [-1.1633583552578293, 0.40464445504122604], [-1.1734563664104563, 0.4214418995697482], [-1.1444219852556352, 0.40076104256908013], [-1.470858535592954, 0.6624666571375569], [-1.4372778751998703, 0.5558117414721095], [-1.2557206464808564, 0.4598501396038609], [-1.3313482030186194, 0.5094411696359018], [-1.2492737938256637, 0.45061488170799663], [-1.3282888433240094, 0.5064659092034356], [-1.2846374221823886, 0.4714064704453993], [-1.2179014840496767, 0.4410453991287213], [-1.2011392174139848, 0.4282293528308573], [-1.298126144210337, 0.486027778890491], [-1.2288942082591205, 0.44880348406811826], [-1.2989655011185048, 0.4862453005047429], [-1.2990555376694974, 0.48632418109050585], [-1.2117690431262393, 0.43101900130196713], [-1.2603863525662675, 0.46286320970226447], [-1.1633583552578293, 0.40464445504122604], [-1.1734563664104563, 0.4214418995697482], [-1.2769284839630481, 0.467128498395114], [-1.2989655011185048, 0.4862453005047429], [-1.1444219852556352, 0.40076104256908013], [-1.470858535592954, 0.6624666571375569], [-1.4372778751998703, 0.5558117414721095], [-1.4685258563467551, 0.6608690170150496], [-1.2557206464808564, 0.4598501396038609], [-1.1734563664104563, 0.4214418995697482], [-1.3154250605468085, 0.4907082933317555], [-1.2492737938256637, 0.45061488170799663], [-1.2769284839630481, 0.467128498395114], [-1.3313482030186194, 0.5094411696359018], [-1.2288942082591205, 0.44880348406811826], [-1.3282888433240094, 0.5064659092034356], [-1.216315534823376, 0.43516719958413996], [-1.2603863525662675, 0.46286320970226447], [-1.2492737938256637, 0.45061488170799663], [-1.331350108877118, 0.5096410770735564], [-1.1633583552578293, 0.40464445504122604], [-1.4359119516425123, 0.5547073231732792], [-1.2011392174139848, 0.4282293528308573], [-1.2117690431262393, 0.43101900130196713], [-1.1478460168788667, 0.39928913058945265], [-1.470858535592954, 0.6624666571375569], [-1.4685258563467551, 0.6608690170150496], [-1.4372778751998703, 0.5558117414721095], [-1.4395148464956138, 0.5577030420355088], [-1.3154250605468085, 0.4907082933317555], [-1.2769284839630481, 0.467128498395114], [-1.1734563664104563, 0.4214418995697482], [-1.2492737938256637, 0.45061488170799663], [-1.3282888433240094, 0.5064659092034356], [-1.2288942082591205, 0.44880348406811826], [-1.331350108877118, 0.5096410770735564], [-1.328963591871128, 0.5080666920376471], [-1.2492737938256637, 0.45061488170799663], [-1.1633583552578293, 0.40464445504122604], [-1.4327812347455962, 0.5430395205467157], [-1.2117690431262393, 0.43101900130196713], [-1.196119191284218, 0.4266436486595889], [-1.2011392174139848, 0.4282293528308573], [-1.4685258563467551, 0.6608690170150496], [-1.1478460168788667, 0.39928913058945265], [-1.470858535592954, 0.6624666571375569], [-1.2769284839630481, 0.467128498395114], [-1.4395148464956138, 0.5577030420355088], [-1.2492818452696977, 0.4505725871171196], [-1.3282888433240094, 0.5064659092034356], [-1.4660832250697116, 0.649085579686997], [-1.4685258563467551, 0.6608690170150496], [-1.4372778751998703, 0.5558117414721095], [-1.2117690431262393, 0.43101900130196713], [-1.1633583552578293, 0.40464445504122604], [-1.3282888433240094, 0.5064659092034356], [-1.4327812347455962, 0.5430395205467157], [-1.3223862983585848, 0.485366994738514], [-1.248133507982182, 0.4421294398953527], [-1.2016960633618914, 0.4245287097971614], [-1.331350108877118, 0.5096410770735564], [-1.4685258563467551, 0.6608690170150496], [-1.4322886385750198, 0.5408922953038267], [-1.1734563664104563, 0.4214418995697482], [-1.1478460168788667, 0.39928913058945265], [-1.470858535592954, 0.6624666571375569], [-1.4395148464956138, 0.5577030420355088], [-1.3282888433240094, 0.5064659092034356], [-1.2492818452696977, 0.4505725871171196], [-1.248133507982182, 0.4421294398953527], [-1.4685258563467551, 0.6608690170150496], [-1.2573338697771774, 0.45316081567004685], [-1.4372778751998703, 0.5558117414721095], [-1.4660832250697116, 0.649085579686997], [-1.1734563664104563, 0.4214418995697482], [-1.212531305578109, 0.4367987804408404], [-1.4327812347455962, 0.5430395205467157], [-1.2030676226288275, 0.4296014906529148], [-1.331350108877118, 0.5096410770735564], [-1.3223862983585848, 0.485366994738514], [-1.4660832250697116, 0.649085579686997], [-1.1724526210024826, 0.421205643775276], [-1.2819120061711422, 0.4619907154154761], [-1.4372778751998703, 0.5558117414721095], [-1.1478460168788667, 0.39928913058945265], [-1.470858535592954, 0.6624666571375569], [-1.4448649734875747, 0.5684604198060224], [-1.3223862983585848, 0.485366994738514], [-1.4395148464956138, 0.5577030420355088], [-1.3282888433240094, 0.5064659092034356], [-1.1745966836540742, 0.42719940411652957], [-1.2492818452696977, 0.4505725871171196], [-1.248133507982182, 0.4421294398953527], [-1.2030676226288275, 0.4296014906529148], [-1.248133507982182, 0.4421294398953527], [-1.2819120061711422, 0.4619907154154761], [-1.4660832250697116, 0.649085579686997], [-1.4327812347455962, 0.5430395205467157], [-1.4372778751998703, 0.5558117414721095], [-1.4660832250697116, 0.649085579686997], [-1.3496595082066725, 0.515364220189274], [-1.4685258563467551, 0.6608690170150496], [-1.4372778751998703, 0.5558117414721095], [-1.4147415300544408, 0.53918403414199], [-1.1478460168788667, 0.39928913058945265], [-1.470858535592954, 0.6624666571375569], [-1.4448649734875747, 0.5684604198060224], [-1.2819120061711422, 0.4619907154154761], [-1.248133507982182, 0.4421294398953527], [-1.4660832250697116, 0.649085579686997], [-1.2492818452696977, 0.4505725871171196], [-1.4395148464956138, 0.5577030420355088], [-1.3223862983585848, 0.485366994738514], [-1.416331593866843, 0.5388605629756311], [-1.3223040793015646, 0.4852439613413154], [-1.248133507982182, 0.4421294398953527], [-1.4395148464956138, 0.5577030420355088], [-1.4660832250697116, 0.649085579686997], [-1.4372778751998703, 0.5558117414721095], [-1.4327812347455962, 0.5430395205467157], [-1.4660832250697116, 0.649085579686997], [-1.4685258563467551, 0.6608690170150496], [-1.4372778751998703, 0.5558117414721095], [-1.4372778751998703, 0.5558117414721095], [-1.0968899266636802, 0.3882582633002397], [-1.1478460168788667, 0.39928913058945265], [-1.470858535592954, 0.6624666571375569], [-1.4729970650117714, 0.6869767865582481], [-1.446446355240617, 0.5722788225296367], [-1.248133507982182, 0.4421294398953527], [-1.3223040793015646, 0.4852439613413154], [-1.4448649734875747, 0.5684604198060224], [-1.3223862983585848, 0.485366994738514], [-1.416331593866843, 0.5388605629756311], [-1.248133507982182, 0.4421294398953527], [-1.2819120061711422, 0.4619907154154761], [-1.2882325406653665, 0.4684874405371284], [-1.4660832250697116, 0.649085579686997], [-1.2819120061711422, 0.4619907154154761], [-1.4395148464956138, 0.5577030420355088], [-1.4395148464956138, 0.5577030420355088], [-1.4374462112180624, 0.5537083200749879], [-1.4398731297382465, 0.5587894892730803], [-1.4660832250697116, 0.649085579686997], [-1.0953934971637243, 0.3878900714876326], [-1.0968899266636802, 0.3882582633002397], [-1.1478460168788667, 0.39928913058945265], [-1.470858535592954, 0.6624666571375569], [-1.4729970650117714, 0.6869767865582481], [-1.4729970650117714, 0.6869767865582481], [-1.446446355240617, 0.5722788225296367], [-1.4660832250697116, 0.649085579686997], [-1.1850373036326014, 0.42537856179351674], [-1.3223040793015646, 0.4852439613413154], [-1.3223862983585848, 0.485366994738514], [-1.2819120061711422, 0.4619907154154761], [-1.248133507982182, 0.4421294398953527], [-1.416331593866843, 0.5388605629756311], [-1.4395148464956138, 0.5577030420355088], [-1.4398731297382465, 0.5587894892730803], [-1.4660832250697116, 0.649085579686997], [-1.2819120061711422, 0.4619907154154761], [-1.4395148464956138, 0.5577030420355088], [-1.248133507982182, 0.4421294398953527], [-1.0953934971637243, 0.3878900714876326], [-1.0968899266636802, 0.3882582633002397], [-1.1478460168788667, 0.39928913058945265], [-1.470858535592954, 0.6624666571375569], [-1.4729970650117714, 0.6869767865582481], [-1.4729970650117714, 0.6869767865582481], [-1.4739826124077053, 0.691694052387398], [-1.2819120061711422, 0.4619907154154761], [-1.4650410084032464, 0.6115229497850936], [-1.446446355240617, 0.5722788225296367], [-1.2819120061711422, 0.4619907154154761], [-1.416331593866843, 0.5388605629756311], [-1.248133507982182, 0.4421294398953527], [-1.4395148464956138, 0.5577030420355088], [-1.248133507982182, 0.4421294398953527], [-1.4398731297382465, 0.5587894892730803], [-1.4395148464956138, 0.5577030420355088], [-1.202754975761966, 0.4143305130904559], [-1.4075966588080606, 0.5211825721692505], [-1.4726463278180586, 0.6861570806441888], [-1.017440959320591, 0.37647820573852], [-1.0968899266636802, 0.3882582633002397], [-1.1478460168788667, 0.39928913058945265], [-1.470858535592954, 0.6624666571375569], [-1.4729970650117714, 0.6869767865582481], [-1.4729970650117714, 0.6869767865582481], [-1.4739826124077053, 0.691694052387398], [-1.4650410084032464, 0.6115229497850936], [-1.428802168244011, 0.5260371343384508], [-1.248133507982182, 0.4421294398953527], [-1.248133507982182, 0.4421294398953527], [-1.4655260921336413, 0.650046078751601], [-1.2819120061711422, 0.4619907154154761], [-1.446446355240617, 0.5722788225296367], [-1.202754975761966, 0.4143305130904559], [-1.4075966588080606, 0.5211825721692505], [-1.4395148464956138, 0.5577030420355088], [-1.4685258563467551, 0.6608690170150496], [-1.2819120061711422, 0.4619907154154761], [-1.4726463278180586, 0.6861570806441888], [-1.017440959320591, 0.37647820573852], [-1.0968899266636802, 0.3882582633002397], [-1.1478460168788667, 0.39928913058945265], [-1.470858535592954, 0.6624666571375569], [-1.4739826124077053, 0.691694052387398], [-1.474024279655027, 0.6986096971976762], [-1.4650410084032464, 0.6115229497850936], [-1.4395148464956138, 0.5577030420355088], [-1.4685258563467551, 0.6608690170150496], [-1.2914472882654666, 0.46315575699950706], [-1.248133507982182, 0.4421294398953527], [-1.248133507982182, 0.4421294398953527], [-1.446446355240617, 0.5722788225296367], [-1.428802168244011, 0.5260371343384508], [-1.202754975761966, 0.4143305130904559], [-1.4075966588080606, 0.5211825721692505], [-1.2846126873279082, 0.4492232760880088], [-1.2046543581858036, 0.4149772723657539], [-1.242259453737027, 0.44031820530775956], [-1.4680742763347854, 0.6442975095873502], [-0.9699400783666088, 0.3723230450941925], [-1.017440959320591, 0.37647820573852], [-1.0968899266636802, 0.3882582633002397], [-1.1478460168788667, 0.39928913058945265], [-1.470858535592954, 0.6624666571375569], [-1.4739826124077053, 0.691694052387398], [-1.474024279655027, 0.6986096971976762], [-1.4685258563467551, 0.6608690170150496], [-1.4075966588080606, 0.5211825721692505], [-1.446446355240617, 0.5722788225296367], [-1.4650410084032464, 0.6115229497850936], [-1.2046543581858036, 0.4149772723657539], [-1.4680742763347854, 0.6442975095873502], [-1.2996792029923732, 0.4863127845740594], [-1.248133507982182, 0.4421294398953527], [-1.436040228032021, 0.5532482520602972], [-1.4300324743526514, 0.5281048964815578], [-1.248133507982182, 0.4421294398953527], [-1.4453374527525398, 0.5576836013016383], [-1.0299380692617548, 0.38038104176259907], [-0.9699400783666088, 0.3723230450941925], [-0.9699400783666088, 0.3723230450941925], [-1.017440959320591, 0.37647820573852], [-1.0968899266636802, 0.3882582633002397], [-1.1478460168788667, 0.39928913058945265], [-1.470858535592954, 0.6624666571375569], [-1.4739826124077053, 0.691694052387398], [-1.474024279655027, 0.6986096971976762], [-1.446446355240617, 0.5722788225296367], [-1.4685258563467551, 0.6608690170150496], [-1.4680742763347854, 0.6442975095873502], [-1.4073140436325327, 0.5206905175083475], [-1.4300324743526514, 0.5281048964815578], [-1.2334872935018177, 0.43194371021266104], [-1.2996792029923732, 0.4863127845740594], [-1.4320434003410556, 0.5486410117291022], [-1.4453374527525398, 0.5576836013016383], [-1.4075966588080606, 0.5211825721692505], [-1.248133507982182, 0.4421294398953527], [-1.436040228032021, 0.5532482520602972], [-0.9699400783666088, 0.3723230450941925], [-0.9699400783666088, 0.3723230450941925], [-0.9699400783666088, 0.3723230450941925], [-1.017440959320591, 0.37647820573852], [-1.1478460168788667, 0.39928913058945265], [-1.470858535592954, 0.6624666571375569], [-1.4739826124077053, 0.691694052387398], [-1.4748223954234323, 0.6923539133656967], [-1.446446355240617, 0.5722788225296367], [-1.4685258563467551, 0.6608690170150496], [-1.2996792029923732, 0.4863127845740594], [-1.2334872935018177, 0.43194371021266104], [-1.4073140436325327, 0.5206905175083475], [-1.4075966588080606, 0.5211825721692505], [-1.4680742763347854, 0.6442975095873502], [-1.3331738667091697, 0.4954592816791159], [-1.4300324743526514, 0.5281048964815578], [-1.248133507982182, 0.4421294398953527], [-1.4366289679989461, 0.5455889715371522], [-1.4739826124077053, 0.691694052387398], [-0.9699400783666088, 0.3723230450941925], [-0.9699400783666088, 0.3723230450941925], [-0.9699400783666088, 0.3723230450941925], [-1.017440959320591, 0.37647820573852], [-1.470858535592954, 0.6624666571375569], [-1.4748319364142108, 0.6903549322077811], [-1.446446355240617, 0.5722788225296367], [-1.4685258563467551, 0.6608690170150496], [-1.2996792029923732, 0.4863127845740594], [-1.4680742763347854, 0.6442975095873502], [-1.2334872935018177, 0.43194371021266104], [-1.3331738667091697, 0.4954592816791159], [-1.43989556221733, 0.5453588581719417], [-1.4073140436325327, 0.5206905175083475], [-1.0300949954036303, 0.3804206582842249], [-1.4075966588080606, 0.5211825721692505], [-1.248133507982182, 0.4421294398953527], [-1.4300324743526514, 0.5281048964815578], [-1.4338292467527127, 0.5384150943467745], [-1.147896480761745, 0.39751988442948305], [-0.9699400783666088, 0.3723230450941925], [-0.9699400783666088, 0.3723230450941925], [-0.9699400783666088, 0.3723230450941925], [-0.9699400783666088, 0.3723230450941925], [-1.017440959320591, 0.37647820573852], [-1.470858535592954, 0.6624666571375569], [-1.4748319364142108, 0.6903549322077811], [-1.4748319364142108, 0.6903549322077811], [-1.4685258563467551, 0.6608690170150496], [-1.4680742763347854, 0.6442975095873502], [-1.2996792029923732, 0.4863127845740594], [-1.43989556221733, 0.5453588581719417], [-1.3062292091259249, 0.48774404334433874], [-1.248133507982182, 0.4421294398953527], [-1.2422099151858406, 0.4351255021687951], [-1.2334872935018177, 0.43194371021266104], [-1.4680032858475158, 0.6440951925605513], [-1.3331738667091697, 0.4954592816791159], [-1.4338292467527127, 0.5384150943467745], [-1.0300949954036303, 0.3804206582842249], [-0.975527550363287, 0.3685538063117853], [-1.017440959320591, 0.37647820573852], [-1.470858535592954, 0.6624666571375569], [-1.4748319364142108, 0.6903549322077811], [-1.4748319364142108, 0.6903549322077811], [-1.474877086362608, 0.7083768577821012], [-1.4685258563467551, 0.6608690170150496], [-1.457332376673139, 0.583960184311303], [-1.3331738667091697, 0.4954592816791159], [-1.4680742763347854, 0.6442975095873502], [-1.4680742763347854, 0.6442975095873502], [-1.248133507982182, 0.4421294398953527], [-1.2549718016459053, 0.44431973775009076], [-1.2422099151858406, 0.4351255021687951], [-1.4680032858475158, 0.6440951925605513], [-1.0300949954036303, 0.3804206582842249], [-1.4748319364142108, 0.6903549322077811], [-1.2334872935018177, 0.43194371021266104], [-1.3331738667091697, 0.4954592816791159], [-1.1961455920831179, 0.42627980069252713], [-0.9599017051055014, 0.36646311888364286], [-0.975527550363287, 0.3685538063117853], [-1.017440959320591, 0.37647820573852], [-1.470858535592954, 0.6624666571375569], [-1.4748319364142108, 0.6903549322077811], [-1.4748319364142108, 0.6903549322077811], [-1.474877086362608, 0.7083768577821012], [-1.3331738667091697, 0.4954592816791159], [-1.457332376673139, 0.583960184311303], [-1.2549718016459053, 0.44431973775009076], [-1.248133507982182, 0.4421294398953527], [-1.4680742763347854, 0.6442975095873502], [-1.457332376673139, 0.583960184311303], [-1.0300949954036303, 0.3804206582842249], [-1.2334872935018177, 0.43194371021266104], [-1.4680742763347854, 0.6442975095873502], [-1.4680742763347854, 0.6442975095873502], [-1.2422099151858406, 0.4351255021687951], [-1.4748319364142108, 0.6903549322077811], [-1.4748319364142108, 0.6903549322077811], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.975527550363287, 0.3685538063117853], [-1.017440959320591, 0.37647820573852], [-1.470858535592954, 0.6624666571375569], [-1.4749454840475145, 0.6886015700667377], [-1.4680742763347854, 0.6442975095873502], [-1.248133507982182, 0.4421294398953527], [-1.4573004808333454, 0.5837462488944127], [-1.2422099151858406, 0.4351255021687951], [-1.4680742763347854, 0.6442975095873502], [-1.457332376673139, 0.583960184311303], [-1.2549718016459053, 0.44431973775009076], [-1.457332376673139, 0.583960184311303], [-1.4581514842216021, 0.6142505361314871], [-0.9918999098428181, 0.3730121527487665], [-1.0370083571917328, 0.3918069793132232], [-1.4680742763347854, 0.6442975095873502], [-1.2334878936129194, 0.43193091686679674], [-1.2549718016459053, 0.44431973775009076], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.975527550363287, 0.3685538063117853], [-1.017440959320591, 0.37647820573852], [-1.470858535592954, 0.6624666571375569], [-1.4749454840475145, 0.6886015700667377], [-1.4752129760548571, 0.6944442949837949], [-1.4550172919466746, 0.5797136777781385], [-1.4685258563467551, 0.6608690170150496], [-1.4685164624460283, 0.6498869072676325], [-1.4680742763347854, 0.6442975095873502], [-1.4581514842216021, 0.6142505361314871], [-1.0370083571917328, 0.3918069793132232], [-1.4680105449884735, 0.6441156768334727], [-1.4573004808333454, 0.5837462488944127], [-0.9918999098428181, 0.3730121527487665], [-1.4680742763347854, 0.6442975095873502], [-1.457332376673139, 0.583960184311303], [-1.457332376673139, 0.583960184311303], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.975527550363287, 0.3685538063117853], [-1.017440959320591, 0.37647820573852], [-1.470858535592954, 0.6624666571375569], [-1.4749454840475145, 0.6886015700667377], [-1.4752129760548571, 0.6944442949837949], [-1.464098758006308, 0.602360356128926], [-1.0370083571917328, 0.3918069793132232], [-1.4553412006500506, 0.5741474007367356], [-1.4693868123199496, 0.6524186624188456], [-1.4680742763347854, 0.6442975095873502], [-1.4717844019117468, 0.6825780528282382], [-0.9918999098428181, 0.3730121527487665], [-1.4661218944986505, 0.6312170678044178], [-1.4680742763347854, 0.6442975095873502], [-1.4573004808333454, 0.5837462488944127], [-1.457332376673139, 0.583960184311303], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.975527550363287, 0.3685538063117853], [-1.017440959320591, 0.37647820573852], [-1.470858535592954, 0.6624666571375569], [-1.4749454840475145, 0.6886015700667377], [-1.4752129760548571, 0.6944442949837949], [-1.0370083571917328, 0.3918069793132232], [-1.0958714182149036, 0.47507668750638077], [-1.464098758006308, 0.602360356128926], [-1.4681528214902253, 0.6451147356511511], [-1.4661218944986505, 0.6312170678044178], [-1.0370083571917328, 0.3918069793132232], [-1.4576046447990876, 0.5859411773120589], [-0.9918999098428181, 0.3730121527487665], [-1.4553412006500506, 0.5741474007367356], [-1.4680742763347854, 0.6442975095873502], [-1.4693868123199496, 0.6524186624188456], [-0.9523283217427345, 0.36551881420051724], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.975527550363287, 0.3685538063117853], [-1.017440959320591, 0.37647820573852], [-1.470858535592954, 0.6624666571375569], [-1.4749454840475145, 0.6886015700667377], [-1.4752129760548571, 0.6944442949837949], [-1.047096966231823, 0.41283467521362927], [-1.0370083571917328, 0.3918069793132232], [-1.0958714182149036, 0.47507668750638077], [-1.4695744432689721, 0.6510349004955382], [-1.464098758006308, 0.602360356128926], [-1.4681528214902253, 0.6451147356511511], [-1.0370083571917328, 0.3918069793132232], [-1.4576046447990876, 0.5859411773120589], [-1.4553412006500506, 0.5741474007367356], [-1.4661218944986505, 0.6312170678044178], [-0.9523283217427345, 0.36551881420051724], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.975527550363287, 0.3685538063117853], [-1.017440959320591, 0.37647820573852], [-1.470858535592954, 0.6624666571375569], [-1.4749454840475145, 0.6886015700667377], [-1.4752129760548571, 0.6944442949837949], [-1.4752129760548571, 0.6944442949837949], [-1.047096966231823, 0.41283467521362927], [-1.047096966231823, 0.41283467521362927], [-1.4698189739882281, 0.6517216655388959], [-1.0370083571917328, 0.3918069793132232], [-1.4553412006500506, 0.5741474007367356], [-1.4576046447990876, 0.5859411773120589], [-1.459535076422573, 0.590973240781559], [-1.4681528214902253, 0.6451147356511511], [-1.464098758006308, 0.602360356128926], [-0.9523283217427345, 0.36551881420051724], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-1.017440959320591, 0.37647820573852], [-1.470858535592954, 0.6624666571375569], [-1.4749454840475145, 0.6886015700667377], [-1.4752129760548571, 0.6944442949837949], [-1.4752129760548571, 0.6944442949837949], [-1.4752129760548571, 0.6944442949837949], [-1.4698189739882281, 0.6517216655388959], [-1.047096966231823, 0.41283467521362927], [-1.047096966231823, 0.41283467521362927], [-1.0370083571917328, 0.3918069793132232], [-1.459535076422573, 0.590973240781559], [-1.079006162580247, 0.4378509168500596], [-1.446928228416654, 0.5628178378151845], [-1.4576046447990876, 0.5859411773120589], [-1.4553412006500506, 0.5741474007367356], [-0.9296861512237027, 0.3615616975815443], [-0.9523283217427345, 0.36551881420051724], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-1.017440959320591, 0.37647820573852], [-1.470858535592954, 0.6624666571375569], [-1.4749454840475145, 0.6886015700667377], [-1.4752129760548571, 0.6944442949837949], [-1.4752129760548571, 0.6944442949837949], [-1.4752129760548571, 0.6944442949837949], [-1.079006162580247, 0.4378509168500596], [-1.459535076422573, 0.590973240781559], [-1.4467442711723943, 0.5579397532568502], [-1.4698189739882281, 0.6517216655388959], [-1.4576046447990876, 0.5859411773120589], [-1.459535076422573, 0.590973240781559], [-1.047096966231823, 0.41283467521362927], [-1.0415471022755982, 0.3859599477309344], [-0.9296861512237027, 0.3615616975815443], [-0.9523283217427345, 0.36551881420051724], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-1.017440959320591, 0.37647820573852], [-1.470858535592954, 0.6624666571375569], [-1.4749454840475145, 0.6886015700667377], [-1.4752129760548571, 0.6944442949837949], [-1.4752129760548571, 0.6944442949837949], [-1.4752129760548571, 0.6944442949837949], [-1.4761413566784876, 0.7083632215716181], [-1.4470392620521157, 0.5576189146666708], [-1.2861386706330036, 0.5517125626354495], [-1.4576046447990876, 0.5859411773120589], [-1.0792886499126242, 0.4379287055224205], [-1.4698189739882281, 0.6517216655388959], [-1.079006162580247, 0.4378509168500596], [-1.0415471022755982, 0.3859599477309344], [-0.9296861512237027, 0.3615616975815443], [-0.9523283217427345, 0.36551881420051724], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-0.9599017051055014, 0.36646311888364286], [-1.017440959320591, 0.37647820573852], [-1.470858535592954, 0.6624666571375569], [-1.4749454840475145, 0.6886015700667377], [-1.4752129760548571, 0.6944442949837949], [-1.4752129760548571, 0.6944442949837949], [-1.4752129760548571, 0.6944442949837949], [-1.4761413566784876, 0.7083632215716181], [-1.4761429940235233, 0.7083755935506475], [-1.4698189739882281, 0.6517216655388959], [-1.0792886499126242, 0.4379287055224205], [-1.2862788191838823, 0.5517638735420873], [-1.4551386337626617, 0.5851576573788706], [-1.079006162580247, 0.4378509168500596], [-1.4470392620521157, 0.5576189146666708], [-0.918122680374084, 0.35219310382681873], [-0.9296861512237027, 0.3615616975815443], [-1.017440959320591, 0.37647820573852], [-1.470858535592954, 0.6624666571375569], [-1.4749454840475145, 0.6886015700667377], [-1.4752129760548571, 0.6944442949837949], [-1.4752129760548571, 0.6944442949837949], [-1.4752129760548571, 0.6944442949837949], [-1.4761413566784876, 0.7083632215716181], [-1.4761429940235233, 0.7083755935506475], [-1.4698189739882281, 0.6517216655388959], [-1.2862788191838823, 0.5517638735420873], [-1.0881340491311382, 0.4404026091397475], [-1.4563910074597366, 0.6062471218782451], [-1.287489911982393, 0.5529599223592919], [-1.4505678951703964, 0.5720599420477614], [-1.0792886499126242, 0.4379287055224205], [-1.4551386337626617, 0.5851576573788706], [-1.079006162580247, 0.4378509168500596], [-1.04719269224377, 0.42926889259740014], [-0.918122680374084, 0.35219310382681873], [-0.9296861512237027, 0.3615616975815443], [-1.017440959320591, 0.37647820573852], [-1.4749454840475145, 0.6886015700667377], [-1.4752129760548571, 0.6944442949837949], [-1.4752129760548571, 0.6944442949837949], [-1.4752129760548571, 0.6944442949837949], [-1.4763364277405007, 0.7046437479373747], [-1.2862788191838823, 0.5517638735420873], [-1.4653321474225491, 0.567612612644729], [-1.195090928652362, 0.46750347383858964], [-1.04719269224377, 0.42926889259740014], [-1.0881340491311382, 0.4404026091397475], [-1.079006162580247, 0.4378509168500596], [-1.287489911982393, 0.5529599223592919], [-1.0792886499126242, 0.4379287055224205], [-1.0271818715241672, 0.42538465699003497], [-1.4728710156687674, 0.5805884864345837], [-1.4752129760548571, 0.6944442949837949], [-1.0792886499126242, 0.4379287055224205], [-0.918122680374084, 0.35219310382681873], [-0.9296861512237027, 0.3615616975815443], [-1.017440959320591, 0.37647820573852], [-1.4749454840475145, 0.6886015700667377], [-1.4752129760548571, 0.6944442949837949], [-1.4752129760548571, 0.6944442949837949], [-1.4752129760548571, 0.6944442949837949], [-1.4763364277405007, 0.7046437479373747], [-1.4653321474225491, 0.567612612644729], [-1.47306612057018, 0.5793964694519286], [-1.0271818715241672, 0.42538465699003497], [-1.04719269224377, 0.42926889259740014], [-1.4752129760548571, 0.6944442949837949], [-0.9941508341546905, 0.3727689902153579], [-1.099269819238906, 0.43705476284448336], [-1.195090928652362, 0.46750347383858964], [-1.3147335119015162, 0.5092079332075364], [-1.2862788191838823, 0.5517638735420873], [-1.4731481024741917, 0.695524271310252], [-1.2855323100387512, 0.5513051965870426], [-0.9226282180264239, 0.35039520638399024], [-0.9296861512237027, 0.3615616975815443], [-1.017440959320591, 0.37647820573852], [-1.4788338131233931, 0.65524461715113], [-1.4653321474225491, 0.567612612644729], [-1.4741586600589598, 0.5842082707095833], [-1.1951396951352877, 0.467395771718967], [-1.099269819238906, 0.43705476284448336], [-1.0271818715241672, 0.42538465699003497], [-1.47306612057018, 0.5793964694519286], [-0.9941508341546905, 0.3727689902153579], [-1.3147335119015162, 0.5092079332075364], [-1.04719269224377, 0.42926889259740014], [-1.4717246250176026, 0.574871900765837], [-0.9941508341546905, 0.3727689902153579], [-1.0972106909870816, 0.435907328926597], [-1.099269819238906, 0.43705476284448336], [-1.473971996289705, 0.5831534797167871], [-1.3147335119015162, 0.5092079332075364], [-0.918122680374084, 0.35219310382681873], [-0.9226282180264239, 0.35039520638399024], [-0.9296861512237027, 0.3615616975815443], [-1.017440959320591, 0.37647820573852], [-1.4788338131233931, 0.65524461715113], [-1.4653321474225491, 0.567612612644729], [-1.4741586600589598, 0.5842082707095833], [-1.4444583495773795, 0.5448799159623777], [-1.322825575017221, 0.511040457317587], [-1.1951396951352877, 0.467395771718967], [-1.099269819238906, 0.43705476284448336], [-0.9941508341546905, 0.3727689902153579], [-1.0271818715241672, 0.42538465699003497], [-1.04719269224377, 0.42926889259740014], [-1.47306612057018, 0.5793964694519286], [-1.3181816416545191, 0.48708089391517984], [-1.4727785294665547, 0.5783402590511558], [-0.9979431963902052, 0.37368173912988434], [-1.1951396951352877, 0.467395771718967], [-0.9941508341546905, 0.3727689902153579], [-1.0972106909870816, 0.435907328926597], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-0.9296861512237027, 0.3615616975815443], [-1.017440959320591, 0.37647820573852], [-1.4788828516311094, 0.6544289024078553], [-1.322825575017221, 0.511040457317587], [-1.4741586600589598, 0.5842082707095833], [-1.4653321474225491, 0.567612612644729], [-1.4741586600589598, 0.5842082707095833], [-1.4444583495773795, 0.5448799159623777], [-1.4444583495773795, 0.5448799159623777], [-0.9941508341546905, 0.3727689902153579], [-1.268458112875296, 0.4796139652292784], [-1.47306612057018, 0.5793964694519286], [-1.0271818715241672, 0.42538465699003497], [-1.3172103060319476, 0.4858341974421133], [-1.322825575017221, 0.511040457317587], [-1.029509990324018, 0.42555977776723314], [-1.04719269224377, 0.42926889259740014], [-1.195863881355462, 0.466378181020934], [-0.8605232428658077, 0.339034863544104], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-0.9296861512237027, 0.3615616975815443], [-1.017440959320591, 0.37647820573852], [-1.4788828516311094, 0.6544289024078553], [-1.4444583495773795, 0.5448799159623777], [-1.4749344672753413, 0.5872177842850379], [-1.4788050732707234, 0.6518150878488745], [-1.322825575017221, 0.511040457317587], [-1.322825575017221, 0.511040457317587], [-1.4741586600589598, 0.5842082707095833], [-1.4444583495773795, 0.5448799159623777], [-0.9941508341546905, 0.3727689902153579], [-1.268458112875296, 0.4796139652292784], [-1.4741586600589598, 0.5842082707095833], [-1.3226577646104674, 0.5108441303117202], [-1.029509990324018, 0.42555977776723314], [-0.8872603209444231, 0.3437143991594525], [-1.3160699322400449, 0.48454391749196435], [-0.8605232428658077, 0.339034863544104], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-0.9296861512237027, 0.3615616975815443], [-1.017440959320591, 0.37647820573852], [-1.4790173211928277, 0.6482633116155742], [-1.4749344672753413, 0.5872177842850379], [-1.437174832843053, 0.5289444504475385], [-1.4444583495773795, 0.5448799159623777], [-1.4444583495773795, 0.5448799159623777], [-1.322825575017221, 0.511040457317587], [-1.322825575017221, 0.511040457317587], [-1.029509990324018, 0.42555977776723314], [-1.4741586600589598, 0.5842082707095833], [-0.9935904511872586, 0.3694822967338672], [-1.3226577646104674, 0.5108441303117202], [-0.9941508341546905, 0.3727689902153579], [-1.4758417689934142, 0.593083989594907], [-1.4741586600589598, 0.5842082707095833], [-1.4383880813178391, 0.5371117351214929], [-0.8593433500605021, 0.33881985076870114], [-0.8605232428658077, 0.339034863544104], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-0.9296861512237027, 0.3615616975815443], [-1.017440959320591, 0.37647820573852], [-1.4790173211928277, 0.6482633116155742], [-1.437174832843053, 0.5289444504475385], [-1.4787160675024225, 0.6449282404477875], [-1.4758417689934142, 0.593083989594907], [-1.4749344672753413, 0.5872177842850379], [-1.3251219799529343, 0.5154449652647707], [-1.4383880813178391, 0.5371117351214929], [-0.9941508341546905, 0.3727689902153579], [-0.9935904511872586, 0.3694822967338672], [-1.4444583495773795, 0.5448799159623777], [-1.4444583495773795, 0.5448799159623777], [-1.029509990324018, 0.42555977776723314], [-1.4741586600589598, 0.5842082707095833], [-1.322825575017221, 0.511040457317587], [-0.8593433500605021, 0.33881985076870114], [-0.8605232428658077, 0.339034863544104], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-0.9296861512237027, 0.3615616975815443], [-1.017440959320591, 0.37647820573852], [-1.4790173211928277, 0.6482633116155742], [-1.4790662865620599, 0.6518008057415078], [-1.029509990324018, 0.42555977776723314], [-1.4787160675024225, 0.6449282404477875], [-1.4780369394647868, 0.6111901650119206], [-1.3251219799529343, 0.5154449652647707], [-1.437174832843053, 0.5289444504475385], [-1.4749344672753413, 0.5872177842850379], [-1.437174832843053, 0.5289444504475385], [-1.4758417689934142, 0.593083989594907], [-1.4383880813178391, 0.5371117351214929], [-1.4758417689934142, 0.593083989594907], [-1.322825575017221, 0.511040457317587], [-0.9941508341546905, 0.3727689902153579], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-1.017440959320591, 0.37647820573852], [-1.4790173211928277, 0.6482633116155742], [-1.4790662865620599, 0.6518008057415078], [-1.4790662865620599, 0.6518008057415078], [-1.4780369394647868, 0.6111901650119206], [-1.4787160675024225, 0.6449282404477875], [-1.3251219799529343, 0.5154449652647707], [-1.029509990324018, 0.42555977776723314], [-1.437174832843053, 0.5289444504475385], [-0.9941508341546905, 0.3727689902153579], [-1.4749344672753413, 0.5872177842850379], [-1.475860642204904, 0.5944898070834096], [-1.437174832843053, 0.5289444504475385], [-1.437174832843053, 0.5289444504475385], [-1.3231027177072652, 0.5074784615621813], [-1.4758417689934142, 0.593083989594907], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-1.017440959320591, 0.37647820573852], [-1.4790173211928277, 0.6482633116155742], [-1.4791164783381938, 0.6509775391923088], [-1.3251219799529343, 0.5154449652647707], [-1.4780369394647868, 0.6111901650119206], [-1.0999556832670272, 0.4422907492084256], [-1.475860642204904, 0.5944898070834096], [-1.029509990324018, 0.42555977776723314], [-1.437174832843053, 0.5289444504475385], [-1.4787160675024225, 0.6449282404477875], [-1.478149848743075, 0.6431294498441062], [-1.4780369394647868, 0.6111901650119206], [-1.4741381415634995, 0.5871813031396865], [-0.9941508341546905, 0.3727689902153579], [-1.437174832843053, 0.5289444504475385], [-0.8593348672085714, 0.3388098449499615], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-1.017440959320591, 0.37647820573852], [-1.4796281696228717, 0.6147667156001552], [-1.437174832843053, 0.5289444504475385], [-0.9465345017396742, 0.3545033548624443], [-1.3251219799529343, 0.5154449652647707], [-1.475860642204904, 0.5944898070834096], [-1.029509990324018, 0.42555977776723314], [-0.9941508341546905, 0.3727689902153579], [-1.437174832843053, 0.5289444504475385], [-1.0201166798881958, 0.3802377847521691], [-0.8705609772890899, 0.34474325082157153], [-1.4789400703249156, 0.595898095128758], [-0.8593433500605021, 0.33881985076870114], [-0.9234596430054969, 0.35221491818630923], [-0.8593348672085714, 0.3388098449499615], [-0.8593348672085714, 0.3388098449499615], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-1.017440959320591, 0.37647820573852], [-1.4796281696228717, 0.6147667156001552], [-1.4373827585476588, 0.5264988807429278], [-1.029509990324018, 0.42555977776723314], [-1.4789400703249156, 0.595898095128758], [-0.9465345017396742, 0.3545033548624443], [-1.471406263488269, 0.5890906959148356], [-1.475860642204904, 0.5944898070834096], [-0.957852861960349, 0.3587089904844208], [-0.9941508341546905, 0.3727689902153579], [-0.9234596430054969, 0.35221491818630923], [-1.0201166798881958, 0.3802377847521691], [-0.9941508341546905, 0.3727689902153579], [-0.8593348672085714, 0.3388098449499615], [-0.8593348672085714, 0.3388098449499615], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-1.017440959320591, 0.37647820573852], [-1.4796281696228717, 0.6147667156001552], [-1.480253590223008, 0.6259708145274879], [-1.4789400703249156, 0.595898095128758], [-1.426618341850304, 0.5153726037292348], [-1.4373827585476588, 0.5264988807429278], [-1.471406263488269, 0.5890906959148356], [-1.0201166798881958, 0.3802377847521691], [-0.9941508341546905, 0.3727689902153579], [-0.9953399703366801, 0.3735553766025713], [-0.9624050328275019, 0.3626534920253443], [-0.9465345017396742, 0.3545033548624443], [-1.4676067526090375, 0.5791869470593471], [-0.8593348672085714, 0.3388098449499615], [-0.8593348672085714, 0.3388098449499615], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-1.017440959320591, 0.37647820573852], [-1.4796281696228717, 0.6147667156001552], [-1.480253590223008, 0.6259708145274879], [-1.0366140047906827, 0.38158589901095336], [-1.4676067526090375, 0.5791869470593471], [-1.124290515401547, 0.4528382195069943], [-1.4792182421357594, 0.5945158956350644], [-0.9624050328275019, 0.3626534920253443], [-1.4373827585476588, 0.5264988807429278], [-0.9941508341546905, 0.3727689902153579], [-1.426618341850304, 0.5153726037292348], [-1.0201166798881958, 0.3802377847521691], [-0.9953399703366801, 0.3735553766025713], [-0.8593348672085714, 0.3388098449499615], [-0.8593348672085714, 0.3388098449499615], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-1.017440959320591, 0.37647820573852], [-1.4796281696228717, 0.6147667156001552], [-1.4803923610911969, 0.6206834017763602], [-1.4792182421357594, 0.5945158956350644], [-0.9624050328275019, 0.3626534920253443], [-1.4676067526090375, 0.5791869470593471], [-1.124290515401547, 0.4528382195069943], [-1.4676067526090375, 0.5791869470593471], [-0.9941508341546905, 0.3727689902153579], [-1.0366140047906827, 0.38158589901095336], [-1.4373827585476588, 0.5264988807429278], [-0.8878937328313363, 0.3475512252966182], [-1.124290515401547, 0.4528382195069943], [-0.85648968142638, 0.3383914427736366], [-0.8593348672085714, 0.3388098449499615], [-0.8593348672085714, 0.3388098449499615], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-1.017440959320591, 0.37647820573852], [-1.4796281696228717, 0.6147667156001552], [-1.4803923610911969, 0.6206834017763602], [-1.4809026242656433, 0.6214671636047462], [-1.4792182421357594, 0.5945158956350644], [-1.0366140047906827, 0.38158589901095336], [-1.4676067526090375, 0.5791869470593471], [-1.1046092207189375, 0.45048300990106205], [-1.4676067526090375, 0.5791869470593471], [-0.9245063537523626, 0.35417885748808303], [-1.4676067526090375, 0.5791869470593471], [-1.124290515401547, 0.4528382195069943], [-0.85648968142638, 0.3383914427736366], [-0.8593348672085714, 0.3388098449499615], [-0.8593348672085714, 0.3388098449499615], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-1.017440959320591, 0.37647820573852], [-1.4796281696228717, 0.6147667156001552], [-1.4809026242656433, 0.6214671636047462], [-1.4792182421357594, 0.5945158956350644], [-1.473759130296467, 0.5803327148759568], [-0.9260876792163628, 0.35107759655115794], [-1.1046092207189375, 0.45048300990106205], [-1.0390070107530298, 0.38518588400524023], [-1.0366140047906827, 0.38158589901095336], [-1.4676067526090375, 0.5791869470593471], [-1.4676067526090375, 0.5791869470593471], [-1.4676067526090375, 0.5791869470593471], [-0.8559914929251313, 0.33831779428539516], [-0.85648968142638, 0.3383914427736366], [-0.8593348672085714, 0.3388098449499615], [-0.8593348672085714, 0.3388098449499615], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-1.017440959320591, 0.37647820573852], [-1.4809026242656433, 0.6214671636047462], [-1.4792182421357594, 0.5945158956350644], [-1.4676067526090375, 0.5791869470593471], [-1.04125297073231, 0.39360805942207855], [-1.473759130296467, 0.5803327148759568], [-0.9260876792163628, 0.35107759655115794], [-0.9401113151804054, 0.370397419617154], [-1.0674054942813522, 0.4469901274619481], [-1.4676067526090375, 0.5791869470593471], [-1.0390070107530298, 0.38518588400524023], [-0.8559914929251313, 0.33831779428539516], [-0.85648968142638, 0.3383914427736366], [-0.8593348672085714, 0.3388098449499615], [-0.8593348672085714, 0.3388098449499615], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-1.017440959320591, 0.37647820573852], [-1.4811437033772257, 0.6203276840982447], [-1.4676067526090375, 0.5791869470593471], [-1.0674054942813522, 0.4469901274619481], [-1.4807132696990026, 0.619221539696411], [-1.4792188465145297, 0.5945129401307001], [-1.4676067526090375, 0.5791869470593471], [-0.9401113151804054, 0.370397419617154], [-0.9260876792163628, 0.35107759655115794], [-1.4787589772330154, 0.5919035335459871], [-1.0396811128724939, 0.383992084401698], [-0.8445068895235903, 0.33660694261165935], [-0.85648968142638, 0.3383914427736366], [-0.8593348672085714, 0.3388098449499615], [-0.8593348672085714, 0.3388098449499615], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-1.017440959320591, 0.37647820573852], [-1.4811437033772257, 0.6203276840982447], [-1.4807132696990026, 0.619221539696411], [-1.11660036333891, 0.4459111158014958], [-1.4676067526090375, 0.5791869470593471], [-1.4797710978892067, 0.5999016268157662], [-1.0396811128724939, 0.383992084401698], [-0.9401113151804054, 0.370397419617154], [-1.4676067526090375, 0.5791869470593471], [-1.4792188465145297, 0.5945129401307001], [-1.4787589772330154, 0.5919035335459871], [-0.8445068895235903, 0.33660694261165935], [-0.8593348672085714, 0.3388098449499615], [-0.8593348672085714, 0.3388098449499615], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-1.017440959320591, 0.37647820573852], [-1.4811437033772257, 0.6203276840982447], [-1.4797710978892067, 0.5999016268157662], [-1.0396811128724939, 0.383992084401698], [-1.11660036333891, 0.4459111158014958], [-1.086080942994675, 0.42141611099082016], [-1.4792188465145297, 0.5945129401307001], [-1.4676067526090375, 0.5791869470593471], [-1.4787589772330154, 0.5919035335459871], [-0.9486218815201601, 0.3721808701906105], [-1.0165653485679746, 0.3763349408056224], [-0.9401113151804054, 0.370397419617154], [-0.8445068895235903, 0.33660694261165935], [-0.8593348672085714, 0.3388098449499615], [-0.8593348672085714, 0.3388098449499615], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-1.017440959320591, 0.37647820573852], [-1.4811437033772257, 0.6203276840982447], [-1.086080942994675, 0.42141611099082016], [-1.11660036333891, 0.4459111158014958], [-1.4806396706985618, 0.5949734113355541], [-1.4676067526090375, 0.5791869470593471], [-1.0165653485679746, 0.3763349408056224], [-1.4787589772330154, 0.5919035335459871], [-1.0396811128724939, 0.383992084401698], [-0.9812589153568818, 0.36162273104149467], [-0.8612574071914547, 0.34399725066836706], [-1.4792835145241265, 0.5941978675805771], [-0.8435555440142639, 0.3359601566417938], [-0.8445068895235903, 0.33660694261165935], [-0.8593348672085714, 0.3388098449499615], [-0.8593348672085714, 0.3388098449499615], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-1.017440959320591, 0.37647820573852], [-1.4811437033772257, 0.6203276840982447], [-1.4811437033772257, 0.6203276840982447], [-1.3142461627896151, 0.5121012510143177], [-1.086080942994675, 0.42141611099082016], [-1.4806402587737875, 0.5953020796610382], [-1.4638811287018831, 0.5658114881166852], [-1.4676067526090375, 0.5791869470593471], [-1.1177967226033105, 0.4455325041984124], [-0.9812589153568818, 0.36162273104149467], [-1.4787589772330154, 0.5919035335459871], [-0.8435555440142639, 0.3359601566417938], [-0.8445068895235903, 0.33660694261165935], [-0.8593348672085714, 0.3388098449499615], [-0.8593348672085714, 0.3388098449499615], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-0.9226282180264239, 0.35039520638399024], [-1.017440959320591, 0.37647820573852], [-1.4811437033772257, 0.6203276840982447], [-1.4811437033772257, 0.6203276840982447], [-1.4811437033772257, 0.6203276840982447], [-1.4806402587737875, 0.5953020796610382], [-0.8596176391523707, 0.3398001679954529], [-1.4676067526090375, 0.5791869470593471], [-1.3142461627896151, 0.5121012510143177], [-1.4787589772330154, 0.5919035335459871], [-1.4623677387019343, 0.564365571615718], [-1.178131535306054, 0.4880251358879405], [-0.8435555440142639, 0.3359601566417938], [-0.8445068895235903, 0.33660694261165935], [-0.8593348672085714, 0.3388098449499615], [-0.8593348672085714, 0.3388098449499615], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-1.017440959320591, 0.37647820573852], [-1.4811437033772257, 0.6203276840982447], [-1.4811437033772257, 0.6203276840982447], [-1.4811437033772257, 0.6203276840982447], [-1.481169922875355, 0.6207594359063903], [-1.4787589772330154, 0.5919035335459871], [-1.4806402587737875, 0.5953020796610382], [-1.4676067526090375, 0.5791869470593471], [-1.447817769456558, 0.5547055360644169], [-0.8596176391523707, 0.3398001679954529], [-1.4623677387019343, 0.564365571615718], [-1.4648721624584038, 0.5763054229534186], [-0.8416006284530924, 0.33396295753415606], [-0.8435555440142639, 0.3359601566417938], [-0.8445068895235903, 0.33660694261165935], [-0.8593348672085714, 0.3388098449499615], [-0.8593348672085714, 0.3388098449499615], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-1.017440959320591, 0.37647820573852], [-1.4814300060201409, 0.6192908594273387], [-1.4806402587737875, 0.5953020796610382], [-1.447817769456558, 0.5547055360644169], [-1.4806402587737875, 0.5953020796610382], [-1.4477155150546652, 0.5510756839598018], [-0.8596176391523707, 0.3398001679954529], [-1.4648721624584038, 0.5763054229534186], [-1.4623677387019343, 0.564365571615718], [-0.9510660735383429, 0.35342015123965287], [-1.478765205429451, 0.5857111392852995], [-0.8416006284530924, 0.33396295753415606], [-0.8435555440142639, 0.3359601566417938], [-0.8445068895235903, 0.33660694261165935], [-0.8593348672085714, 0.3388098449499615], [-0.8593348672085714, 0.3388098449499615], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-1.017440959320591, 0.37647820573852], [-1.4814300060201409, 0.6192908594273387], [-1.4814300060201409, 0.6192908594273387], [-1.4806402587737875, 0.5953020796610382], [-1.4477155150546652, 0.5510756839598018], [-1.4648721624584038, 0.5763054229534186], [-0.9551429240634213, 0.3634218240632035], [-0.9510660735383429, 0.35342015123965287], [-0.8600775487608056, 0.34118454162870904], [-1.1659629655510617, 0.4996020325071529], [-1.4623677387019343, 0.564365571615718], [-0.8416006284530924, 0.33396295753415606], [-0.8593348672085714, 0.3388098449499615], [-0.8593348672085714, 0.3388098449499615], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-1.017440959320591, 0.37647820573852], [-1.4814300060201409, 0.6192908594273387], [-1.4814300060201409, 0.6192908594273387], [-1.4806402587737875, 0.5953020796610382], [-1.4648721624584038, 0.5763054229534186], [-1.0622650410785022, 0.428829307619495], [-0.9551429240634213, 0.3634218240632035], [-0.9510660735383429, 0.35342015123965287], [-1.4477155150546652, 0.5510756839598018], [-1.4625471775106746, 0.5696149703618619], [-1.4623677387019343, 0.564365571615718], [-0.8916595875824798, 0.34504588315113105], [-1.4699294279739288, 0.5824828626944349], [-0.8416006284530924, 0.33396295753415606], [-0.8593348672085714, 0.3388098449499615], [-0.8593348672085714, 0.3388098449499615], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.8593433500605021, 0.33881985076870114], [-0.9225719737015836, 0.35028861795758925], [-1.017440959320591, 0.37647820573852], [-1.4814300060201409, 0.6192908594273387], [-1.4814300060201409, 0.6192908594273387], [-1.0622650410785022, 0.428829307619495], [-1.4806402587737875, 0.5953020796610382], [-1.4648721624584038, 0.5763054229534186], [-1.4808403375210155, 0.5970518677494048], [-0.9551429240634213, 0.3634218240632035], [-0.9510660735383429, 0.35342015123965287], [-1.4640316763053292, 0.5653830512903274], [-1.4477155150546652, 0.5510756839598018], [-1.4623677387019343, 0.564365571615718], [-1.0007910163118239, 0.371284847026502]]]

    import pylab as pl
    from matplotlib import rc
    rc('text', usetex=True)
    rc('font', size=16)

    pl.figure(figsize=[10, 6])
    nsga_x = []
    nsga_y = []
    for i in range(len(multi_nsga_x)):
        nsga_x.append(multi_nsga_y[i][0])
        nsga_y.append(multi_nsga_y[i][1])
    pl.plot(nsga_y, nsga_x, color='blue', marker='o', markersize=5, linestyle='none', label="direct evaluation")
    nsga_kpls_x = []
    nsga_kpls_y = []
    for i in range(len(multi_nsga_kpls_x)):
        nsga_kpls_x.append(multi_nsga_kpls_y[i][0])
        nsga_kpls_y.append(multi_nsga_kpls_y[i][1])
    pl.plot(nsga_kpls_y, nsga_kpls_x, color='red', marker='o', markersize=5, linestyle='none', label="with surrogate model")
    pl.grid()
    pl.xlabel("$V~(\mathrm{m^3})$")
    pl.ylabel("$BL~(\mathrm{T \cdot m})$")
    pl.xlim(0.3, 0.8)
    pl.ylim(0.13, 0.25)
    pl.legend()
    pl.tight_layout()
    pl.savefig("/tmp/multi_nsgaii.pdf")

# sweep()
# single()
# nsga2()

# check_regressor()
results_single()
# results_multi()